///////////////////////////////////////////////////////////////////////////////
// Copyright 2004,2006,2007,2009 Sony Corporation
///////////////////////////////////////////////////////////////////////////////

#ifndef __CEDXMathTypes_h__
#define __CEDXMathTypes_h__

#include "CESysDefs.h"
#include "CEUITypes.h"

//! Matrix (4x4)
//! we use Direct3D-style matrix. 
//! - the elements are stored in memoy in the row-major order.
//! - when transforming a vector, a row vector and a matrix are
//!   multiplied. that is, vec * mat, not mat * vec.
//! - the fourth row (_m41, _m42, _m43) holds translation factors.
typedef struct _CEDXMatrix44 
{
    float _m11; float _m12; float _m13; float _m14;
    float _m21; float _m22; float _m23; float _m24;
    float _m31; float _m32; float _m33; float _m34;
    float _m41; float _m42; float _m43; float _m44;
} CEDXMatrix44;

//! Matrix (3x3)
//! we use Direct3D-style matrix. 
//! - the elements are stored in memory in the row-major order.
//! - when transforming a vector, a row vector and a matrix are
//!   multiplied. that is, vec * mat, not mat * vec.
//! - the third row (_m31, _m32) holds translation factors.
typedef struct _CEDXMatrix33 
{
    float _m11; float _m12; float _m13;
    float _m21; float _m22; float _m23;
    float _m31; float _m32; float _m33;
} CEDXMatrix33;

//! Matrix (2D affine)
//! a point (x, y) is transformed as follows:
//!*     new x = _xx * x + _xy * y + _x0;
//!*     new y = _yx * x + _yy * y + _y0;
typedef struct _CEDXMatrix2DAffineD
{
	double _xx; double _yx;
	double _xy; double _yy;
	double _x0; double _y0;
} CEDXMatrix2DAffineD;

// TODO: these inline functions should be moved to appropriate place

inline bool CEIsRectEmpty(const CERect* pRect)
{
	return (!pRect->_width || !pRect->_height);
}

inline bool CEIsPointInRect(const CEPointBase* pPoint, const CERect* pRect)
{
	return((pRect->_x <= pPoint->_x) && (pPoint->_x < (pRect->_x + pRect->_width)) &&
			(pRect->_y <= pPoint->_y) && (pPoint->_y < (pRect->_y + pRect->_height)));
}

inline bool CEIsPointInRectF(const CEPointF* pPoint, const CERectF* pRect)
{
	return((pRect->_x <= pPoint->_x) && (pPoint->_x < (pRect->_x + pRect->_width)) &&
			(pRect->_y <= pPoint->_y) && (pPoint->_y < (pRect->_y + pRect->_height)));
}

// Note: since global functions must be used from C (not C++), you cannot use templates.
#define DECL_INTERSECT_RECT(func, TRect, TScalar)										\
inline bool func(const TRect* pRect1, const TRect* pRect2, TRect* pRectOut)				\
{																						\
	CEASSERT(pRect1 && pRect2 && pRectOut);											\
																						\
	TScalar x1 = CE_MAX(pRect1->_x, pRect2->_x);										\
	TScalar x2 = CE_MIN(pRect1->_x + pRect1->_width, pRect2->_x + pRect2->_width);	\
	TScalar y1 = CE_MAX(pRect1->_y, pRect2->_y);										\
	TScalar y2 = CE_MIN(pRect1->_y + pRect1->_height, pRect2->_y + pRect2->_height);	\
																						\
	pRectOut->_x = x1;																	\
	pRectOut->_y = y1;																	\
	pRectOut->_width  = (x1 < x2) ? (x2 - x1) : 0;										\
	pRectOut->_height = (y1 < y2) ? (y2 - y1) : 0;										\
																						\
	return (pRectOut->_width && pRectOut->_height);										\
}

DECL_INTERSECT_RECT(CEIntersectRect, CERect, INT32)
DECL_INTERSECT_RECT(CEIntersectRectF, CERectF, float)

#undef DECL_INTERSECT_RECT

// Note: since global functions must be used from C (not C++), you cannot use templates.
#define DECL_UNION_RECT(func, TRect, TScalar)												\
inline bool func(const TRect* pRect1, const TRect* pRect2, TRect* pRectOut)			\
{																							\
	CEASSERT(pRect1 && pRect2 && pRectOut);												\
																							\
	if (!pRect1->_width || !pRect1->_height)												\
	{																						\
		*pRectOut = *pRect2;																\
	}																						\
	else if (!pRect2->_width || !pRect2->_height)											\
	{																						\
		*pRectOut = *pRect1;																\
	}																						\
	else																					\
	{																						\
		TScalar x1 = CE_MIN(pRect1->_x, pRect2->_x);										\
		TScalar x2 = CE_MAX(pRect1->_x + pRect1->_width, pRect2->_x + pRect2->_width);	\
		TScalar y1 = CE_MIN(pRect1->_y, pRect2->_y);										\
		TScalar y2 = CE_MAX(pRect1->_y + pRect1->_height, pRect2->_y + pRect2->_height);	\
																							\
		pRectOut->_x = x1;																	\
		pRectOut->_y = y1;																	\
		pRectOut->_width  = x2 - x1;														\
		pRectOut->_height = y2 - y1;														\
	}																						\
																							\
	return (pRectOut->_width && pRectOut->_height);											\
}

DECL_UNION_RECT(CEUnionRect, CERect, INT32)
DECL_UNION_RECT(CEUnionRectF, CERectF, float)

#undef DECL_UNION_RECT


inline CEHResult CEDXMatrixIdentity33(CEDXMatrix33* pOut)
{
	if(!pOut)
	{
		return(CE_SILK_ERR_BADARGS);
	}

	pOut->_m11 = 1.0f;	pOut->_m12 = 0.0f;	pOut->_m13 = 0.0f;
	pOut->_m21 = 0.0f;	pOut->_m22 = 1.0f;	pOut->_m23 = 0.0f;
	pOut->_m31 = 0.0f;	pOut->_m32 = 0.0f;	pOut->_m33 = 1.0f;

	return(CE_S_OK);
}

inline CEDXMatrix33* CEDXMatrix33Multiply(CEDXMatrix33* pOut, const CEDXMatrix33* pM1, const CEDXMatrix33* pM2)
{
	CEDXMatrix33 tmp1, tmp2;

	if(pOut && pM1 && pM2)
	{
		if(pOut == pM1)
		{
			tmp1 = *pM1;
			pM1 = &tmp1;
		}
		if(pOut == pM2)
		{
			tmp2 = *pM2;
			pM2 = &tmp2;
		}

		// now we are sure that pOut, pM1 and pM2 point to different memory blocks.
		pOut->_m11 = pM1->_m11 * pM2->_m11 + pM1->_m12 * pM2->_m21 + pM1->_m13 * pM2->_m31;
		pOut->_m12 = pM1->_m11 * pM2->_m12 + pM1->_m12 * pM2->_m22 + pM1->_m13 * pM2->_m32;
		pOut->_m13 = pM1->_m11 * pM2->_m13 + pM1->_m12 * pM2->_m23 + pM1->_m13 * pM2->_m33;
		pOut->_m21 = pM1->_m21 * pM2->_m11 + pM1->_m22 * pM2->_m21 + pM1->_m23 * pM2->_m31;
		pOut->_m22 = pM1->_m21 * pM2->_m12 + pM1->_m22 * pM2->_m22 + pM1->_m23 * pM2->_m32;
		pOut->_m23 = pM1->_m21 * pM2->_m13 + pM1->_m22 * pM2->_m23 + pM1->_m23 * pM2->_m33;
		pOut->_m31 = pM1->_m31 * pM2->_m11 + pM1->_m32 * pM2->_m21 + pM1->_m33 * pM2->_m31;
		pOut->_m32 = pM1->_m31 * pM2->_m12 + pM1->_m32 * pM2->_m22 + pM1->_m33 * pM2->_m32;
		pOut->_m33 = pM1->_m31 * pM2->_m13 + pM1->_m32 * pM2->_m23 + pM1->_m33 * pM2->_m33;
	}
	else
	{
		pOut = NULL;
	}

	return(pOut);
}

inline CEDXMatrix33* CEDXMatrix33Transpose(CEDXMatrix33* pOut, const CEDXMatrix33* pM)
{
	CEDXMatrix33 tmp;

	if(pOut && pM)
	{
		if(pOut == pM)
		{
			tmp = *pM;
			pM = &tmp;
		}

		// now we are sure that pOut and pM point to different memory blocks.
		pOut->_m11 = pM->_m11;
		pOut->_m12 = pM->_m21;
		pOut->_m13 = pM->_m31;

		pOut->_m21 = pM->_m12;
		pOut->_m22 = pM->_m22;
		pOut->_m23 = pM->_m32;

		pOut->_m31 = pM->_m13;
		pOut->_m32 = pM->_m23;
		pOut->_m33 = pM->_m33;
	}
	else
	{
		pOut = NULL;
	}

	return(pOut);
}

inline CEDXMatrix33* CEDXMatrix33Translation(CEDXMatrix33* pOut, float x, float y)
{
	if(pOut)
	{
		pOut->_m11 = 1.0f;	pOut->_m12 = 0.0f;	pOut->_m13 = 0.0f;
		pOut->_m21 = 0.0f;	pOut->_m22 = 1.0f;	pOut->_m23 = 0.0f;
		pOut->_m31 = x;     pOut->_m32 = y;	    pOut->_m33 = 1.0f;
	}
	else
	{
		pOut = NULL;
	}

	return(pOut);
}

inline bool CEDXMatrix33IsRotating(const CEDXMatrix33* pMat)
{
	CEASSERT(pMat);

	if((pMat->_m12 == 0.0f) && (pMat->_m21 == 0.0f))
	{
		return(false);
	}
	else
	{
		return(true);
	}
}

inline CEPointF* CEDXPointFTransformMatrix33(CEPointF* pOut, const CEPointF* pPoint, const CEDXMatrix33* pMat)
{
	CEPointF tmp;

	if(pOut && pPoint && pMat)
	{
		if(pOut == pPoint)
		{
			tmp = *pPoint;
			pPoint = &tmp;
		}

		// now we are sure that pOut and pPoint point to different memory. 
		float z = pMat->_m13 * pPoint->_x + pMat->_m23 * pPoint->_y + pMat->_m33;
		CEASSERT(0.0f != z);

		pOut->_x = (pMat->_m11 * pPoint->_x + pMat->_m21 * pPoint->_y + pMat->_m31) / z;
		pOut->_y = (pMat->_m12 * pPoint->_x + pMat->_m22 * pPoint->_y + pMat->_m32) / z;

	}
	else
	{
		pOut = NULL;
	}

	return(pOut);
}

inline CEPointBase* CEDXPointTransformMatrix33(CEPointBase* pOut, const CEPointBase* pPoint, const CEDXMatrix33* pMat)
{
	if(pOut && pPoint && pMat)
	{
		if (!CEDXMatrix33IsRotating(pMat) && pMat->_m11==1.0f && pMat->_m22==1.0f && pMat->_m33==1.0f)
		{
			pOut->_x = pPoint->_x + static_cast<INT32>(pMat->_m31);
			pOut->_y = pPoint->_y + static_cast<INT32>(pMat->_m32);
		}
		else
		{
			CEASSERT(false && "support translating matrix only.");
			pOut = NULL;
		}
	}
	else
	{
		pOut = NULL;
	}

	return(pOut);
}

inline CERectF* CEDXRectFTransformMatrix33(CERectF* pOut, const CERectF* pRect, const CEDXMatrix33* pMat)
{
	if(pOut && pRect && pMat)
	{
		// TODO: currently this code supports only non-rotating case.
		CEASSERT(! CEDXMatrix33IsRotating(pMat));

		CEPointF srcLT;		// left top
		CEPointF srcRB;		// right bottom
		CEPointF dstLT;
		CEPointF dstRB;

		srcLT._x = pRect->_x;
		srcLT._y = pRect->_y;
		srcRB._x = srcLT._x + pRect->_width;
		srcRB._y = srcLT._y + pRect->_height;

		// transform the two vertices.
		CEDXPointFTransformMatrix33(&dstLT, &srcLT, pMat);		
		CEDXPointFTransformMatrix33(&dstRB, &srcRB, pMat);

		// sort coordinates
		float minX, maxX, minY, maxY;
		if(dstLT._x < dstRB._x)
		{
			minX = dstLT._x;
			maxX = dstRB._x;
		}
		else
		{
			minX = dstRB._x;
			maxX = dstLT._x;
		}

		if(dstLT._y < dstRB._y)
		{
			minY = dstLT._y;
			maxY = dstRB._y;
		}
		else
		{
			minY = dstRB._y;
			maxY = dstLT._y;
		}

		// fill the result.
		pOut->_x = minX;
		pOut->_y = minY;
		pOut->_width = maxX - minX;
		pOut->_height = maxY - minY;
	}
	else
	{
		pOut = NULL;
	}

	return(pOut);
}

inline CERect* CEDXRectTransformMatrix33(CERect* pOut, const CERect* pRect, const CEDXMatrix33* pMat)
{
	if (pOut && pRect && pMat)
	{
		// TODO: conversion between int->float or float->int is slow!
		//       this function should be rewritten not to use these kinds of slow type casting
		CERectF inF = 
		{
			static_cast<float>(pRect->_x), 
			static_cast<float>(pRect->_y),
			static_cast<float>(pRect->_width), 
			static_cast<float>(pRect->_height)
		};

		CERectF outF;
		CEDXRectFTransformMatrix33(&outF, &inF, pMat);

		pOut->_x = static_cast<INT32>(outF._x);
		pOut->_y = static_cast<INT32>(outF._y);
		pOut->_width  = static_cast<INT32>(outF._width);
		pOut->_height = static_cast<INT32>(outF._height);
	}
	else
	{
		pOut = NULL;
	}

	return(pOut);
}

#define CE_PI 3.14159265358979323846


inline UINT8 _clampColorElm(float elm)
{
	UINT8 clampedOut = 0;
	if(elm < 0)
	{
		clampedOut = 0;
	}
	else if(elm > 255)
	{
		clampedOut = 255;
	}
	else
	{
		clampedOut = (UINT8)(elm + 0.5f);
	}

	return clampedOut;
}

inline CERGBAColor* CEDXTransformRGBAColor(CERGBAColor* pOut, const CERGBAColor* pIn, const float* pCCM)
{
	if(pOut && pIn)
	{
		if(pCCM)
		{
			UINT8 orgR = pIn->_color._components.red;
			UINT8 orgG = pIn->_color._components.green;
			UINT8 orgB = pIn->_color._components.blue;
			UINT8 orgA = pIn->_color._components.alpha;

			 //                                  | m11 m12 m13 m14 |    
			 //  (Rs Gs Bs As) = (Rf Gf Bf Af) * | m21 m22 m23 m24 | +  | m51 m52 m53 m54 |
			 //                                  | m31 m32 m33 m34 |
			 //                                  | m41 m42 m43 m44 |
			float newR = orgR * pCCM[0] + orgG * pCCM[4] + orgB * pCCM[8]  + orgA * pCCM[12] + pCCM[16] * 255;
			float newG = orgR * pCCM[1] + orgG * pCCM[5] + orgB * pCCM[9]  + orgA * pCCM[13] + pCCM[17] * 255;
			float newB = orgR * pCCM[2] + orgG * pCCM[6] + orgB * pCCM[10] + orgA * pCCM[14] + pCCM[18] * 255;
			float newA = orgR * pCCM[3] + orgG * pCCM[7] + orgB * pCCM[11] + orgA * pCCM[15] + pCCM[19] * 255;
				
			pOut->_color._components.red	= _clampColorElm(newR);
			pOut->_color._components.green = _clampColorElm(newG);
			pOut->_color._components.blue	= _clampColorElm(newB);
			pOut->_color._components.alpha = _clampColorElm(newA);

		}
		else
		{
			*pOut = *pIn;
		}
	}

	return pOut;
}






#endif // __CEDXMathTypes_h__

