///////////////////////////////////////////////////////////////////////////////
// Copyright 2011 Sony Corporation
///////////////////////////////////////////////////////////////////////////////

#ifndef CEApiImage_h
#define CEApiImage_h

//----------------------------------------------------------------
// Includes
//----------------------------------------------------------------

#include "CEApiUnknown.h"

#ifdef __cplusplus
extern "C" {
#endif

//----------------------------------------------------------------
// Prototypes
//----------------------------------------------------------------

struct ICEDXSurface;

//----------------------------------------------------------------
// Enumerators
//----------------------------------------------------------------

//----------------------------------------------------------------
// Structures
//----------------------------------------------------------------

//----------------------------------------------------------------
// Interfaces
//----------------------------------------------------------------

/*! \defgroup ICEImage ICEImage
 * @{
 */

/*!
 * ID of ICEImage
 */
#define CEComIID_ICEImage 0x1d517a0e


/*!
 * ICEImage
 */
typedef struct ICEImage
{
	const struct ICEImage_vtbl* _vtbl;
} ICEImage;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iImage	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEImage_queryInterface) (ICEImage* iImage, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEImage_addRef) (ICEImage* iImage);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEImage_release) (ICEImage* iImage);


/*!
 * <b>Summary:</b>
 * Get dimension.<br>
 * 
 * \param[in]	iImage	Specifies interface pointer
 * \param[out]	widthOut	Width.
 * \param[out]	heightOut	Height.
 *
 * \return error status
 * 
 * <b>Description:</b>
 * None
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEImage_getDimension) (ICEImage* iImage, INT32 *const widthOut, INT32 *const heightOut);


/*!
 * <b>Summary:</b>
 * Get getSurface.<br>
 * 
 * \param[in]	iImage	Specifies interface pointer
 * \param[out]	pSurface	Surface.
 *
 * \return error status
 * 
 * <b>Description:</b>
 * None
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEImage_getSurface) (ICEImage* iImage, struct ICEDXSurface **const pSurface);

/*!
 * V-table of ICEImage
 */
struct ICEImage_vtbl
{
	iCEImage_queryInterface	_queryInterface;	//!< Query interface.
	iCEImage_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEImage_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEImage_getDimension	_getDimension;	//!< Get dimension.
	iCEImage_getSurface	_getSurface;	//!< Get getSurface.
};

/*! @}
 * end of ICEImage
 */

#ifdef __cplusplus
} // end of extern "C"
#endif

//----------------------------------------------------------------
// Interface Wrappers
//----------------------------------------------------------------

#ifdef __cplusplus

/*! \defgroup CEComICEImageRef CEComICEImageRef
 * @{
 */

class CEComICEImageRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEImageRef() : _iImage(0) {}
	CEComICEImageRef(ICEImage* iOther) : _iImage(0)
	{
		if (iOther)
		{
			_iImage = iOther;
			_iImage->_vtbl->_addRef(_iImage);
		}
	}

	CEComICEImageRef(const CEComICEImageRef& other) : _iImage(0)
	{
		if (other._iImage)
		{
			_iImage = other._iImage;
			_iImage->_vtbl->_addRef(_iImage);
		}
	}

	~CEComICEImageRef()
	{
		if (_iImage)
		{
			ICEImage* tmp = _iImage;
			_iImage = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEImage* iImage;
				void* _ptr;
			} uIntf;
			uIntf.iImage = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEImage, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iImage)
				{
					ICEImage* tmp = _iImage;
					_iImage = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iImage = uIntf.iImage;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEImage* object() const	{ return _iImage; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEImage* iOther)
	{
		if (_iImage)
		{
			ICEImage* tmp = _iImage;
			_iImage = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iImage = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEImage* detach()
	{
		ICEImage* iIntf = _iImage;
		_iImage = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEImage* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iImage;
		if (_iImage)
		{
			_iImage->_vtbl->_addRef(_iImage);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEImage*() const	{ return _iImage; }
	FORCEINLINE_WITHOUT_DEBUG ICEImage& operator*() const	{ return *_iImage; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iImage == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEImage* iOther) const	{ return (_iImage != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEImage* iOther) const	{ return (_iImage == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEImageRef& operator=(const CEComICEImageRef& other)	{ return operator=(other._iImage); }

	CEComICEImageRef& operator=(const ICEImage* iOther)
	{
		if (_iImage != iOther)
		{
			if (_iImage)
			{
				ICEImage* tmp = _iImage;
				_iImage = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iImage = const_cast<ICEImage*>(iOther);
			if (_iImage)
			{
				_iImage->_vtbl->_addRef(_iImage);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEImage** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEImageRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iImage && "has a com object reference. clear first.");
		return & _iImage; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEImageRef& other)
	{
		bool result = false;
		if (_iImage)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iImage);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iImage ? _iImage->_vtbl->_queryInterface(_iImage, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iImage) { _iImage->_vtbl->_addRef(_iImage); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iImage) { _iImage->_vtbl->_release(_iImage); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getDimension(INT32 *const widthOut, INT32 *const heightOut)	{ return _iImage ? _iImage->_vtbl->_getDimension(_iImage, widthOut, heightOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getSurface(struct ICEDXSurface **const pSurface)	{ return _iImage ? _iImage->_vtbl->_getSurface(_iImage, pSurface) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEImage* _iImage;
};

/*! @}
 * end of CEComICEImageRef
 */

#endif // __cplusplus

#endif
