///////////////////////////////////////////////////////////////////////////////
// Copyright 2011 Sony Corporation
///////////////////////////////////////////////////////////////////////////////

#ifndef CEApiFSFile_h
#define CEApiFSFile_h

//----------------------------------------------------------------
// Includes
//----------------------------------------------------------------

#include "CEApiUnknown.h"

#ifdef __cplusplus
extern "C" {
#endif

//----------------------------------------------------------------
// Prototypes
//----------------------------------------------------------------


//----------------------------------------------------------------
// Enumerators
//----------------------------------------------------------------

typedef enum
{
	eCEFileSeekFrom_Begin = 0,			//!< Seek from beginning of file.
	eCEFileSeekFrom_Current,			//!< Seek from current position of file.
	eCEFileSeekFrom_End,				//!< Seek from end of file.
} eCEFileSeekFrom;

typedef enum
{
	eCEFileLockCommand_Unlock			= 0x0000,	//!< Unlock a previously locked region.

	eCEFileLockCommand_SharedLock		= 0x0001,	//!< Lock a region from write access of others.
	eCEFileLockCommand_ExeclusiveLock	= 0x0002,	//!< Lock a region for exclusive use.

	eCEFileLockCommand_FailImmediately	= 0x0010,	//!< Return immediately if file is already locked.

	eCEFileLockCommand_LockTypeMask		= 0x000f,	//!< Lock Type.
	eCEFileLockCommand_LockOptionMask	= 0x00f0,	//!< Lock Type.
} eCEFileLockCommand;

typedef enum
{
	eCEFileSyncCommand_All = 0,			//!< Sync all buffer (data + meta data).
	eCEFileSyncCommand_DataOnly,		//!< Sync only data area.
} eCEFileSyncCommand;

typedef enum
{
	eMappingProtection_ReadOnly		= 0,	//! read-only access.
} eMappingProtection;

typedef enum
{
	eViewDesiredAccess_Read		= 1, //! Read only access.
} eViewDesiredAccess;

//----------------------------------------------------------------
// Structures
//----------------------------------------------------------------

//----------------------------------------------------------------
// Interfaces
//----------------------------------------------------------------

/*! \defgroup ICEFSFile ICEFSFile
 * @{
 */

/*!
 * ID of ICEFSFile
 */
#define CEComIID_ICEFSFile 0x7cc919e0


/*!
 * ICEFile
 *
 *
 *   This interface is for operation of open-ed file.
 *   The interface is made by ICEFSFileSystem::open or ICEFSFileSystem::openByID
 *   and destroyed when reference count will be 0. (i.e. ICEFSFile::_release)
 */
typedef struct ICEFSFile
{
	const struct ICEFSFile_vtbl* _vtbl;
} ICEFSFile;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iFSFile	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEFSFile_queryInterface) (ICEFSFile* iFSFile, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEFSFile_addRef) (ICEFSFile* iFSFile);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEFSFile_release) (ICEFSFile* iFSFile);


/*!
 * <b>Summary:</b>
 * Read data from file to buffer.
 *
 * \param[in]	iFSFile	Specifies interface pointer
 * \param[in]	buf	Pointer to buffer.  This buffer must be larger than nbyte.
 * \param[in]	nbyte	Byte count to read.
 * \param[out]	numReadOut	The number of bytes actually read.
 *
 * \return Error status
 *
 * <b>Description:</b>
 * None
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 *
 */
typedef CEHResult (*iCEFSFile_read) (ICEFSFile* iFSFile, void* buf, unsigned int nbyte, int *const numReadOut);


/*!
 * <b>Summary:</b>
 * Read data from buffer to file.
 *
 * \param[in]	iFSFile	Specifies interface pointer
 * \param[in]	buf	Pointer to buffer.  This buffer must be larger than nbyte.
 * \param[in]	nbyte	Byte count to read.
 * \param[out]	numWriteOut	The number of bytes actually write.
 *
 * \return Error status
 *
 * <b>Description:</b>
 * None
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 *
 */
typedef CEHResult (*iCEFSFile_write) (ICEFSFile* iFSFile, const void* buf, unsigned int nbyte, int *const numWrittenOut);


/*!
 * <b>Summary:</b>
 * Seek the file pointer.
 *
 * \param[in]	iFSFile	Specifies interface pointer
 * \param[in]	offset	Byte offset.  This value can be negative.
 * \param[in]	whence	Specifies the position seek from.
 * \param[out]	positionOut	Byte offset of the new position.
 *
 * \return Error status
 *
 * <b>Description:</b>
 * None
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 *
 */
typedef CEHResult (*iCEFSFile_seek) (ICEFSFile* iFSFile, ce_off_t offset, eCEFileSeekFrom whence, ce_off_t *const positionOut);


/*!
 * <b>Summary:</b>
 * Lock/unlock the file.
 *
 * \param[in]	iFSFile	Specifies interface pointer
 * \param[in]	cmd	Bit-or of eCEFileLockCommand
 *
 * \return Error status
 *
 * <b>Description:</b>
 * None
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 *
 */
typedef CEHResult (*iCEFSFile_lock) (ICEFSFile* iFSFile, UINT32 operation);


/*!
 * <b>Summary:</b>
 * Change the file size.
 *
 * \param[in]	iFSFile	Specifies interface pointer
 * \param[in]	length	New file length.
 *
 * \return Error status
 *
 * <b>Description:</b>
 * None
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 *
 */
typedef CEHResult (*iCEFSFile_truncate) (ICEFSFile* iFSFile, UINT32 length);


/*!
 * <b>Summary:</b>
 * Flush the cached data to disk.
 *
 * \param[in]	iFSFile	Specifies interface pointer
 * \param[in]	cmd	Operation mode by eCEFileSyncCommand.
 *
 * \return Error status
 *
 * <b>Description:</b>
 * None
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 *
 */
typedef CEHResult (*iCEFSFile_sync) (ICEFSFile* iFSFile, eCEFileSyncCommand cmd);


/*!
 * <b>Summary:</b>
 * Get file status and file size from the opened file.
 *
 * \param[in]	iFSFile	Specifies interface pointer
 * \param[out]	status	Bit field.  (Reserved for the future. Currently, always returns 0)
 * \param[out]	fileSize	File size.  This will be returned when the file exists and is a regurar file.
 *
 * \return Error status
 *
 * <b>Description:</b>
 * None
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 *
 */
typedef CEHResult (*iCEFSFile_getStatus) (ICEFSFile* iFSFile, int* status, UINT32* fileSize);

/*!
 * V-table of ICEFSFile
 */
struct ICEFSFile_vtbl
{
	iCEFSFile_queryInterface	_queryInterface;	//!< Query interface.
	iCEFSFile_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEFSFile_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEFSFile_read	_read;
	iCEFSFile_write	_write;
	iCEFSFile_seek	_seek;
	iCEFSFile_lock	_lock;
	iCEFSFile_truncate	_truncate;
	iCEFSFile_sync	_sync;
	iCEFSFile_getStatus	_getStatus;
};

/*! @}
 * end of ICEFSFile
 */

/*! \defgroup ICEFSFileMappingFactory ICEFSFileMappingFactory
 * @{
 */

/*!
 * ID of ICEFSFileMappingFactory
 */
#define CEComIID_ICEFSFileMappingFactory 0xc5a0ce74


/*!
 * ICEFSFileMappingFactory
 *
 *
 *   This interface is for operation of memory mapped file.
 *   The interface can be retrieved by ICEFSFile::queryInterface with CEComIID_ICEFSFileMappingFactory.
 */
typedef struct ICEFSFileMappingFactory
{
	const struct ICEFSFileMappingFactory_vtbl* _vtbl;
} ICEFSFileMappingFactory;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iFSFileMappingFactory	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEFSFileMappingFactory_queryInterface) (ICEFSFileMappingFactory* iFSFileMappingFactory, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEFSFileMappingFactory_addRef) (ICEFSFileMappingFactory* iFSFileMappingFactory);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEFSFileMappingFactory_release) (ICEFSFileMappingFactory* iFSFileMappingFactory);

typedef CEHResult (*iCEFSFileMappingFactory_mapFile) (ICEFSFileMappingFactory* iFSFileMappingFactory, eMappingProtection mappingProtection, eViewDesiredAccess viewDesiredAccess, struct ICEFSFileMapping* *const fileMappingOut);

/*!
 * V-table of ICEFSFileMappingFactory
 */
struct ICEFSFileMappingFactory_vtbl
{
	iCEFSFileMappingFactory_queryInterface	_queryInterface;	//!< Query interface.
	iCEFSFileMappingFactory_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEFSFileMappingFactory_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEFSFileMappingFactory_mapFile	_mapFile;
};

/*! @}
 * end of ICEFSFileMappingFactory
 */

/*! \defgroup ICEFSFileMapping ICEFSFileMapping
 * @{
 */

/*!
 * ID of ICEFSFileMapping
 */
#define CEComIID_ICEFSFileMapping 0x107c772d

/*!
 * ICEFSFileMapping
 */
typedef struct ICEFSFileMapping
{
	const struct ICEFSFileMapping_vtbl* _vtbl;
} ICEFSFileMapping;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iFSFileMapping	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEFSFileMapping_queryInterface) (ICEFSFileMapping* iFSFileMapping, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEFSFileMapping_addRef) (ICEFSFileMapping* iFSFileMapping);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEFSFileMapping_release) (ICEFSFileMapping* iFSFileMapping);

typedef CEHResult (*iCEFSFileMapping_getData) (ICEFSFileMapping* iFSFileMapping, void* *const pDataOut, size_t *const mappingSizeOut);

/*!
 * V-table of ICEFSFileMapping
 */
struct ICEFSFileMapping_vtbl
{
	iCEFSFileMapping_queryInterface	_queryInterface;	//!< Query interface.
	iCEFSFileMapping_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEFSFileMapping_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEFSFileMapping_getData	_getData;
};

/*! @}
 * end of ICEFSFileMapping
 */

#ifdef __cplusplus
} // end of extern "C"
#endif

//----------------------------------------------------------------
// Interface Wrappers
//----------------------------------------------------------------

#ifdef __cplusplus

/*! \defgroup CEComICEFSFileRef CEComICEFSFileRef
 * @{
 */

class CEComICEFSFileRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEFSFileRef() : _iFSFile(0) {}
	CEComICEFSFileRef(ICEFSFile* iOther) : _iFSFile(0)
	{
		if (iOther)
		{
			_iFSFile = iOther;
			_iFSFile->_vtbl->_addRef(_iFSFile);
		}
	}

	CEComICEFSFileRef(const CEComICEFSFileRef& other) : _iFSFile(0)
	{
		if (other._iFSFile)
		{
			_iFSFile = other._iFSFile;
			_iFSFile->_vtbl->_addRef(_iFSFile);
		}
	}

	~CEComICEFSFileRef()
	{
		if (_iFSFile)
		{
			ICEFSFile* tmp = _iFSFile;
			_iFSFile = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEFSFile* iFSFile;
				void* _ptr;
			} uIntf;
			uIntf.iFSFile = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEFSFile, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iFSFile)
				{
					ICEFSFile* tmp = _iFSFile;
					_iFSFile = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iFSFile = uIntf.iFSFile;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEFSFile* object() const	{ return _iFSFile; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEFSFile* iOther)
	{
		if (_iFSFile)
		{
			ICEFSFile* tmp = _iFSFile;
			_iFSFile = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iFSFile = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEFSFile* detach()
	{
		ICEFSFile* iIntf = _iFSFile;
		_iFSFile = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEFSFile* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iFSFile;
		if (_iFSFile)
		{
			_iFSFile->_vtbl->_addRef(_iFSFile);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEFSFile*() const	{ return _iFSFile; }
	FORCEINLINE_WITHOUT_DEBUG ICEFSFile& operator*() const	{ return *_iFSFile; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iFSFile == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEFSFile* iOther) const	{ return (_iFSFile != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEFSFile* iOther) const	{ return (_iFSFile == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEFSFileRef& operator=(const CEComICEFSFileRef& other)	{ return operator=(other._iFSFile); }

	CEComICEFSFileRef& operator=(const ICEFSFile* iOther)
	{
		if (_iFSFile != iOther)
		{
			if (_iFSFile)
			{
				ICEFSFile* tmp = _iFSFile;
				_iFSFile = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iFSFile = const_cast<ICEFSFile*>(iOther);
			if (_iFSFile)
			{
				_iFSFile->_vtbl->_addRef(_iFSFile);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEFSFile** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEFSFileRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iFSFile && "has a com object reference. clear first.");
		return & _iFSFile; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEFSFileRef& other)
	{
		bool result = false;
		if (_iFSFile)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iFSFile);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iFSFile ? _iFSFile->_vtbl->_queryInterface(_iFSFile, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iFSFile) { _iFSFile->_vtbl->_addRef(_iFSFile); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iFSFile) { _iFSFile->_vtbl->_release(_iFSFile); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult read(void* buf, unsigned int nbyte, int *const numReadOut)	{ return _iFSFile ? _iFSFile->_vtbl->_read(_iFSFile, buf, nbyte, numReadOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult write(const void* buf, unsigned int nbyte, int *const numWrittenOut)	{ return _iFSFile ? _iFSFile->_vtbl->_write(_iFSFile, buf, nbyte, numWrittenOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult seek(ce_off_t offset, eCEFileSeekFrom whence, ce_off_t *const positionOut)	{ return _iFSFile ? _iFSFile->_vtbl->_seek(_iFSFile, offset, whence, positionOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult lock(UINT32 operation)	{ return _iFSFile ? _iFSFile->_vtbl->_lock(_iFSFile, operation) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult truncate(UINT32 length)	{ return _iFSFile ? _iFSFile->_vtbl->_truncate(_iFSFile, length) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult sync(eCEFileSyncCommand cmd)	{ return _iFSFile ? _iFSFile->_vtbl->_sync(_iFSFile, cmd) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getStatus(int* status, UINT32* fileSize)	{ return _iFSFile ? _iFSFile->_vtbl->_getStatus(_iFSFile, status, fileSize) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEFSFile* _iFSFile;
};

/*! @}
 * end of CEComICEFSFileRef
 */

/*! \defgroup CEComICEFSFileMappingFactoryRef CEComICEFSFileMappingFactoryRef
 * @{
 */

class CEComICEFSFileMappingFactoryRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEFSFileMappingFactoryRef() : _iFSFileMappingFactory(0) {}
	CEComICEFSFileMappingFactoryRef(ICEFSFileMappingFactory* iOther) : _iFSFileMappingFactory(0)
	{
		if (iOther)
		{
			_iFSFileMappingFactory = iOther;
			_iFSFileMappingFactory->_vtbl->_addRef(_iFSFileMappingFactory);
		}
	}

	CEComICEFSFileMappingFactoryRef(const CEComICEFSFileMappingFactoryRef& other) : _iFSFileMappingFactory(0)
	{
		if (other._iFSFileMappingFactory)
		{
			_iFSFileMappingFactory = other._iFSFileMappingFactory;
			_iFSFileMappingFactory->_vtbl->_addRef(_iFSFileMappingFactory);
		}
	}

	~CEComICEFSFileMappingFactoryRef()
	{
		if (_iFSFileMappingFactory)
		{
			ICEFSFileMappingFactory* tmp = _iFSFileMappingFactory;
			_iFSFileMappingFactory = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEFSFileMappingFactory* iFSFileMappingFactory;
				void* _ptr;
			} uIntf;
			uIntf.iFSFileMappingFactory = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEFSFileMappingFactory, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iFSFileMappingFactory)
				{
					ICEFSFileMappingFactory* tmp = _iFSFileMappingFactory;
					_iFSFileMappingFactory = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iFSFileMappingFactory = uIntf.iFSFileMappingFactory;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEFSFileMappingFactory* object() const	{ return _iFSFileMappingFactory; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEFSFileMappingFactory* iOther)
	{
		if (_iFSFileMappingFactory)
		{
			ICEFSFileMappingFactory* tmp = _iFSFileMappingFactory;
			_iFSFileMappingFactory = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iFSFileMappingFactory = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEFSFileMappingFactory* detach()
	{
		ICEFSFileMappingFactory* iIntf = _iFSFileMappingFactory;
		_iFSFileMappingFactory = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEFSFileMappingFactory* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iFSFileMappingFactory;
		if (_iFSFileMappingFactory)
		{
			_iFSFileMappingFactory->_vtbl->_addRef(_iFSFileMappingFactory);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEFSFileMappingFactory*() const	{ return _iFSFileMappingFactory; }
	FORCEINLINE_WITHOUT_DEBUG ICEFSFileMappingFactory& operator*() const	{ return *_iFSFileMappingFactory; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iFSFileMappingFactory == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEFSFileMappingFactory* iOther) const	{ return (_iFSFileMappingFactory != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEFSFileMappingFactory* iOther) const	{ return (_iFSFileMappingFactory == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEFSFileMappingFactoryRef& operator=(const CEComICEFSFileMappingFactoryRef& other)	{ return operator=(other._iFSFileMappingFactory); }

	CEComICEFSFileMappingFactoryRef& operator=(const ICEFSFileMappingFactory* iOther)
	{
		if (_iFSFileMappingFactory != iOther)
		{
			if (_iFSFileMappingFactory)
			{
				ICEFSFileMappingFactory* tmp = _iFSFileMappingFactory;
				_iFSFileMappingFactory = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iFSFileMappingFactory = const_cast<ICEFSFileMappingFactory*>(iOther);
			if (_iFSFileMappingFactory)
			{
				_iFSFileMappingFactory->_vtbl->_addRef(_iFSFileMappingFactory);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEFSFileMappingFactory** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEFSFileMappingFactoryRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iFSFileMappingFactory && "has a com object reference. clear first.");
		return & _iFSFileMappingFactory; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEFSFileMappingFactoryRef& other)
	{
		bool result = false;
		if (_iFSFileMappingFactory)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iFSFileMappingFactory);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iFSFileMappingFactory ? _iFSFileMappingFactory->_vtbl->_queryInterface(_iFSFileMappingFactory, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iFSFileMappingFactory) { _iFSFileMappingFactory->_vtbl->_addRef(_iFSFileMappingFactory); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iFSFileMappingFactory) { _iFSFileMappingFactory->_vtbl->_release(_iFSFileMappingFactory); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult mapFile(eMappingProtection mappingProtection, eViewDesiredAccess viewDesiredAccess, struct ICEFSFileMapping* *const fileMappingOut)	{ return _iFSFileMappingFactory ? _iFSFileMappingFactory->_vtbl->_mapFile(_iFSFileMappingFactory, mappingProtection, viewDesiredAccess, fileMappingOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEFSFileMappingFactory* _iFSFileMappingFactory;
};

/*! @}
 * end of CEComICEFSFileMappingFactoryRef
 */

/*! \defgroup CEComICEFSFileMappingRef CEComICEFSFileMappingRef
 * @{
 */

class CEComICEFSFileMappingRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEFSFileMappingRef() : _iFSFileMapping(0) {}
	CEComICEFSFileMappingRef(ICEFSFileMapping* iOther) : _iFSFileMapping(0)
	{
		if (iOther)
		{
			_iFSFileMapping = iOther;
			_iFSFileMapping->_vtbl->_addRef(_iFSFileMapping);
		}
	}

	CEComICEFSFileMappingRef(const CEComICEFSFileMappingRef& other) : _iFSFileMapping(0)
	{
		if (other._iFSFileMapping)
		{
			_iFSFileMapping = other._iFSFileMapping;
			_iFSFileMapping->_vtbl->_addRef(_iFSFileMapping);
		}
	}

	~CEComICEFSFileMappingRef()
	{
		if (_iFSFileMapping)
		{
			ICEFSFileMapping* tmp = _iFSFileMapping;
			_iFSFileMapping = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEFSFileMapping* iFSFileMapping;
				void* _ptr;
			} uIntf;
			uIntf.iFSFileMapping = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEFSFileMapping, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iFSFileMapping)
				{
					ICEFSFileMapping* tmp = _iFSFileMapping;
					_iFSFileMapping = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iFSFileMapping = uIntf.iFSFileMapping;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEFSFileMapping* object() const	{ return _iFSFileMapping; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEFSFileMapping* iOther)
	{
		if (_iFSFileMapping)
		{
			ICEFSFileMapping* tmp = _iFSFileMapping;
			_iFSFileMapping = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iFSFileMapping = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEFSFileMapping* detach()
	{
		ICEFSFileMapping* iIntf = _iFSFileMapping;
		_iFSFileMapping = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEFSFileMapping* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iFSFileMapping;
		if (_iFSFileMapping)
		{
			_iFSFileMapping->_vtbl->_addRef(_iFSFileMapping);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEFSFileMapping*() const	{ return _iFSFileMapping; }
	FORCEINLINE_WITHOUT_DEBUG ICEFSFileMapping& operator*() const	{ return *_iFSFileMapping; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iFSFileMapping == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEFSFileMapping* iOther) const	{ return (_iFSFileMapping != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEFSFileMapping* iOther) const	{ return (_iFSFileMapping == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEFSFileMappingRef& operator=(const CEComICEFSFileMappingRef& other)	{ return operator=(other._iFSFileMapping); }

	CEComICEFSFileMappingRef& operator=(const ICEFSFileMapping* iOther)
	{
		if (_iFSFileMapping != iOther)
		{
			if (_iFSFileMapping)
			{
				ICEFSFileMapping* tmp = _iFSFileMapping;
				_iFSFileMapping = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iFSFileMapping = const_cast<ICEFSFileMapping*>(iOther);
			if (_iFSFileMapping)
			{
				_iFSFileMapping->_vtbl->_addRef(_iFSFileMapping);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEFSFileMapping** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEFSFileMappingRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iFSFileMapping && "has a com object reference. clear first.");
		return & _iFSFileMapping; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEFSFileMappingRef& other)
	{
		bool result = false;
		if (_iFSFileMapping)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iFSFileMapping);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iFSFileMapping ? _iFSFileMapping->_vtbl->_queryInterface(_iFSFileMapping, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iFSFileMapping) { _iFSFileMapping->_vtbl->_addRef(_iFSFileMapping); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iFSFileMapping) { _iFSFileMapping->_vtbl->_release(_iFSFileMapping); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getData(void* *const pDataOut, size_t *const mappingSizeOut)	{ return _iFSFileMapping ? _iFSFileMapping->_vtbl->_getData(_iFSFileMapping, pDataOut, mappingSizeOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEFSFileMapping* _iFSFileMapping;
};

/*! @}
 * end of CEComICEFSFileMappingRef
 */

#endif // __cplusplus

#endif
