# -*- tab-width: 8 -*- 
use strict;

########################################################################################################
### read ccd
########################################################################################################
### Data structure:
###
### %CCD = ('file1.ccd' => $FILE1_REF, 'file2.ccd' => $FILE2_REF, ...);
###
### %FILE = ('name'          => $NAME,
###          'precomp'       => $PRECOMP,
###          'include-h'     => $INCLUDES_REF, # generated .h file will contain #include
###          'include-cpp'   => $INCLUDES_REF, # generated .cpp file will contain #include
###          'class'         => $CLASSES_REF,
###          'module'        => $MODULE_REF,
###          'wizard'        => $WIZARD_REF);
###
### @INCLUDES = ($INCLUDE1, $INCLUDE2, ...);
###
### @CLASSES = ($CLASS_REF1, $CLASS_REF2, ...);
###
### %CLASS = ('name'            => $NAME,
###           'interface'       => $INTERFACES_REF,
###           'inheritance'     => $INHERITANCE,
###           'embed'           => $EMBED,
###           'marshal'         => $MARSHAL,
###           'marshal-factory' => $MARSHAL_FACTORY,
###           'creatable'       => $CLASSES_REF,
###           'mt-safe'         => 0 or 1 (1 means lock around refcount),
###           'referred'        => 0 or 1 (1 means only referred not defined),
###           'no-allocator'    => 0 or 1 (1 means allocator not defined),
###           'fnd-trans'       => $FND_TRANS,
###           'function-prefix' => $FUNCTION_PREFIX,
###
###  @INTERFACES = ($INTERFACE_REF1, $INTERFACE_REF2, ...);
###
###  %INTERFACE = ('name'   => $NAME,
###                'vptr'   => $VPTR);
###
###  %MODULE = ('name'      => $NAME,
###             'factory'   => $FACTORIES_REF,
###             'header'    => $HEADER_FILE
###             'validate'  => $VALIDATION_TYPE
###             'referred'  => 0 or 1 (1 means only referred not defined);
###
###  %FACTORY = ( 'class'       => $CLASS_REF,
###               'interface'   => $INTERFACE_REF,
###               'creatable'   => $CLASSES_REF,
###               'apartment'   => $APARTMENT);
###
###  %WIZARD =  ( 'component'  => $COMPONENT_NAME  # 'BrowserComponent' or 'BrowserManager' or 'BrowserLibrary'
###               'version'    => $VERSION # e.g. "1.00"
###               'options'    => $OPTIONS # "no-allocator java-to-c"
###             );
###
###
###
###               Note) @OBJECTS = @$OBJECTS_REF, %OBJECT = %$OBJECT_REF
###
########################################################################################################

sub readCcdFile
{
    my($CCD_REF, $IDL_REF, $CPPHEADER_REF, $CCD_FULLPATH, $REFERRED, @INCLUDEPATHS) = @_;

    $CCD_FULLPATH =~ s/\\/\//g;
    while ($CCD_FULLPATH =~ s/\/\//\//g)
    {
	# do nothing
    }

    ### delete path for database
    my($CCD_FILE) = &deletePath($CCD_FULLPATH);
    if ($CCD_REF->{$CCD_FILE})
    {
	### this ccd file is still reading.
	return;
    }

    # ignore unknown idl files, if INCLUDEPATHS has '#ignore'
    my($FORCE_READ) = 0;
    if ($INCLUDEPATHS[0] eq '#ignore')
    {
        $FORCE_READ = 1;
        
        shift @INCLUDEPATHS;
    }

#    print(STDERR "Info: Read file '$CCD_FULLPATH'.\n");

    my(@LINES);
    if ($CCD_FULLPATH)
    {
	open(FP, $CCD_FULLPATH) || die("Error: Can not open file '$CCD_FULLPATH' as input\n");
	@LINES = <FP>;
	close(FP);

	$CCD_REF->{$CCD_FILE}->{'name'} = $CCD_FULLPATH;
    }

    my($IS_CSD) = $CCD_FULLPATH =~ /\.csd$/;

    foreach my $LINE (@LINES)
    {
	### CCD Wizard Info
	if ($LINE =~ /^; CCD_TYPE\s*:\s*(\S*)\s*/)
	{
	    $CCD_REF->{$CCD_FILE}{'wizard'}{'component'} = $1;
	}
	elsif ($LINE =~ /^; CCDWIZ_VERSION\s*:\s*(.*)\s*/)
	{
	    $CCD_REF->{$CCD_FILE}{'wizard'}{'version'} = $1;
	}
	elsif ($LINE =~ /^; CCDWIZ_OPTIONS\s*:\s*(.*)$/)
	{
	    $CCD_REF->{$CCD_FILE}{'wizard'}{'options'} = $1;
	}
    }

    my $parser = new Parser(join("", @LINES));
    my($MODE);
    my(@MODES);
    push(@MODES, 'normal');

    while (1)
    {
	### get line.
	my ($sexp, $loc) = $parser->next();
	last unless $sexp;
	my $LINE = &writeSExp($sexp);
	my($LINE_NO) = $loc->{line};

	### update parse mode.
	$MODE = $MODES[$#MODES];

	### if class define
	if (($LINE =~ /^\s*\(\s*(comclass)\s*(\w+)\s+\(([^\)]+)\)\s*([^\)]*)\)\s*$/ ||
	    $LINE =~ /^\s*\(\s*(comclass)\s*(\w+)\s+\(([\s*\(\w*\s*\w*\)\s*]+)\)\s*([^\)]*)\)\s*$/) && !$IS_CSD)
	{
	    my($INTERFACE_NAME_LIST, $OPTION_LIST) = ($3, $4);
	    my($CLASS_REF) = {'name' => $2, 'mt-safe' => 1, 'referred' => $REFERRED, 'no-allocator' => 0, 'delegator-mirror' => 0};

	    ### perse intarface list
	    if ($INTERFACE_NAME_LIST =~ /\(\s*\w+\s*\w+\)/)
	    {
		my(@INTERFACE_PAIRS) = split(/[\(\)]\s*/, $INTERFACE_NAME_LIST);
		my($INTERFACE_PAIR);
		foreach $INTERFACE_PAIR (@INTERFACE_PAIRS)
		{

		    if ($INTERFACE_PAIR)
		    {
			my($INTERFACE_NAME, $INTERFACE_VPTR) = split(/\s+/, $INTERFACE_PAIR);
			if ($INTERFACE_NAME =~ /^\w+$/ && $INTERFACE_VPTR =~ /^\w+$/)
			{
			    if (&getInterfaceHash($IDL_REF, $INTERFACE_NAME))
			    {
				my($INTERFACE) = {'name' => $INTERFACE_NAME, 'vptr' => $INTERFACE_VPTR };
				push(@{$CLASS_REF->{'interface'}}, $INTERFACE);
			    }
			    else
			    {
				dieMessage($FORCE_READ, "$CCD_FULLPATH:$LINE_NO: Can not find interface '$INTERFACE_NAME'.\n");
				if ($FORCE_READ)
				{
				    my($INTERFACE) = {'name' => $INTERFACE_NAME};
				    push(@{$CLASS_REF->{'interface'}}, $INTERFACE);
				}
			    }
			}
			else
			{
			    die("Error: $CCD_FULLPATH:$LINE_NO: Syntax error. unknown interface '$INTERFACE_NAME' or vptr '$INTERFACE_VPTR'.\n");
			}
		    }
		}
	    }
	    else
	    {
		my(@INTERFACES_NAME) = split(/\s+/, $INTERFACE_NAME_LIST);
		my($INTERFACE_NAME);
		foreach $INTERFACE_NAME (@INTERFACES_NAME)
		{
		    if ($INTERFACE_NAME =~ /^\w+$/)
		    {
			if (&getInterfaceHash($IDL_REF, $INTERFACE_NAME))
			{
			    my($VPTR); # = &getVptrName($INTERFACE_NAME);
			    my($INTERFACE) = {'name' => $INTERFACE_NAME, 'vptr' => $VPTR };
			    push(@{$CLASS_REF->{'interface'}}, $INTERFACE);
			}
			else
			{
			    dieMessage($FORCE_READ, "$CCD_FULLPATH:$LINE_NO: Can not find interface '$INTERFACE_NAME'.\n");
			    if ($FORCE_READ)
			    {
				my($INTERFACE) = {'name' => $INTERFACE_NAME};
				push(@{$CLASS_REF->{'interface'}}, $INTERFACE);
			    }
			}
		    }
		    else
		    {
			die("Error: $CCD_FULLPATH:$LINE_NO: Syntax error. unknown interface '$INTERFACE_NAME'.\n");
		    }
		}
	    }

	    ### perse option list
	    if ($OPTION_LIST)
	    {
		my(@OPTIONS) = split(/\s+/, $OPTION_LIST);
		while(@OPTIONS)
		{
		    #for JNI.
		    $CLASS_REF->{'java-to-c'} = 0;
		    #for JNI.
		    $CLASS_REF->{'c-to-java'} = 0;

		    my($PARAM) = shift(@OPTIONS);
		    if ($PARAM eq ':inherit')
		    {
			$CLASS_REF->{'inheritance'} = &parseOptionParameter($CPPHEADER_REF, $CCD_FULLPATH, $LINE_NO, $CLASS_REF->{'inheritance'}, $CLASS_REF->{'embed'}, ':inherit', shift(@OPTIONS), $FORCE_READ);
		    }
		    elsif ($PARAM eq ':embed')
		    {
			$CLASS_REF->{'embed'} = &parseOptionParameter($CPPHEADER_REF, $CCD_FULLPATH, $LINE_NO, $CLASS_REF->{'inheritance'}, $CLASS_REF->{'embed'}, ':embed', shift(@OPTIONS), $FORCE_READ);
		    }
		    elsif ($PARAM eq ':std-marshal' || $PARAM eq ':lw-marshal' || $PARAM eq ':ipc-marshal')
		    {
			$CLASS_REF->{'marshal'} = $PARAM;
# 			if ($PARAM =~ /std/)
# 			{
# 			    die("Error: $CCD_FULLPATH:$LINE_NO: not implemented '$PARAM'.\n");
# 			}
		    }
		    elsif ($PARAM eq ':std-factory' || $PARAM eq ':lw-factory')
		    {
			$CLASS_REF->{'marshal-factory'} = $PARAM;
# 			if ($PARAM =~ /std/)
# 			{
# 			    die("Error: $CCD_FULLPATH:$LINE_NO: not implemented '$PARAM'.\n");
# 			}
		    }
		    elsif ($PARAM eq ':no-mt-safe')
		    {
			$CLASS_REF->{'mt-safe'} = 0;
		    }
		    elsif ($PARAM eq ':mt-safe')
		    {
			# keep compatibility
		    }
		    elsif ($PARAM eq ':no-allocator')
		    {
			$CLASS_REF->{'no-allocator'} = 1;
		    }
		    elsif ($PARAM eq ':delegator-mirror')
		    {
			my($DELEGATED) = shift(@OPTIONS);
			my($DELEGATED_CLASS_REF);
			#if (!&getClassHash($CPPHEADER_REF, $DELEGATED))
			if ($DELEGATED =~ /:\w+/)
			{
			    $DELEGATED_CLASS_REF = $CLASS_REF;
			    unshift(@OPTIONS, $DELEGATED);
			}
			else
			{
			    $DELEGATED_CLASS_REF = \&getClassHash($CPPHEADER_REF, $DELEGATED);
			}
			$CLASS_REF->{'delegator-mirror'} = $DELEGATED_CLASS_REF;
		    }
		    elsif ($PARAM eq ':trans-to-fnd' || $PARAM eq ':trans-from-fnd')
		    {
			$CLASS_REF->{'fnd-trans'} = $PARAM;
		    }
		    #for JNI.
		    elsif ($PARAM eq ':java-to-c')
		    {
			$CLASS_REF->{'java-to-c'} = 1;
		    }
		    #for JNI.
		    elsif ($PARAM eq ':c-to-java')
		    {
			$CLASS_REF->{'c-to-java'} = 1;
		    }
		    elsif ($PARAM eq ':function-prefix')
		    {
			$CLASS_REF->{'function-prefix'} = shift(@OPTIONS);
		    }
		    else
		    {
			if ($CLASS_REF->{'marshal-factory'})
			{
			    if (&getComClassHash($CCD_REF, $PARAM))
			    {
				my(%CLASS_HASH) = &getComClassHash($CCD_REF, $PARAM);
				my(@INTERFACES) = @{$CLASS_HASH{'interface'}};
				if ($#INTERFACES == 0)
				{
				    push(@{$CLASS_REF->{'creatable'}}, \%CLASS_HASH);
				}
				else
				{
				    die("Error: $CCD_FULLPATH:$LINE_NO: too much or too less interfaces for marshaller class: '$PARAM'.\n");
				}
			    }
			    else
			    {
				die("Error: $CCD_FULLPATH:$LINE_NO: unknown class to create as marshaller: '$PARAM'.\n");
			    }
			}
			else
			{
			    die("Error: $CCD_FULLPATH:$LINE_NO: unknown option '$PARAM'.\n");
			}
		    }
		}
	    }

	    if ($CLASS_REF->{'mt-safe'} != 0)
	    {
		push(@{$CCD_REF->{$CCD_FILE}->{'include-h'}}, 'CEComAtomicSupport.h');
	    }
	    push(@{$CCD_REF->{$CCD_FILE}->{'class'}}, $CLASS_REF);
	}

	### if including header
	elsif ($LINE =~ /^\s*\(\s*(load)\s+\"(\w+\.(idl|h|ccd|csd))\"\s*\)\s*$/)
	{
	    my($INCLUDE) = $2;
	    if (!$REFERRED && !($2 =~ /\.ccd$/))
	    {
		push(@{$CCD_REF->{$CCD_FILE}->{'include-h'}}, $INCLUDE);
	    }
	    else
	    {
		push(@{$CCD_REF->{$CCD_FILE}->{'include-cpp'}}, $INCLUDE);
	    }

	    my($WANT_FILE) = &findIncludingFile($INCLUDE, @INCLUDEPATHS);
	    if (!$WANT_FILE)
	    {
		dieMessage($FORCE_READ, "$CCD_FULLPATH:$LINE_NO: Can not find file '$INCLUDE' as input.\n");
	    }

	    if ($INCLUDE =~ /^(\w+).idl$/)
	    {
		if ($WANT_FILE)
		{
		    &readIdlFile($IDL_REF, $WANT_FILE, @INCLUDEPATHS);
		}
		else
		{
		    print "Info: make dummy interface idl for '$INCLUDE'.\n";
		    &makeDummyIdlFile($IDL_REF, $WANT_FILE, @INCLUDEPATHS);
		}
	    }
	    elsif ($INCLUDE =~ /^(\w+).h$/)
	    {
		&readHeaderFile($CPPHEADER_REF, $WANT_FILE, @INCLUDEPATHS);
	    }
	    elsif ($INCLUDE =~ /^(\w+).(ccd|csd)$/)
	    {
		&readCcdFile($CCD_REF, $IDL_REF, $CPPHEADER_REF, $WANT_FILE, 1, @INCLUDEPATHS);
	    }
	}

	### if precompile header
	elsif ($LINE =~ /^\s*\(\s*(load-precompiled-header)\s+\"(\w+\.(h))\"\s*\)\s*$/)
	{
	    if (!$REFERRED)
	    {
		my($PRECOMP) = $2;
		if ($CCD_REF->{$CCD_FILE}->{'precomp'})
		{
		    die("Error: $CCD_FULLPATH:$LINE_NO: Duplicate decralation 'precomp'.\n");
		}
		else
		{
		    $CCD_REF->{$CCD_FILE}->{'precomp'} = $PRECOMP;

		    my($WANT_FILE) = &findIncludingFile($PRECOMP, @INCLUDEPATHS);
		    if (!$WANT_FILE)
		    {
			die("Error: $CCD_FULLPATH:$LINE_NO: Can not find precompiled header file '$PRECOMP' as input.\n");
		    }
		}
	    }
	}

	### if module definition
	elsif (($LINE =~ /^\s*\(\s*(comserver)\s*(\w+)\s+(\(\s*[\(\s*\w+\s+\w+\s+:\w+\s*\)\s+.*]+)\s*\)\s*$/ ||
		$LINE =~ /^\s*\(\s*(comserver)\s*(\w+)\s+([\:\-\w]+)\s+(\(\s*[\(\s*\w+\s+\w+\s+:\w+\s*\)\s+.*]+)\s*\)\s*$/) && $IS_CSD)
	{
	    my($MODULE_NAME) = $2;
	    my($VALIDATION_TYPE);
	    my($CLASS_FACTORY_OPTION_LIST) = $3;
	    if ($3 eq ':factory-id-validation' || $3 eq ':creatable-id-validation')
	    {
		$VALIDATION_TYPE = $3;
		$CLASS_FACTORY_OPTION_LIST = $4;
	    }
	    my($HEADER_FILE) = &getFileName($CCD_FULLPATH, 'csd', 'h');
	    my($MODULE_REF) = {'name' => $MODULE_NAME, 'header' => $HEADER_FILE, 'validation' => $VALIDATION_TYPE, 'referred' => $REFERRED};
	    my(@FACTORIES);

	    if ($REFERRED)
	    {
		## do nothing
	    }
	    elsif ($CLASS_FACTORY_OPTION_LIST =~ /\s*\(([\(\s*\w+\s+\w+\s+:\w+\s*\)\s+.*\)]+)\)\s*/)
	    {
		$CLASS_FACTORY_OPTION_LIST = $1;
		my(@CLASS_FACTORY_INFO_SET_ARRAY) = split(/\) \(/, $CLASS_FACTORY_OPTION_LIST);
		my($CLASS_FACTORY_INFO_SET);
		foreach $CLASS_FACTORY_INFO_SET (@CLASS_FACTORY_INFO_SET_ARRAY)
		{
		    if ($CLASS_FACTORY_INFO_SET =~ /\s*\((.*)\)\s*(.*)$/)
		    {
			my($FACTORY_NAME, $FACTORY_INTERFACE, $APARTMENT_NAME) = split(/\s+/, $1);
			my(@MODCLASSES_NAME) = split(/\s+/, $2);

			if (&getComClassHash($CCD_REF, $FACTORY_NAME))
			{
			    my(%FACTORY_CLASS_HASH) = &getComClassHash($CCD_REF, $FACTORY_NAME);

			    if (&getInterfaceHash($IDL_REF, $FACTORY_INTERFACE))
			    {
				my(@CREATABLES);
				my($MODCLASS_NAME);
				foreach $MODCLASS_NAME (@MODCLASSES_NAME)
				{
				    if (&getComClassHash($CCD_REF, $MODCLASS_NAME))
				    {
					my(%CLASS_HASH) = &getComClassHash($CCD_REF, $MODCLASS_NAME);
					push (@CREATABLES, \%CLASS_HASH);
				    }
				    else
				    {
					die "Error: com class not found: $MODCLASS_NAME";
				    }
				}
				my(%FACTORY_INTERFACE_HASH) = &getInterfaceHash($IDL_REF, $FACTORY_INTERFACE);

				my($FACTORY_REF) = { 'class' => \%FACTORY_CLASS_HASH, 'interface' => \%FACTORY_INTERFACE_HASH, 'creatable' => \@CREATABLES, 'apartment' => $APARTMENT_NAME };
				push(@{$MODULE_REF->{'factory'}}, $FACTORY_REF);
			    }
			    else
			    {
				die("Error: $CCD_FULLPATH:$LINE_NO: Can not find factory interface '$FACTORY_INTERFACE'.\n");
			    }
			}
			else
			{
			    die("Error: $CCD_FULLPATH:$LINE_NO: Can not find factory class '$FACTORY_NAME'.\n");
			}
		    }
		}
	    }
	    else
	    {
		die "Error: $CCD_FILE:$LINE_NO: invalid option: $CLASS_FACTORY_OPTION_LIST";
	    }

	    my(@SORTED) = sort { $a->{'name'} cmp $b->{'name'} }(@FACTORIES);
	    my($FACTORY_REF, $PREV_NAME);
	    foreach $FACTORY_REF (@SORTED)
	    {
		if ($FACTORY_REF->{'name'} ne $PREV_NAME)
		{
		    push(@{$MODULE_REF->{'factory'}}, $FACTORY_REF);
		    $PREV_NAME = $FACTORY_REF->{'name'};
		}
	    }
	    $CCD_REF->{$CCD_FILE}->{'module'} = $MODULE_REF;
	    push(@{$CCD_REF->{$CCD_FILE}->{'include-h'}}, 'CEComSingleIntfObjectT.h');
	    push(@{$CCD_REF->{$CCD_FILE}->{'include-h'}}, 'CEComAtomicSupport.h');
	    push(@{$CCD_REF->{$CCD_FILE}->{'include-h'}}, 'CEComPrimitiveSupport.h');
	}

	elsif ($LINE =~ /^\s*$/)
	{
	    ### empty line.
	}

	else
	{
	    print "$LINE\n\n";;
	    die("Error: $CCD_FULLPATH:$LINE_NO: Syntax error.\n");
	}
    }
}

sub parseOptionParameter
{
    my($CPPHEADER_REF, $CCD_FULLPATH, $LINE_NO, $INHERITANCE, $EMBED, $OPTION, $PARAM, $FORCE_READ) = @_;

    if ($INHERITANCE || $EMBED)
    {
	die("Error: $CCD_FULLPATH:$LINE_NO: Duplicate option '$OPTION'.\n");
    }
    if ($PARAM)
    {
	if (!&getClassHash($CPPHEADER_REF, $PARAM))
	{
            dieMessage($FORCE_READ, "$CCD_FULLPATH:$LINE_NO: Can not find class '$PARAM'.\n");
	}
    }
    else
    {
	die("Error: $CCD_FULLPATH:$LINE_NO: Can not find parameter of '$OPTION'.\n");
    }

    return $PARAM;
}

########################################################################################################
### read CPP header file
########################################################################################################
### Data structure:
###
### %CPPHEADER = ('file1.h' => $FILE1_REF, 'file2.h' => $FILE2_REF, ...);
###
### %FILE = ('name'  => $NAME,
###          'class' => $CLASSES_REF);
###
### @CLASSES = ($CLASS_REF1, $CLASS_REF2, ...);
###
### %CLASS = ('name'        => $NAME,
###           'constructor' => $CONSTRUCTORS_REF);
###
### @CONSTRUCTORS = ($CONSTRUCTOR1, $CONSTRUCTOR2, ...);
###
### %CONSTRUCTOR = ('parameter' => $PARAMETER,
###                 'access'    => $ACCESS,
###                 'pre_ctor_ifdef' => $PRE_CTOR_IFDEFS,
###                 'post_ctor_ifdef' => $POST_CTOR_IFDEFS);
###
###
###               Note) @OBJECTS = @$OBJECTS_REF, %OBJECT = %$OBJECT_REF
###
########################################################################################################

sub readHeaderFile
{
    my($CPPHEADER_REF, $CPPHEADER_FULLPATH, @INCLUDEPATHS) = @_;

    ### delete path for databese.
    my($CPPHEADER_FILE) = &deletePath($CPPHEADER_FULLPATH);
    if ($CPPHEADER_REF->{$CPPHEADER_FILE})
    {
	### this header file is still reading.
	return;
    }

#    print(STDERR "Info: Read file '$CPPHEADER_FILE'.\n");

    my(@LINES);
    if ($CPPHEADER_FULLPATH)
    {
	open(FP, $CPPHEADER_FULLPATH) || die("Error: Can not open file '$CPPHEADER_FULLPATH' as input\n");
	@LINES = <FP>;
	close(FP);

	$CPPHEADER_REF->{$CPPHEADER_FILE}->{'name'} = $CPPHEADER_FULLPATH;
    }

    my($LINE_NO) = 0;
    my($LINE);

    my($MODE);
    my(@MODES);
    push(@MODES, 'normal');

    my($CLASS_NAME);
    my($INDENT) = 0;
    my($ACCESS) = 'private';

    my(@IFDEFS) = ();
    my($IFDEF_INDENT) = 0;
    my($CLASS_NO) = 0;

    ### parse file.
    while (@LINES)
    {
	### update parse mode.
	$MODE = $MODES[$#MODES];

	### increment line number.
	$LINE_NO++;

	### get line.
	$LINE = shift(@LINES);

	if (($INDENT == 1) && $CLASS_NAME && ($MODE eq 'class') && ($LINE =~ /^\s*($CLASS_NAME)\s*\(([^\)]*).*$/))
	{
	    my(@PRE_IFDEFS) = @IFDEFS;
	    my(@EMPTY_ARRAY) = ();
	    my($CONSTRUCTOR_REF) = {'parameter' => $2, 'access' => $ACCESS, 'pre_ctor_ifdef' => \@PRE_IFDEFS, 'post_ctor_ifdef' => \@EMPTY_ARRAY};
	    @IFDEFS = ();

	    push(@{$CPPHEADER_REF->{$CPPHEADER_FILE}->{'class'}->[$CLASS_NO]->{'constructor'}}, $CONSTRUCTOR_REF);
	}

	### if starting class.
	elsif (($MODE eq 'normal') && ($LINE =~ /^(class)\s+(\w+)\s*(\{?)\s*$/))
	{
	    $CLASS_NAME = $2;
	    $ACCESS = 'private';
	    @MODES = &pushMode('class', $CPPHEADER_FILE, $LINE_NO, @MODES);
	    my(@EMPTY_ARRAY) = ();
	    my($CLASS_REF) = {'name' => $CLASS_NAME, 'constructor' => \@EMPTY_ARRAY};
	    push(@{$CPPHEADER_REF->{$CPPHEADER_FILE}->{'class'}}, $CLASS_REF);
	}
	elsif (($MODE eq 'normal') && ($LINE =~ /^(class)\s+(\w+)\s*\:\s*\w+\s*.*(\{?)\s*$/))
	{
	    $CLASS_NAME = $2;
	    $ACCESS = 'private';
	    @MODES = &pushMode('class', $CPPHEADER_FILE, $LINE_NO, @MODES);
	    my(@EMPTY_ARRAY) = ();
	    my($CLASS_REF) = {'name' => $CLASS_NAME, 'constructor' => \@EMPTY_ARRAY};
	    push(@{$CPPHEADER_REF->{$CPPHEADER_FILE}->{'class'}}, $CLASS_REF);
	}

	### if ending class.
	elsif (($MODE eq 'class') && ($INDENT == 1) && ($LINE =~ /^\}\s*\;\s*$/))
	{
 	    my(@CONSTRUCTORS) = @{$CPPHEADER_REF->{$CPPHEADER_FILE}->{'class'}->[$CLASS_NO]->{'constructor'}};
 	    if ($#CONSTRUCTORS == -1)
 	    {
		my(@EMPTY_ARRAY) = ();
 		my($DEF_CONSTRUCTOR_REF) = {'parameter' => '', 'access' => 'public', 'pre_ctor_ifdef' => \@EMPTY_ARRAY, 'post_ctor_ifdef' => \@EMPTY_ARRAY};
 		push(@{$CPPHEADER_REF->{$CPPHEADER_FILE}->{'class'}->[$CLASS_NO]->{'constructor'}}, $DEF_CONSTRUCTOR_REF);
 	    }
	    $CLASS_NO++;
	    pop(@MODES);
	}

	### if access control operator.
	elsif (($MODE eq 'class') && ($INDENT == 1) && ($LINE =~ /^\s*(private|protected|public)\s*\:\s*$/))
	{
	    $ACCESS = $1;
	}

	### if '#ifdef..' section
	elsif (($MODE eq 'class') && ($INDENT == 1) && $LINE =~ /^\#(if|else|endif)/)
	{
	    push(@IFDEFS, $LINE);
	    if ($LINE =~ /^(\s*)\#if/)
	    {
		$IFDEF_INDENT++;
	    }
	    elsif ($LINE =~ /^(\s*)\#endif/)
	    {
		$IFDEF_INDENT--;
		if ($IFDEF_INDENT == 0)
		{
		    my(@CTORS) = @{$CPPHEADER_REF->{$CPPHEADER_FILE}->{'class'}->[$CLASS_NO]->{'constructor'}};

		    if ($#CTORS >= 0)
		    {
			my(@POST_IFDEFS) = @IFDEFS;

			my(@POST_CTOR_IFDEFS) = @{$CPPHEADER_REF->{$CPPHEADER_FILE}->{'class'}->[$CLASS_NO]->{'constructor'}->[$#CTORS]->{'post_ctor_ifdef'}};
			if ($#POST_CTOR_IFDEFS == -1)
			{
			    $CPPHEADER_REF->{$CPPHEADER_FILE}->{'class'}->[$CLASS_NO]->{'constructor'}->[$#CTORS]->{'post_ctor_ifdef'} = \@POST_IFDEFS;
			}
		    }
		    @IFDEFS = ();
		}
	    }
	}

	### count indentation.
	if ($INDENT += $LINE =~ /\{/)
	{
	}
	if ($INDENT -= $LINE =~ /\}/)
	{
	}
    }
}

########################################################################################################
### utility
########################################################################################################

sub getClassHash
{
    my($CPPHEADER_REF, $NAME) = @_;

    my($CPPHEADER_FILE);
    foreach $CPPHEADER_FILE (keys(%$CPPHEADER_REF))
    {
	my($CLASSES_REF) = $CPPHEADER_REF->{$CPPHEADER_FILE}->{'class'};
	my($CLASS_REF);
	foreach $CLASS_REF (@$CLASSES_REF)
	{
	    my(%CLASS) = %$CLASS_REF;
	    if ($CLASS{'name'} eq $NAME)
	    {
		return %CLASS;
	    }
	}
    }

    return 0;
}

sub getComClassHash
{
    my($CCD_REF, $NAME) = @_;
    my($CLASS_REF);

    my($CCD_FILE);
    foreach $CCD_FILE (keys(%$CCD_REF))
    {
	my($CLASSES_REF) = $CCD_REF->{$CCD_FILE}->{'class'};
	my($CLASS_REF);
	foreach $CLASS_REF (@$CLASSES_REF)
	{
	    my(%CLASS) = %$CLASS_REF;
	    if ($CLASS{'name'} eq $NAME)
	    {
		return %CLASS;
	    }
	}
    }

    return 0;
}

sub getComModuleHash
{
    my($CCD_REF) = @_;
    my($CCD_FILE);

    foreach $CCD_FILE (keys(%$CCD_REF))
    {
	my($MODULE_REF) = $CCD_REF->{$CCD_FILE}->{'module'};
	if ($MODULE_REF)
	{
	    return %$MODULE_REF;
	}
    }

    return 0;
}

sub dieMessage
{
    my($FORCE_RUN, $MSG) = @_;
    if ($FORCE_RUN)
    {
        print 'Warning: '.$MSG;
    }
    else
    {
        die 'Error: '.$MSG;
    }
}

sub readAndWriteSExp
{
    my ($s) = @_;
    return join("\n", map { &writeSExp($_); } &parse($s));
}

sub parse($)
{
    my ($s) = @_;
    my $parser = new Parser($s);
    return $parser->parse();
}

sub tokens($)
{
    my ($s) = @_;
    my $tknz = new Tokenizer($s);
    my @tokens;
    while ($_ = $tknz->get)
    {
	 push(@tokens, $_);
    }
    return \@tokens;
}

sub writeSExp
{
    my ($sexp) = @_;
    if (ref($sexp) eq 'ARRAY') {
	return &writeList($sexp);
    } elsif ($sexp->{type} eq 'string') {
	return '"' . $sexp->{str} . '"';
    } else {
	return $sexp->{str};
    }
}

sub writeList
{
    my ($list) = @_;
    my $s = '(';
    $s .= &writeSExp(shift @$list) if @$list > 0;
    foreach (@$list) {
	$s .= ' ' . &writeSExp($_);
    }
    $s .= ')';
    return $s;
}

package Parser;

sub new
{
    my ($class, $input) = @_;
    my $tokenizer = new Tokenizer($input);
    my $self = {tokenizer => $tokenizer};
    return bless $self, $class;
}

sub parse
{
    my ($self) = @_;
    my @sexp;
    while (1) {
	my ($sexp, $loc) = $self->next();
	return @sexp unless $sexp;
	push(@sexp, $sexp);
    }
}

sub next
{
    my ($self) = @_;
    my $token = $self->{tokenizer}->get();
    if (!$token) {
	return undef;
    } elsif ($token->{str} eq "(") {
	return $self->parseList($token), $token->{location};
    } elsif ($token->{str} eq ")") {
	die "Error: $token->{location}->{line}:$token->{location}->{col}: unexpected closing parenthesis"
    } else {
	return $token, $token->{location};
    }
}

sub parseList
{
    my ($self, $beginToken) = @_;
    my @list;
    while (my $token = $self->{tokenizer}->get()) {
	return \@list if ($token->{str} eq ")");
	$self->{tokenizer}->unget($token);
	my ($sexp, $loc) = $self->next();
	push(@list, $sexp);
    }
    die "Error: $beginToken->{location}->{line}:$beginToken->{location}->{col}: unbalanced parenthesis"
}

package Tokenizer;

sub new
{
    my ($class, $input) = @_;
    my $location = {line => 1, col => 0};
    my $self = {input => $input,
                location => $location,
    		next => undef};
    return bless $self, $class;
}

sub get
{
    my ($self) = @_;
    if ($self->{next}) {
	my $token = $self->{next};
	$self->{next} = undef;
	return $token;
    } else {
	return $self->next();
    }
}

sub unget()
{
    my ($self, $token) = @_;
    $self->{next} = $token;
}

sub next
{
    my ($self) = @_;
    $self->skipWhiteSpaceAndComments();
    my $type;
    my $str;
    $_ = $self->{input};
    if (m/\A\(/) {
	$type = 'open';
	$str = $&;
    } elsif (m/\A\)/) {
	$type = 'close';
	$str = $&;
    } elsif (m/\A"([^"]+)"/) {
	$type = 'string';
	$str = $1;
    } elsif (m/\A[^\(\)\s]+/) {
	$type = 'symbol';
	$str = $&;
    } else {
	return;
    }
    my $token = {type => $type, str => $str, location => $self->{location}};
    $self->advance($&);
    return $token;
}

sub skipWhiteSpaceAndComments
{
    my ($self) = @_;
    while (1) {
	$_ = $self->{input};
	if (m/\A\s+/) {
	     $self->advance($&);
	} elsif (m/\A;[^\n]*/) {
	     $self->advance($&);
	} else {
	    return;
	}
    }
}

sub advance
{
    my ($self, $str) = @_;
    my @lines = split(/\n/, $str, -1);
    my $line = $self->{location}->{line};
    my $col = $self->{location}->{col};
    $line += @lines - 1;
    $col = 1 if @lines > 1;
    $col += length($lines[-1]);
    $self->{location} = {line => $line, col => $col};
    $self->{input} = substr($self->{input}, length($str));
}

sub location
{
    my ($self) = @_;
    return $self->{location};
}

########################################################################################################
### gen{j}comclass argument utility
########################################################################################################
package ceccdutil;

sub getArgs
{
    my($argEnvVal, $INCLUDEPATHS, $OUTDIR, $CWDIR, $RULE_FILE_FLAG, $RULE_FILE_OUTDIR) = @_;
    my($myArgStr) = $ENV{$argEnvVal};
    $myArgStr =~ s/[\s]*$//;
    print $argEnvVal . "=[" . $myArgStr  . "]\n";
    my(@myArgs);
    push @myArgs, $1 while $myArgStr =~ /((\"[^\"]*?\"|\'[^\']*?\'|\S)+)\s*/g;

#my(@tmyargs) = @myArgs;
#while (@tmyargs)
#{
#	print "++[" . shift(@tmyargs) . "]++";
#	print "...\n";
#}

    &parseArgs(\@myArgs, $INCLUDEPATHS, $OUTDIR, $CWDIR, $RULE_FILE_FLAG, $RULE_FILE_OUTDIR);
    return;
}

sub parseArgs
{
    my($args, $INCLUDEPATHS, $OUTDIR, $CWDIR, $RULE_FILE_FLAG, $RULE_FILE_OUTDIR) = @_;
    my($PARAM);

    while (@$args)
    {
	$PARAM = shift(@$args);
	$PARAM =~ s/\"//g;

	### if -I option
	if ($PARAM =~ /^(-I)([\/\.\w].*)$/)
	{
	    push(@$INCLUDEPATHS, $2);
	}
	
	### if -d option
	elsif ($PARAM =~ /^(-d)$/)
	{
	    $PARAM = shift(@$args);
	    if ($PARAM)
	    {
		$$OUTDIR = $PARAM;
	    }
	    else
	    {
		die("Error: Specify output directory name after '-d' option\n");
	    }
	}
	elsif ($PARAM =~ /^(-cwd)$/)
	{
	    $PARAM = shift(@$args);
	    if ($PARAM)
	    {
		$$CWDIR = $PARAM;
	    }
	    else
	    {
		die("Error: Specify current directory name after '-cwd' option\n");
	    }
	}
	
	### if -MMD option
	elsif ($PARAM =~ /^(-MMD)$/)
	{
	    $$RULE_FILE_FLAG = 1;
	}
	elsif ($PARAM =~ /^(-WMMD)$/)
	{
	    $$RULE_FILE_FLAG = 1;
	    $PARAM = shift(@$args);
	    if ($PARAM)
	    {
		$$RULE_FILE_OUTDIR = $PARAM;
	    }
	    else
	    {
		die("Error: Specify output directory name after '-WMMD' option\n");
	    }
	}
	
	# @@response_env_variable
	elsif ($PARAM =~ /^(@@)([\S]+)$/)
	{
	    &getArgs($2, $INCLUDEPATHS, $OUTDIR, $CWDIR, $RULE_FILE_FLAG, $RULE_FILE_OUTDIR);
	}
	
	elsif ($PARAM =~ /^-/)
	{
	    die("Error: Unknown option '$PARAM'\n");
	}
	else
	{
	    die("Error: Unknown parameter '$PARAM'\n");
	}
    }
}


1; ### for required call.
