///////////////////////////////////////////////////////////////////////////////
// Copyright 2004 Sony Corporation
///////////////////////////////////////////////////////////////////////////////

#ifndef __CEUSTR_MACROS_H__
#define __CEUSTR_MACROS_H__

#include "ceutypes.h"

/*

ex...:

UTF-16 represents USC-4's U+10000 - U+10FFFF area characters as
surrogate pair:

high surrogate : U+D800 - U+DBFF
low surrogate  : U+DC00 - U+DFFF

After U+110000 characters are not contained in UTF-16.

The following is a "how to convert UCS-4 (U+10000 - U+10FFFF) to
UFT-16 surrogate pair:

1. UTF4-CHAR - 0x10000(U'). U+10000 => 0x000000U, U+10FFFF => 0xFFFFF.
2. Convert U' to 20 bit binary array.
3. high-surrogate: W1, low-surrogate: W2:

U' = yyyyyyyyyyxxxxxxxxxx (UCS-4 = U - 0x10000 (20bit))
W1 = 110110yyyyyyyyyy     (high surrogateFD800 ... DBFF)
W2 = 110111xxxxxxxxxx     (low surrogate FDC00 ... DFFF)

ex. U+12345 -> D8 08 DF 45 (UTF-16BE) or  08 D8 45 DF (UTF-16 LE).
*/

#define CEU_U16_IS_SURROGATE_LEAD(_c)    (((_c)&0xfffffc00)==0xd800)
#define CEU_U16_IS_SURROGATE_TRAIL(_c)   (((_c)&0xfffffc00)==0xdc00)
#define CEU_U16_IS_SURROGATE(_c)    (CEU_U16_IS_SURROGATE_LEAD(_c)||CEU_U16_IS_SURROGATE_TRAIL(_c)) 

#define CEU_U32_UNKNOWN_CHAR 0x003f  

#define CEU_U32_IS_SURROGATE(_c)  ((_c) & 0xff0000)
#define CEU_U32_TO_U16_LEN(_c) ((0xff0000 & _c)?2:1)


#define CEU_U16_SURROGATE_PAIR_TO_U32(_lead, _trail, _c32) \
{ \
	_c32 = (0x3ff & _lead) << 10; \
	_c32 |= (0x3ff & _trail); \
	_c32 += 0x10000; \
}

//! _u16buff2[0]: surrogate lead char16 or single char16.
//! _u16buff2[1]: surrogate trail char16.
#define CEU_U32_TO_U16(_c32, _u16buff2, _lenOut) \
{ \
	UTF32CHAR __tc32 = _c32; \
	if ( 0x10000 <= __tc32 ) \
	{ \
		__tc32 -= 0x10000; \
		_u16buff2[0] = (UTF16CHAR)(0xD800 | __tc32 >> 10); \
		_u16buff2[1] = (UTF16CHAR)(0xDC00 | 0x3ff & __tc32); \
		_lenOut = 2; \
	} \
	else \
	{ \
		_u16buff2[0] = (UTF16CHAR)((0x0ffff & _c32)); \
		_lenOut = 1; \
	} \
}

//! _u16buff2[0]: surrogate lead char16 or single char16.
//! _u16buff2[1]: surrogate trail char16.
#define CEU_U16_TO_U32(_c16buff2, _num16, _c32Out) \
{ \
	_c32Out = CEU_U32_UNKNOWN_CHAR; \
	if ( _num16 == 1 ) { \
		if ( !CEU_U16_IS_SURROGATE(_c16buff2[0]) ){ \
			_c32Out = (UTF32CHAR) _c16buff2[0]; \
		} \
	} \
	else { \
		if ( CEU_U16_IS_SURROGATE_LEAD(_c16buff2[0]) ) { \
			if ( CEU_U16_IS_SURROGATE_TRAIL(_c16buff2[1]) ) { \
				CEU_U16_SURROGATE_PAIR_TO_U32(_c16buff2[0], _c16buff2[1], _c32Out); \
			} \
		} \
	} \
}

#endif //__CEUSTR_MACROS

