///////////////////////////////////////////////////////////////////////////////
// Copyright 2011 Sony Corporation
///////////////////////////////////////////////////////////////////////////////

#ifndef ICETime_h
#define ICETime_h

//----------------------------------------------------------------
// Includes
//----------------------------------------------------------------

#include "CEApiUnknown.h"

#ifdef __cplusplus
extern "C" {
#endif

//----------------------------------------------------------------
// Prototypes
//----------------------------------------------------------------

struct ICEClockDevice;
struct ICEClock;
struct ICETime;
struct ICETimeBase;
struct ICETimeBaseDevice;

//----------------------------------------------------------------
// Enumerators
//----------------------------------------------------------------

//----------------------------------------------------------------
// Structures
//----------------------------------------------------------------

//----------------------------------------------------------------
// Interfaces
//----------------------------------------------------------------

/*! \defgroup ICEClockDevice ICEClockDevice
 * @{
 */

/*!
 * ID of ICEClockDevice
 */
#define CEComIID_ICEClockDevice 0x9f93b413

/*!
 * ICEClockDevice
 */
typedef struct ICEClockDevice
{
	const struct ICEClockDevice_vtbl* _vtbl;
} ICEClockDevice;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iClockDevice	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEClockDevice_queryInterface) (ICEClockDevice* iClockDevice, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEClockDevice_addRef) (ICEClockDevice* iClockDevice);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEClockDevice_release) (ICEClockDevice* iClockDevice);


	/*!
	 * Get the current clock.
	 * \param[in] iClockDevice Specifies interface pointer
	 * \param[out] iClockOut return the current valid ICEClock
	 * \return error code
	 */
typedef CEHResult (*iCEClockDevice_getCurrentClock) (ICEClockDevice* iClockDevice, struct ICEClock* *const iClockOut);

/*!
 * V-table of ICEClockDevice
 */
struct ICEClockDevice_vtbl
{
	iCEClockDevice_queryInterface	_queryInterface;	//!< Query interface.
	iCEClockDevice_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEClockDevice_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEClockDevice_getCurrentClock	_getCurrentClock;
};

/*! @}
 * end of ICEClockDevice
 */

/*! \defgroup ICEClock ICEClock
 * @{
 */

/*!
 * ID of ICEClock
 */
#define CEComIID_ICEClock 0x8e1a96aa


/*!
 * ICEClock
 * \note 
 * An ICEClock is an absolute clock that has a beginning from some point and 
 * continues to tick indefinitely. An ICEClock does not stop and never returns tick values out of incremental sequence.
 */
typedef struct ICEClock
{
	const struct ICEClock_vtbl* _vtbl;
} ICEClock;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iClock	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEClock_queryInterface) (ICEClock* iClock, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEClock_addRef) (ICEClock* iClock);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEClock_release) (ICEClock* iClock);


	/*!
	 * Get the clockDevice that created this clock.
	 * \param[in] iClock Specifies interface pointer
	 * \param[out] iClockDeviceOut return the ICEClockDevice that created this clock
	 * \return error code
	 */
typedef CEHResult (*iCEClock_getClockDevice) (ICEClock* iClock, struct ICEClockDevice* *const iClockDeviceOut);


	/*!
	 * Get the current time on this clock. 
	 * \param[in] iClock Specifies interface pointer
	 * \param[out] iTimeOut A point on the timeline of this clock.
	 * \return error code
	 */
typedef CEHResult (*iCEClock_getCurrentTime) (ICEClock* iClock, struct ICETime* *const iTimeOut);


	/*!
	 * Compare two iCETime values. The comparison will fail if the comparing clock is not shared
	 * by both ICETime values.
	 * \param[in] iClock Specifies interface pointer
	 * \param[in] iTime1 An ICETime value to be compared;
	 * \param[in] iTime2 An ICETime value to be compared;
	 * \param[out] resultOut resultOut < 0 iTime1 is less than iTime2, resultOut = 0 : iTime1 equals iTime2, resultOut > 1 : iTime1 is greater than iTime2
	 * \return error code
	 */
typedef CEHResult (*iCEClock_compare) (ICEClock* iClock, struct ICETime* iTime1, struct ICETime* iTime2, INT32* resultOut);


	/*!
	 * Get tickcount in milliseconds.
	 * \param[in] iClock Specifies interface pointer
	 * \param[out] msecOut milliseconds.
	 * \return error code
	 */
typedef CEHResult (*iCEClock_getTickcountInMilliseconds) (ICEClock* iClock, UINT32 *const msecOut);

/*!
 * V-table of ICEClock
 */
struct ICEClock_vtbl
{
	iCEClock_queryInterface	_queryInterface;	//!< Query interface.
	iCEClock_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEClock_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEClock_getClockDevice	_getClockDevice;
	iCEClock_getCurrentTime	_getCurrentTime;
	iCEClock_compare	_compare;
	iCEClock_getTickcountInMilliseconds	_getTickcountInMilliseconds;
};

/*! @}
 * end of ICEClock
 */

/*! \defgroup ICETime ICETime
 * @{
 */

/*!
 * ID of ICETime
 */
#define CEComIID_ICETime 0x5053a6c5


/*!
* ICETime
*
* \note
* ICETime represents a point on the timeline
*/
typedef struct ICETime
{
	const struct ICETime_vtbl* _vtbl;
} ICETime;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iTime	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCETime_queryInterface) (ICETime* iTime, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCETime_addRef) (ICETime* iTime);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCETime_release) (ICETime* iTime);


	/*!
	 * \param[in] iTime Specifies interface pointer
	 * \param[out] secOut milliseconds.
	 * \param[out] usecOut microseconds.
	 * \return error code
	 */
typedef CEHResult (*iCETime_getTimeValue) (ICETime* iTime, INT32 *const secOut, UINT32 *const usecOut);


	/*!
	 * \param[in] iTime Specifies interface pointer
	 * \param[out] iClockOut Clock associated with this ICETime
	 * \return error code
	 */
typedef CEHResult (*iCETime_getClock) (ICETime* iTime, struct ICEClock* *const iClockOut);

/*!
 * V-table of ICETime
 */
struct ICETime_vtbl
{
	iCETime_queryInterface	_queryInterface;	//!< Query interface.
	iCETime_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCETime_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCETime_getTimeValue	_getTimeValue;
	iCETime_getClock	_getClock;
};

/*! @}
 * end of ICETime
 */

/*! \defgroup ICETimeBaseDevice ICETimeBaseDevice
 * @{
 */

/*!
 * ID of ICETimeBaseDevice
 */
#define CEComIID_ICETimeBaseDevice 0x0e0785e8

/*!
 * ICETimeBaseDevice
 */
typedef struct ICETimeBaseDevice
{
	const struct ICETimeBaseDevice_vtbl* _vtbl;
} ICETimeBaseDevice;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iTimeBaseDevice	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCETimeBaseDevice_queryInterface) (ICETimeBaseDevice* iTimeBaseDevice, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCETimeBaseDevice_addRef) (ICETimeBaseDevice* iTimeBaseDevice);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCETimeBaseDevice_release) (ICETimeBaseDevice* iTimeBaseDevice);


	/*!
	 * if settimeofday() is called between this call and previous one, new instance of ICETimeBase will be returned
	 * otherwise, same instance of ICETimeBase will be returned
	 * \param[in] iTimeBaseDevice Specifies interface pointer
	 * \param[out] iCETimeBaseOut Time base
	 * \return error code
	 * \note This described behavior may change, do not depend on it.
	 */
typedef CEHResult (*iCETimeBaseDevice_getTimeBase) (ICETimeBaseDevice* iTimeBaseDevice, struct ICETimeBase* *const iCETimeBaseOut);

/*!
 * V-table of ICETimeBaseDevice
 */
struct ICETimeBaseDevice_vtbl
{
	iCETimeBaseDevice_queryInterface	_queryInterface;	//!< Query interface.
	iCETimeBaseDevice_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCETimeBaseDevice_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCETimeBaseDevice_getTimeBase	_getTimeBase;
};

/*! @}
 * end of ICETimeBaseDevice
 */

/*! \defgroup ICETimeBase ICETimeBase
 * @{
 */

/*!
 * ID of ICETimeBase
 */
#define CEComIID_ICETimeBase 0xeeba021f

/*!
 * ICETimeBase
 */
typedef struct ICETimeBase
{
	const struct ICETimeBase_vtbl* _vtbl;
} ICETimeBase;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iTimeBase	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCETimeBase_queryInterface) (ICETimeBase* iTimeBase, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCETimeBase_addRef) (ICETimeBase* iTimeBase);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCETimeBase_release) (ICETimeBase* iTimeBase);


	/*!
	 * \param[out]] iCETimeBaseDeviceOut Associated time base
	 * \param[in]	iTimeBase	Specifies interface pointer
	 *
	 * \return error code
	 */
typedef CEHResult (*iCETimeBase_getTimeBaseDevice) (ICETimeBase* iTimeBase, struct ICETimeBaseDevice* *const iCETimeBaseDeviceOut);


	/*!
	 * \param[in] iTimeBase Specifies interface pointer
	 * \param[out] iTimeOut The point in time this ICETimeBase was instantiated
	 * \return error code
	 */
typedef CEHResult (*iCETimeBase_getInstantiatedTime) (ICETimeBase* iTimeBase, struct ICETime* *const iTimeOut);


	/*!
	 * Import a time using this ICETimeBase. The imported time should be the # of secs and usecs
	 * from 1970.1.1 0:00 for this to work properly
	 * \param[in] iTimeBase Specifies interface pointer
	 * \param[in] sec Seconds from 1970.1.1 0:00
	 * \param[in] usec USeconds from 1970.1.1 0:00
	 * \param[out] iTimeOut Imported time value
	 * \return error code
	 */
typedef CEHResult (*iCETimeBase_importTimeValue) (ICETimeBase* iTimeBase, INT32 sec, UINT32 usec, struct ICETime* *const iTimeOut);


	/*!
	 * Normalize and export a time value.
	 * \param[in] iTimeBase Specifies interface pointer
	 * \param[in] iTime Time value to be exported
	 * \param[out] secOut Time from 1970.1.1 0:00 GMT
	 * \param[out] usecOut Time from 1970.1.1 0:00 GMT
	 * \return error code
	 */
typedef CEHResult (*iCETimeBase_exportTimeValue) (ICETimeBase* iTimeBase, struct ICETime* iTime, INT32* secOut, UINT32* usecOut);

/*!
 * V-table of ICETimeBase
 */
struct ICETimeBase_vtbl
{
	iCETimeBase_queryInterface	_queryInterface;	//!< Query interface.
	iCETimeBase_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCETimeBase_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCETimeBase_getTimeBaseDevice	_getTimeBaseDevice;
	iCETimeBase_getInstantiatedTime	_getInstantiatedTime;
	iCETimeBase_importTimeValue	_importTimeValue;
	iCETimeBase_exportTimeValue	_exportTimeValue;
};

/*! @}
 * end of ICETimeBase
 */

#ifdef __cplusplus
} // end of extern "C"
#endif

//----------------------------------------------------------------
// Interface Wrappers
//----------------------------------------------------------------

#ifdef __cplusplus

/*! \defgroup CEComICEClockDeviceRef CEComICEClockDeviceRef
 * @{
 */

class CEComICEClockDeviceRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEClockDeviceRef() : _iClockDevice(0) {}
	CEComICEClockDeviceRef(ICEClockDevice* iOther) : _iClockDevice(0)
	{
		if (iOther)
		{
			_iClockDevice = iOther;
			_iClockDevice->_vtbl->_addRef(_iClockDevice);
		}
	}

	CEComICEClockDeviceRef(const CEComICEClockDeviceRef& other) : _iClockDevice(0)
	{
		if (other._iClockDevice)
		{
			_iClockDevice = other._iClockDevice;
			_iClockDevice->_vtbl->_addRef(_iClockDevice);
		}
	}

	~CEComICEClockDeviceRef()
	{
		if (_iClockDevice)
		{
			ICEClockDevice* tmp = _iClockDevice;
			_iClockDevice = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEClockDevice* iClockDevice;
				void* _ptr;
			} uIntf;
			uIntf.iClockDevice = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEClockDevice, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iClockDevice)
				{
					ICEClockDevice* tmp = _iClockDevice;
					_iClockDevice = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iClockDevice = uIntf.iClockDevice;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEClockDevice* object() const	{ return _iClockDevice; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEClockDevice* iOther)
	{
		if (_iClockDevice)
		{
			ICEClockDevice* tmp = _iClockDevice;
			_iClockDevice = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iClockDevice = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEClockDevice* detach()
	{
		ICEClockDevice* iIntf = _iClockDevice;
		_iClockDevice = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEClockDevice* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iClockDevice;
		if (_iClockDevice)
		{
			_iClockDevice->_vtbl->_addRef(_iClockDevice);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEClockDevice*() const	{ return _iClockDevice; }
	FORCEINLINE_WITHOUT_DEBUG ICEClockDevice& operator*() const	{ return *_iClockDevice; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iClockDevice == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEClockDevice* iOther) const	{ return (_iClockDevice != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEClockDevice* iOther) const	{ return (_iClockDevice == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEClockDeviceRef& operator=(const CEComICEClockDeviceRef& other)	{ return operator=(other._iClockDevice); }

	CEComICEClockDeviceRef& operator=(const ICEClockDevice* iOther)
	{
		if (_iClockDevice != iOther)
		{
			if (_iClockDevice)
			{
				ICEClockDevice* tmp = _iClockDevice;
				_iClockDevice = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iClockDevice = const_cast<ICEClockDevice*>(iOther);
			if (_iClockDevice)
			{
				_iClockDevice->_vtbl->_addRef(_iClockDevice);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEClockDevice** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEClockDeviceRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iClockDevice && "has a com object reference. clear first.");
		return & _iClockDevice; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEClockDeviceRef& other)
	{
		bool result = false;
		if (_iClockDevice)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iClockDevice);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iClockDevice ? _iClockDevice->_vtbl->_queryInterface(_iClockDevice, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iClockDevice) { _iClockDevice->_vtbl->_addRef(_iClockDevice); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iClockDevice) { _iClockDevice->_vtbl->_release(_iClockDevice); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getCurrentClock(struct ICEClock* *const iClockOut)	{ return _iClockDevice ? _iClockDevice->_vtbl->_getCurrentClock(_iClockDevice, iClockOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEClockDevice* _iClockDevice;
};

/*! @}
 * end of CEComICEClockDeviceRef
 */

/*! \defgroup CEComICEClockRef CEComICEClockRef
 * @{
 */

class CEComICEClockRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEClockRef() : _iClock(0) {}
	CEComICEClockRef(ICEClock* iOther) : _iClock(0)
	{
		if (iOther)
		{
			_iClock = iOther;
			_iClock->_vtbl->_addRef(_iClock);
		}
	}

	CEComICEClockRef(const CEComICEClockRef& other) : _iClock(0)
	{
		if (other._iClock)
		{
			_iClock = other._iClock;
			_iClock->_vtbl->_addRef(_iClock);
		}
	}

	~CEComICEClockRef()
	{
		if (_iClock)
		{
			ICEClock* tmp = _iClock;
			_iClock = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEClock* iClock;
				void* _ptr;
			} uIntf;
			uIntf.iClock = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEClock, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iClock)
				{
					ICEClock* tmp = _iClock;
					_iClock = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iClock = uIntf.iClock;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEClock* object() const	{ return _iClock; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEClock* iOther)
	{
		if (_iClock)
		{
			ICEClock* tmp = _iClock;
			_iClock = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iClock = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEClock* detach()
	{
		ICEClock* iIntf = _iClock;
		_iClock = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEClock* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iClock;
		if (_iClock)
		{
			_iClock->_vtbl->_addRef(_iClock);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEClock*() const	{ return _iClock; }
	FORCEINLINE_WITHOUT_DEBUG ICEClock& operator*() const	{ return *_iClock; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iClock == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEClock* iOther) const	{ return (_iClock != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEClock* iOther) const	{ return (_iClock == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEClockRef& operator=(const CEComICEClockRef& other)	{ return operator=(other._iClock); }

	CEComICEClockRef& operator=(const ICEClock* iOther)
	{
		if (_iClock != iOther)
		{
			if (_iClock)
			{
				ICEClock* tmp = _iClock;
				_iClock = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iClock = const_cast<ICEClock*>(iOther);
			if (_iClock)
			{
				_iClock->_vtbl->_addRef(_iClock);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEClock** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEClockRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iClock && "has a com object reference. clear first.");
		return & _iClock; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEClockRef& other)
	{
		bool result = false;
		if (_iClock)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iClock);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iClock ? _iClock->_vtbl->_queryInterface(_iClock, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iClock) { _iClock->_vtbl->_addRef(_iClock); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iClock) { _iClock->_vtbl->_release(_iClock); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getClockDevice(struct ICEClockDevice* *const iClockDeviceOut)	{ return _iClock ? _iClock->_vtbl->_getClockDevice(_iClock, iClockDeviceOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getCurrentTime(struct ICETime* *const iTimeOut)	{ return _iClock ? _iClock->_vtbl->_getCurrentTime(_iClock, iTimeOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult compare(struct ICETime* iTime1, struct ICETime* iTime2, INT32* resultOut)	{ return _iClock ? _iClock->_vtbl->_compare(_iClock, iTime1, iTime2, resultOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getTickcountInMilliseconds(UINT32 *const msecOut)	{ return _iClock ? _iClock->_vtbl->_getTickcountInMilliseconds(_iClock, msecOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEClock* _iClock;
};

/*! @}
 * end of CEComICEClockRef
 */

/*! \defgroup CEComICETimeRef CEComICETimeRef
 * @{
 */

class CEComICETimeRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICETimeRef() : _iTime(0) {}
	CEComICETimeRef(ICETime* iOther) : _iTime(0)
	{
		if (iOther)
		{
			_iTime = iOther;
			_iTime->_vtbl->_addRef(_iTime);
		}
	}

	CEComICETimeRef(const CEComICETimeRef& other) : _iTime(0)
	{
		if (other._iTime)
		{
			_iTime = other._iTime;
			_iTime->_vtbl->_addRef(_iTime);
		}
	}

	~CEComICETimeRef()
	{
		if (_iTime)
		{
			ICETime* tmp = _iTime;
			_iTime = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICETime* iTime;
				void* _ptr;
			} uIntf;
			uIntf.iTime = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICETime, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iTime)
				{
					ICETime* tmp = _iTime;
					_iTime = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iTime = uIntf.iTime;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICETime* object() const	{ return _iTime; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICETime* iOther)
	{
		if (_iTime)
		{
			ICETime* tmp = _iTime;
			_iTime = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iTime = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICETime* detach()
	{
		ICETime* iIntf = _iTime;
		_iTime = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICETime* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iTime;
		if (_iTime)
		{
			_iTime->_vtbl->_addRef(_iTime);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICETime*() const	{ return _iTime; }
	FORCEINLINE_WITHOUT_DEBUG ICETime& operator*() const	{ return *_iTime; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iTime == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICETime* iOther) const	{ return (_iTime != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICETime* iOther) const	{ return (_iTime == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICETimeRef& operator=(const CEComICETimeRef& other)	{ return operator=(other._iTime); }

	CEComICETimeRef& operator=(const ICETime* iOther)
	{
		if (_iTime != iOther)
		{
			if (_iTime)
			{
				ICETime* tmp = _iTime;
				_iTime = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iTime = const_cast<ICETime*>(iOther);
			if (_iTime)
			{
				_iTime->_vtbl->_addRef(_iTime);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICETime** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICETimeRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iTime && "has a com object reference. clear first.");
		return & _iTime; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICETimeRef& other)
	{
		bool result = false;
		if (_iTime)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iTime);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iTime ? _iTime->_vtbl->_queryInterface(_iTime, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iTime) { _iTime->_vtbl->_addRef(_iTime); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iTime) { _iTime->_vtbl->_release(_iTime); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getTimeValue(INT32 *const secOut, UINT32 *const usecOut)	{ return _iTime ? _iTime->_vtbl->_getTimeValue(_iTime, secOut, usecOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getClock(struct ICEClock* *const iClockOut)	{ return _iTime ? _iTime->_vtbl->_getClock(_iTime, iClockOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICETime* _iTime;
};

/*! @}
 * end of CEComICETimeRef
 */

/*! \defgroup CEComICETimeBaseDeviceRef CEComICETimeBaseDeviceRef
 * @{
 */

class CEComICETimeBaseDeviceRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICETimeBaseDeviceRef() : _iTimeBaseDevice(0) {}
	CEComICETimeBaseDeviceRef(ICETimeBaseDevice* iOther) : _iTimeBaseDevice(0)
	{
		if (iOther)
		{
			_iTimeBaseDevice = iOther;
			_iTimeBaseDevice->_vtbl->_addRef(_iTimeBaseDevice);
		}
	}

	CEComICETimeBaseDeviceRef(const CEComICETimeBaseDeviceRef& other) : _iTimeBaseDevice(0)
	{
		if (other._iTimeBaseDevice)
		{
			_iTimeBaseDevice = other._iTimeBaseDevice;
			_iTimeBaseDevice->_vtbl->_addRef(_iTimeBaseDevice);
		}
	}

	~CEComICETimeBaseDeviceRef()
	{
		if (_iTimeBaseDevice)
		{
			ICETimeBaseDevice* tmp = _iTimeBaseDevice;
			_iTimeBaseDevice = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICETimeBaseDevice* iTimeBaseDevice;
				void* _ptr;
			} uIntf;
			uIntf.iTimeBaseDevice = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICETimeBaseDevice, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iTimeBaseDevice)
				{
					ICETimeBaseDevice* tmp = _iTimeBaseDevice;
					_iTimeBaseDevice = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iTimeBaseDevice = uIntf.iTimeBaseDevice;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICETimeBaseDevice* object() const	{ return _iTimeBaseDevice; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICETimeBaseDevice* iOther)
	{
		if (_iTimeBaseDevice)
		{
			ICETimeBaseDevice* tmp = _iTimeBaseDevice;
			_iTimeBaseDevice = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iTimeBaseDevice = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICETimeBaseDevice* detach()
	{
		ICETimeBaseDevice* iIntf = _iTimeBaseDevice;
		_iTimeBaseDevice = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICETimeBaseDevice* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iTimeBaseDevice;
		if (_iTimeBaseDevice)
		{
			_iTimeBaseDevice->_vtbl->_addRef(_iTimeBaseDevice);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICETimeBaseDevice*() const	{ return _iTimeBaseDevice; }
	FORCEINLINE_WITHOUT_DEBUG ICETimeBaseDevice& operator*() const	{ return *_iTimeBaseDevice; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iTimeBaseDevice == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICETimeBaseDevice* iOther) const	{ return (_iTimeBaseDevice != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICETimeBaseDevice* iOther) const	{ return (_iTimeBaseDevice == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICETimeBaseDeviceRef& operator=(const CEComICETimeBaseDeviceRef& other)	{ return operator=(other._iTimeBaseDevice); }

	CEComICETimeBaseDeviceRef& operator=(const ICETimeBaseDevice* iOther)
	{
		if (_iTimeBaseDevice != iOther)
		{
			if (_iTimeBaseDevice)
			{
				ICETimeBaseDevice* tmp = _iTimeBaseDevice;
				_iTimeBaseDevice = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iTimeBaseDevice = const_cast<ICETimeBaseDevice*>(iOther);
			if (_iTimeBaseDevice)
			{
				_iTimeBaseDevice->_vtbl->_addRef(_iTimeBaseDevice);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICETimeBaseDevice** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICETimeBaseDeviceRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iTimeBaseDevice && "has a com object reference. clear first.");
		return & _iTimeBaseDevice; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICETimeBaseDeviceRef& other)
	{
		bool result = false;
		if (_iTimeBaseDevice)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iTimeBaseDevice);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iTimeBaseDevice ? _iTimeBaseDevice->_vtbl->_queryInterface(_iTimeBaseDevice, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iTimeBaseDevice) { _iTimeBaseDevice->_vtbl->_addRef(_iTimeBaseDevice); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iTimeBaseDevice) { _iTimeBaseDevice->_vtbl->_release(_iTimeBaseDevice); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getTimeBase(struct ICETimeBase* *const iCETimeBaseOut)	{ return _iTimeBaseDevice ? _iTimeBaseDevice->_vtbl->_getTimeBase(_iTimeBaseDevice, iCETimeBaseOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICETimeBaseDevice* _iTimeBaseDevice;
};

/*! @}
 * end of CEComICETimeBaseDeviceRef
 */

/*! \defgroup CEComICETimeBaseRef CEComICETimeBaseRef
 * @{
 */

class CEComICETimeBaseRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICETimeBaseRef() : _iTimeBase(0) {}
	CEComICETimeBaseRef(ICETimeBase* iOther) : _iTimeBase(0)
	{
		if (iOther)
		{
			_iTimeBase = iOther;
			_iTimeBase->_vtbl->_addRef(_iTimeBase);
		}
	}

	CEComICETimeBaseRef(const CEComICETimeBaseRef& other) : _iTimeBase(0)
	{
		if (other._iTimeBase)
		{
			_iTimeBase = other._iTimeBase;
			_iTimeBase->_vtbl->_addRef(_iTimeBase);
		}
	}

	~CEComICETimeBaseRef()
	{
		if (_iTimeBase)
		{
			ICETimeBase* tmp = _iTimeBase;
			_iTimeBase = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICETimeBase* iTimeBase;
				void* _ptr;
			} uIntf;
			uIntf.iTimeBase = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICETimeBase, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iTimeBase)
				{
					ICETimeBase* tmp = _iTimeBase;
					_iTimeBase = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iTimeBase = uIntf.iTimeBase;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICETimeBase* object() const	{ return _iTimeBase; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICETimeBase* iOther)
	{
		if (_iTimeBase)
		{
			ICETimeBase* tmp = _iTimeBase;
			_iTimeBase = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iTimeBase = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICETimeBase* detach()
	{
		ICETimeBase* iIntf = _iTimeBase;
		_iTimeBase = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICETimeBase* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iTimeBase;
		if (_iTimeBase)
		{
			_iTimeBase->_vtbl->_addRef(_iTimeBase);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICETimeBase*() const	{ return _iTimeBase; }
	FORCEINLINE_WITHOUT_DEBUG ICETimeBase& operator*() const	{ return *_iTimeBase; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iTimeBase == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICETimeBase* iOther) const	{ return (_iTimeBase != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICETimeBase* iOther) const	{ return (_iTimeBase == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICETimeBaseRef& operator=(const CEComICETimeBaseRef& other)	{ return operator=(other._iTimeBase); }

	CEComICETimeBaseRef& operator=(const ICETimeBase* iOther)
	{
		if (_iTimeBase != iOther)
		{
			if (_iTimeBase)
			{
				ICETimeBase* tmp = _iTimeBase;
				_iTimeBase = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iTimeBase = const_cast<ICETimeBase*>(iOther);
			if (_iTimeBase)
			{
				_iTimeBase->_vtbl->_addRef(_iTimeBase);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICETimeBase** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICETimeBaseRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iTimeBase && "has a com object reference. clear first.");
		return & _iTimeBase; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICETimeBaseRef& other)
	{
		bool result = false;
		if (_iTimeBase)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iTimeBase);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iTimeBase ? _iTimeBase->_vtbl->_queryInterface(_iTimeBase, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iTimeBase) { _iTimeBase->_vtbl->_addRef(_iTimeBase); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iTimeBase) { _iTimeBase->_vtbl->_release(_iTimeBase); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getTimeBaseDevice(struct ICETimeBaseDevice* *const iCETimeBaseDeviceOut)	{ return _iTimeBase ? _iTimeBase->_vtbl->_getTimeBaseDevice(_iTimeBase, iCETimeBaseDeviceOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getInstantiatedTime(struct ICETime* *const iTimeOut)	{ return _iTimeBase ? _iTimeBase->_vtbl->_getInstantiatedTime(_iTimeBase, iTimeOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult importTimeValue(INT32 sec, UINT32 usec, struct ICETime* *const iTimeOut)	{ return _iTimeBase ? _iTimeBase->_vtbl->_importTimeValue(_iTimeBase, sec, usec, iTimeOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult exportTimeValue(struct ICETime* iTime, INT32* secOut, UINT32* usecOut)	{ return _iTimeBase ? _iTimeBase->_vtbl->_exportTimeValue(_iTimeBase, iTime, secOut, usecOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICETimeBase* _iTimeBase;
};

/*! @}
 * end of CEComICETimeBaseRef
 */

#endif // __cplusplus

#endif
