///////////////////////////////////////////////////////////////////////////////
// Copyright 2011 Sony Corporation
///////////////////////////////////////////////////////////////////////////////

#ifndef ICENetwork_h
#define ICENetwork_h

//----------------------------------------------------------------
// Includes
//----------------------------------------------------------------

#include "CEApiUnknown.h"
#include "CEUITypes.h"
#include "CECom.h"
#include "ceulinearlists.h"

#ifdef __cplusplus
extern "C" {
#endif

//----------------------------------------------------------------
// Prototypes
//----------------------------------------------------------------


//----------------------------------------------------------------
// Enumerators
//----------------------------------------------------------------

typedef enum
{
	CENETWORK_HTTP_UNKNOWN,
	CENETWORK_HTTP_GET,
	CENETWORK_HTTP_HEAD,
	CENETWORK_HTTP_POST,
	CENETWORK_HTTP_PUT,
} CENetworkHttpMethod;

typedef enum
{
	CENETWORK_DATABASE_ALL_ON_MEMORY        = 1 << 0,
	CENETWORK_DATABASE_USE_LCACHE           = 1 << 1,
	CENETWORK_DATABASE_NOT_USE_AUTH_STORE   = 1 << 2,
	CENETWORK_DATABASE_NOT_USE_COOKIE_STORE = 1 << 3,
} CENetworkDatabaseFlag;

//----------------------------------------------------------------
// Structures
//----------------------------------------------------------------

//----------------------------------------------------------------
// Interfaces
//----------------------------------------------------------------

/*! \defgroup ICENetwork ICENetwork
 * @{
 */

/*!
 * ID of ICENetwork
 */
#define CEComIID_ICENetwork 0x138520bb

/*!
 * ICENetwork
 */
typedef struct ICENetwork
{
	const struct ICENetwork_vtbl* _vtbl;
} ICENetwork;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iNetwork	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCENetwork_queryInterface) (ICENetwork* iNetwork, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCENetwork_addRef) (ICENetwork* iNetwork);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCENetwork_release) (ICENetwork* iNetwork);

typedef CEHResult (*iCENetwork_init) (ICENetwork* iNetwork, const CERawDataBuffer* proxyHostPort, const CERawDataBuffer* caCertPath, unsigned dbFlags, const CERawDataBuffer* dbPath, unsigned maxCacheSize, unsigned maxLCacheSize);

typedef CEHResult (*iCENetwork_shutdown) (ICENetwork* iNetwork);

typedef CEHResult (*iCENetwork_setProxy) (ICENetwork* iNetwork, bool isProxyEnable, const CERawDataBuffer* proxyhostport, const CERawDataBuffer* userName, const CERawDataBuffer* password);

typedef CEHResult (*iCENetwork_setPlatformData) (ICENetwork* iNetwork, void* platformData);

typedef CEHResult (*iCENetwork_getCertificateChain) (ICENetwork* iNetwork, const CERawDataBuffer* domain, char* buf, unsigned bufSize, unsigned* resSize);

typedef CEHResult (*iCENetwork_getServerCertificateInformationCount) (ICENetwork* iNetwork, CERawDataBuffer* url, UINT32* certCountOut);

typedef CEHResult (*iCENetwork_getServerCertificateInformationPosition) (ICENetwork* iNetwork, INT32 pos, const char** subjectOut, const char** issuerOut, double* issuedDateOut, double* expireDateOut, const char** serialOut, const char** md5Out);

typedef CEHResult (*iCENetwork_setCookies) (ICENetwork* iNetwork, const CERawDataBuffer* domain, const CERawDataBuffer* path, const CERawDataBuffer* cookie);

typedef CEHResult (*iCENetwork_getCookies) (ICENetwork* iNetwork, const CERawDataBuffer* domain, const CERawDataBuffer* path, bool secure, char* buf, unsigned bufSize, unsigned* resSize);

typedef CEHResult (*iCENetwork_enableCookies) (ICENetwork* iNetwork);

typedef CEHResult (*iCENetwork_disableCookies) (ICENetwork* iNetwork);

typedef CEHResult (*iCENetwork_cookiesEnabled) (ICENetwork* iNetwork, bool* enabled);

typedef CEHResult (*iCENetwork_deleteAllCookies) (ICENetwork* iNetwork);

typedef CEHResult (*iCENetwork_exportCookie) (ICENetwork* iNetwork, const char* path);

typedef CEHResult (*iCENetwork_importCookie) (ICENetwork* iNetwork, const char* path);

typedef CEHResult (*iCENetwork_flushCookie) (ICENetwork* iNetwork);

typedef CEHResult (*iCENetwork_enableCache) (ICENetwork* iNetwork);

typedef CEHResult (*iCENetwork_disableCache) (ICENetwork* iNetwork);

typedef CEHResult (*iCENetwork_setMaxCacheSize) (ICENetwork* iNetwork, unsigned max);

typedef CEHResult (*iCENetwork_setExtensionHeaderFields) (ICENetwork* iNetwork, const CERawDataBuffer* domain, const CERawDataBuffer* path, const CERawDataBuffer* fields);

typedef CEHResult (*iCENetwork_unsetExtensionHeaderFields) (ICENetwork* iNetwork, const CERawDataBuffer* domain, const CERawDataBuffer* path);

typedef CEHResult (*iCENetwork_enableExtensionHeaderFields) (ICENetwork* iNetwork);

typedef CEHResult (*iCENetwork_disableExtensionHeaderFields) (ICENetwork* iNetwork);

typedef CEHResult (*iCENetwork_submit) (ICENetwork* iNetwork, bool toReload, bool isMainResource, unsigned long identifier, void* webCoreJob, struct ICENetworkEventListener* webCoreClient, const CERawDataBuffer* protocol, CENetworkHttpMethod method, const CERawDataBuffer* host, UINT16 port, const CERawDataBuffer* path, const CERawDataBuffer* query, const CERawDataBuffer* hash, char* customHeader, char* postData, UINT32 postSize, bool isUpload, const CERawDataBuffer* filePaths, bool* contextCreated, void* userArg);

typedef CEHResult (*iCENetwork_syncSubmit) (ICENetwork* iNetwork, bool isMainResource, unsigned long identifier, void* webCoreJob, struct ICENetworkEventListener* webCoreClient, const CERawDataBuffer* protocol, CENetworkHttpMethod method, const CERawDataBuffer* host, UINT16 port, const CERawDataBuffer* path, const CERawDataBuffer* query, const CERawDataBuffer* hash, char* customHeader, char* postData, UINT32 postSize, bool isUpload, const CERawDataBuffer* filePaths, bool* contextCreated, void* userArg);

typedef CEHResult (*iCENetwork_cancel) (ICENetwork* iNetwork, unsigned long identifier, void* userArg);

typedef CEHResult (*iCENetwork_didFinishLoading) (ICENetwork* iNetwork, unsigned long identifier, const CERawDataBuffer* metaTagInfo);

typedef CEHResult (*iCENetwork_replyBadCertMessage) (ICENetwork* iNetwork, unsigned long identifier, bool sslContinue);

typedef CEHResult (*iCENetwork_replyAuthMessage) (ICENetwork* iNetwork, unsigned long identifier, const CERawDataBuffer* site, const CERawDataBuffer* realm, const CERawDataBuffer* username, const CERawDataBuffer* password, bool authContinue, bool updateSavedFlag);

typedef CEHResult (*iCENetwork_setMaxConnections) (ICENetwork* iNetwork, unsigned short maxConns);

typedef CEHResult (*iCENetwork_setMaxConnectionsPerHost) (ICENetwork* iNetwork, unsigned short maxConns);

typedef CEHResult (*iCENetwork_setMaxRequestsPerConnection) (ICENetwork* iNetwork, unsigned short maxReqs);

typedef CEHResult (*iCENetwork_setMaxConcurrentRequests) (ICENetwork* iNetwork, unsigned int maxReqs);

typedef CEHResult (*iCENetwork_setRecvBufferSize) (ICENetwork* iNetwork, unsigned int recvBytes);

typedef CEHResult (*iCENetwork_setPoolSizeForHTTP) (ICENetwork* iNetwork, unsigned long poolsize);

typedef CEHResult (*iCENetwork_setPoolSizeForSSL) (ICENetwork* iNetwork, unsigned long poolsize);

typedef CEHResult (*iCENetwork_setTimeOut) (ICENetwork* iNetwork, unsigned int timeout_sec);

typedef CEHResult (*iCENetwork_setApplicationInterface) (ICENetwork* iNetwork, struct ICEUnknown* app);

typedef CEHResult (*iCENetwork_deleteBasicAuthenticationCache) (ICENetwork* iNetwork);

typedef CEHResult (*iCENetwork_setPermissiveDirectoryList) (ICENetwork* iNetwork, CEUStringList* directoryArray);

typedef CEHResult (*iCENetwork_deleteCertForCredential) (ICENetwork* iNetwork, void* cert);

/*!
 * V-table of ICENetwork
 */
struct ICENetwork_vtbl
{
	iCENetwork_queryInterface	_queryInterface;	//!< Query interface.
	iCENetwork_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCENetwork_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCENetwork_init	_init;
	iCENetwork_shutdown	_shutdown;
	iCENetwork_setProxy	_setProxy;
	iCENetwork_setPlatformData	_setPlatformData;
	iCENetwork_getCertificateChain	_getCertificateChain;
	iCENetwork_getServerCertificateInformationCount	_getServerCertificateInformationCount;
	iCENetwork_getServerCertificateInformationPosition	_getServerCertificateInformationPosition;
	iCENetwork_setCookies	_setCookies;
	iCENetwork_getCookies	_getCookies;
	iCENetwork_enableCookies	_enableCookies;
	iCENetwork_disableCookies	_disableCookies;
	iCENetwork_cookiesEnabled	_cookiesEnabled;
	iCENetwork_deleteAllCookies	_deleteAllCookies;
	iCENetwork_exportCookie	_exportCookie;
	iCENetwork_importCookie	_importCookie;
	iCENetwork_flushCookie	_flushCookie;
	iCENetwork_enableCache	_enableCache;
	iCENetwork_disableCache	_disableCache;
	iCENetwork_setMaxCacheSize	_setMaxCacheSize;
	iCENetwork_setExtensionHeaderFields	_setExtensionHeaderFields;
	iCENetwork_unsetExtensionHeaderFields	_unsetExtensionHeaderFields;
	iCENetwork_enableExtensionHeaderFields	_enableExtensionHeaderFields;
	iCENetwork_disableExtensionHeaderFields	_disableExtensionHeaderFields;
	iCENetwork_submit	_submit;
	iCENetwork_syncSubmit	_syncSubmit;
	iCENetwork_cancel	_cancel;
	iCENetwork_didFinishLoading	_didFinishLoading;
	iCENetwork_replyBadCertMessage	_replyBadCertMessage;
	iCENetwork_replyAuthMessage	_replyAuthMessage;
	iCENetwork_setMaxConnections	_setMaxConnections;
	iCENetwork_setMaxConnectionsPerHost	_setMaxConnectionsPerHost;
	iCENetwork_setMaxRequestsPerConnection	_setMaxRequestsPerConnection;
	iCENetwork_setMaxConcurrentRequests	_setMaxConcurrentRequests;
	iCENetwork_setRecvBufferSize	_setRecvBufferSize;
	iCENetwork_setPoolSizeForHTTP	_setPoolSizeForHTTP;
	iCENetwork_setPoolSizeForSSL	_setPoolSizeForSSL;
	iCENetwork_setTimeOut	_setTimeOut;
	iCENetwork_setApplicationInterface	_setApplicationInterface;
	iCENetwork_deleteBasicAuthenticationCache	_deleteBasicAuthenticationCache;
	iCENetwork_setPermissiveDirectoryList	_setPermissiveDirectoryList;
	iCENetwork_deleteCertForCredential	_deleteCertForCredential;
};

/*! @}
 * end of ICENetwork
 */

/*! \defgroup ICENetworkEventListener ICENetworkEventListener
 * @{
 */

/*!
 * ID of ICENetworkEventListener
 */
#define CEComIID_ICENetworkEventListener 0x1c593acb

/*!
 * ICENetworkEventListener
 */
typedef struct ICENetworkEventListener
{
	const struct ICENetworkEventListener_vtbl* _vtbl;
} ICENetworkEventListener;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iNetworkEventListener	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCENetworkEventListener_queryInterface) (ICENetworkEventListener* iNetworkEventListener, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCENetworkEventListener_addRef) (ICENetworkEventListener* iNetworkEventListener);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCENetworkEventListener_release) (ICENetworkEventListener* iNetworkEventListener);

typedef CEHResult (*iCENetworkEventListener_didReceiveResponse) (ICENetworkEventListener* iNetworkEventListener, const void* webCoreJob, const char* responseHeaderBuffer, UINT32 responseHeaderSize, const char* url, UINT32 httpCode, UINT32 expectedContentLength, bool noContext, bool syncRequest, UINT32 sslError);

typedef CEHResult (*iCENetworkEventListener_didReceiveData) (ICENetworkEventListener* iNetworkEventListener, const void* webCoreJob, UINT32 httpCode, const char* url, const void* data, UINT32 size, bool syncRequest);

typedef CEHResult (*iCENetworkEventListener_didFinishLoading) (ICENetworkEventListener* iNetworkEventListener, const void* webCoreJob, bool syncRequest, void* userArg);

typedef CEHResult (*iCENetworkEventListener_didFinishLoadingAllInOne) (ICENetworkEventListener* iNetworkEventListener, const void* webCoreJob, const char* responseHeaderBuffer, UINT32 responseHeaderSize, const char* url, UINT32 httpCode, UINT32 expectedContentLength, bool noContext, const void* data, UINT32 size, bool syncRequest, void* userArg);

typedef CEHResult (*iCENetworkEventListener_didFail) (ICENetworkEventListener* iNetworkEventListener, const void* webCoreJob, const char* url, INT32 lowLevelError, CEHResult resultCode, bool noContext, bool syncRequest, void* userArg);

typedef CEHResult (*iCENetworkEventListener_willSendReqeust) (ICENetworkEventListener* iNetworkEventListener, const void* webCoreJob, const char* url, bool syncRequest, void* userArg);

typedef CEHResult (*iCENetworkEventListener_badCertificate) (ICENetworkEventListener* iNetworkEventListener, unsigned long identifier, UINT32 errCode, const char* iIssuedTo, const char* iIssuedBy, double issuedDate, double expireDate, const char* iSerialNumber, const char* iFingerPrint, void* userArg, bool withServerCert);

typedef CEHResult (*iCENetworkEventListener_authentication) (ICENetworkEventListener* iNetworkEventListener, unsigned long identifier, const char* site, const char* realm, char* defaultUser, char* defaultPass, bool wasSaved, void* userArg);

typedef CEHResult (*iCENetworkEventListener_setCertificate) (ICENetworkEventListener* iNetworkEventListener, const void* webCoreJob, void* cert);

typedef CEHResult (*iCENetworkEventListener_uploadStartNotification) (ICENetworkEventListener* iNetworkEventListener, INT32 total);

typedef CEHResult (*iCENetworkEventListener_uploadProgressNotification) (ICENetworkEventListener* iNetworkEventListener, INT32 progress);

typedef CEHResult (*iCENetworkEventListener_uploadFinishedNotification) (ICENetworkEventListener* iNetworkEventListener);

typedef CEHResult (*iCENetworkEventListener_getWindowAlive) (ICENetworkEventListener* iNetworkEventListener, bool* isAlive);

/*!
 * V-table of ICENetworkEventListener
 */
struct ICENetworkEventListener_vtbl
{
	iCENetworkEventListener_queryInterface	_queryInterface;	//!< Query interface.
	iCENetworkEventListener_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCENetworkEventListener_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCENetworkEventListener_didReceiveResponse	_didReceiveResponse;
	iCENetworkEventListener_didReceiveData	_didReceiveData;
	iCENetworkEventListener_didFinishLoading	_didFinishLoading;
	iCENetworkEventListener_didFinishLoadingAllInOne	_didFinishLoadingAllInOne;
	iCENetworkEventListener_didFail	_didFail;
	iCENetworkEventListener_willSendReqeust	_willSendReqeust;
	iCENetworkEventListener_badCertificate	_badCertificate;
	iCENetworkEventListener_authentication	_authentication;
	iCENetworkEventListener_setCertificate	_setCertificate;
	iCENetworkEventListener_uploadStartNotification	_uploadStartNotification;
	iCENetworkEventListener_uploadProgressNotification	_uploadProgressNotification;
	iCENetworkEventListener_uploadFinishedNotification	_uploadFinishedNotification;
	iCENetworkEventListener_getWindowAlive	_getWindowAlive;
};

/*! @}
 * end of ICENetworkEventListener
 */

/*! \defgroup ICENetworkPromptHandler ICENetworkPromptHandler
 * @{
 */

/*!
 * ID of ICENetworkPromptHandler
 */
#define CEComIID_ICENetworkPromptHandler 0xb4e82906

/*!
 * ICENetworkPromptHandler
 */
typedef struct ICENetworkPromptHandler
{
	const struct ICENetworkPromptHandler_vtbl* _vtbl;
} ICENetworkPromptHandler;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iNetworkPromptHandler	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCENetworkPromptHandler_queryInterface) (ICENetworkPromptHandler* iNetworkPromptHandler, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCENetworkPromptHandler_addRef) (ICENetworkPromptHandler* iNetworkPromptHandler);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCENetworkPromptHandler_release) (ICENetworkPromptHandler* iNetworkPromptHandler);

typedef CEHResult (*iCENetworkPromptHandler_requestBadCertificate) (ICENetworkPromptHandler* iNetworkPromptHandler, UINT32 errCode, const char* iIssuedTo, const char* iIssuedBy, double issuedDate, double expireDate, const char* iSerialNumber, const char* iFingerPrint, void* userArg, bool withServerCert);

/*!
 * V-table of ICENetworkPromptHandler
 */
struct ICENetworkPromptHandler_vtbl
{
	iCENetworkPromptHandler_queryInterface	_queryInterface;	//!< Query interface.
	iCENetworkPromptHandler_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCENetworkPromptHandler_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCENetworkPromptHandler_requestBadCertificate	_requestBadCertificate;
};

/*! @}
 * end of ICENetworkPromptHandler
 */

#ifdef __cplusplus
} // end of extern "C"
#endif

//----------------------------------------------------------------
// Interface Wrappers
//----------------------------------------------------------------

#ifdef __cplusplus

/*! \defgroup CEComICENetworkRef CEComICENetworkRef
 * @{
 */

class CEComICENetworkRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICENetworkRef() : _iNetwork(0) {}
	CEComICENetworkRef(ICENetwork* iOther) : _iNetwork(0)
	{
		if (iOther)
		{
			_iNetwork = iOther;
			_iNetwork->_vtbl->_addRef(_iNetwork);
		}
	}

	CEComICENetworkRef(const CEComICENetworkRef& other) : _iNetwork(0)
	{
		if (other._iNetwork)
		{
			_iNetwork = other._iNetwork;
			_iNetwork->_vtbl->_addRef(_iNetwork);
		}
	}

	~CEComICENetworkRef()
	{
		if (_iNetwork)
		{
			ICENetwork* tmp = _iNetwork;
			_iNetwork = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICENetwork* iNetwork;
				void* _ptr;
			} uIntf;
			uIntf.iNetwork = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICENetwork, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iNetwork)
				{
					ICENetwork* tmp = _iNetwork;
					_iNetwork = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iNetwork = uIntf.iNetwork;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICENetwork* object() const	{ return _iNetwork; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICENetwork* iOther)
	{
		if (_iNetwork)
		{
			ICENetwork* tmp = _iNetwork;
			_iNetwork = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iNetwork = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICENetwork* detach()
	{
		ICENetwork* iIntf = _iNetwork;
		_iNetwork = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICENetwork* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iNetwork;
		if (_iNetwork)
		{
			_iNetwork->_vtbl->_addRef(_iNetwork);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICENetwork*() const	{ return _iNetwork; }
	FORCEINLINE_WITHOUT_DEBUG ICENetwork& operator*() const	{ return *_iNetwork; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iNetwork == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICENetwork* iOther) const	{ return (_iNetwork != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICENetwork* iOther) const	{ return (_iNetwork == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICENetworkRef& operator=(const CEComICENetworkRef& other)	{ return operator=(other._iNetwork); }

	CEComICENetworkRef& operator=(const ICENetwork* iOther)
	{
		if (_iNetwork != iOther)
		{
			if (_iNetwork)
			{
				ICENetwork* tmp = _iNetwork;
				_iNetwork = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iNetwork = const_cast<ICENetwork*>(iOther);
			if (_iNetwork)
			{
				_iNetwork->_vtbl->_addRef(_iNetwork);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICENetwork** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICENetworkRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iNetwork && "has a com object reference. clear first.");
		return & _iNetwork; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICENetworkRef& other)
	{
		bool result = false;
		if (_iNetwork)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iNetwork);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iNetwork ? _iNetwork->_vtbl->_queryInterface(_iNetwork, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iNetwork) { _iNetwork->_vtbl->_addRef(_iNetwork); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iNetwork) { _iNetwork->_vtbl->_release(_iNetwork); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult init(const CERawDataBuffer* proxyHostPort, const CERawDataBuffer* caCertPath, unsigned dbFlags, const CERawDataBuffer* dbPath, unsigned maxCacheSize, unsigned maxLCacheSize)	{ return _iNetwork ? _iNetwork->_vtbl->_init(_iNetwork, proxyHostPort, caCertPath, dbFlags, dbPath, maxCacheSize, maxLCacheSize) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult shutdown()	{ return _iNetwork ? _iNetwork->_vtbl->_shutdown(_iNetwork) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setProxy(bool isProxyEnable, const CERawDataBuffer* proxyhostport, const CERawDataBuffer* userName, const CERawDataBuffer* password)	{ return _iNetwork ? _iNetwork->_vtbl->_setProxy(_iNetwork, isProxyEnable, proxyhostport, userName, password) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setPlatformData(void* platformData)	{ return _iNetwork ? _iNetwork->_vtbl->_setPlatformData(_iNetwork, platformData) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getCertificateChain(const CERawDataBuffer* domain, char* buf, unsigned bufSize, unsigned* resSize)	{ return _iNetwork ? _iNetwork->_vtbl->_getCertificateChain(_iNetwork, domain, buf, bufSize, resSize) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getServerCertificateInformationCount(CERawDataBuffer* url, UINT32* certCountOut)	{ return _iNetwork ? _iNetwork->_vtbl->_getServerCertificateInformationCount(_iNetwork, url, certCountOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getServerCertificateInformationPosition(INT32 pos, const char** subjectOut, const char** issuerOut, double* issuedDateOut, double* expireDateOut, const char** serialOut, const char** md5Out)	{ return _iNetwork ? _iNetwork->_vtbl->_getServerCertificateInformationPosition(_iNetwork, pos, subjectOut, issuerOut, issuedDateOut, expireDateOut, serialOut, md5Out) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setCookies(const CERawDataBuffer* domain, const CERawDataBuffer* path, const CERawDataBuffer* cookie)	{ return _iNetwork ? _iNetwork->_vtbl->_setCookies(_iNetwork, domain, path, cookie) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getCookies(const CERawDataBuffer* domain, const CERawDataBuffer* path, bool secure, char* buf, unsigned bufSize, unsigned* resSize)	{ return _iNetwork ? _iNetwork->_vtbl->_getCookies(_iNetwork, domain, path, secure, buf, bufSize, resSize) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult enableCookies()	{ return _iNetwork ? _iNetwork->_vtbl->_enableCookies(_iNetwork) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult disableCookies()	{ return _iNetwork ? _iNetwork->_vtbl->_disableCookies(_iNetwork) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult cookiesEnabled(bool* enabled)	{ return _iNetwork ? _iNetwork->_vtbl->_cookiesEnabled(_iNetwork, enabled) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult deleteAllCookies()	{ return _iNetwork ? _iNetwork->_vtbl->_deleteAllCookies(_iNetwork) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult exportCookie(const char* path)	{ return _iNetwork ? _iNetwork->_vtbl->_exportCookie(_iNetwork, path) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult importCookie(const char* path)	{ return _iNetwork ? _iNetwork->_vtbl->_importCookie(_iNetwork, path) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult flushCookie()	{ return _iNetwork ? _iNetwork->_vtbl->_flushCookie(_iNetwork) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult enableCache()	{ return _iNetwork ? _iNetwork->_vtbl->_enableCache(_iNetwork) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult disableCache()	{ return _iNetwork ? _iNetwork->_vtbl->_disableCache(_iNetwork) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setMaxCacheSize(unsigned max)	{ return _iNetwork ? _iNetwork->_vtbl->_setMaxCacheSize(_iNetwork, max) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setExtensionHeaderFields(const CERawDataBuffer* domain, const CERawDataBuffer* path, const CERawDataBuffer* fields)	{ return _iNetwork ? _iNetwork->_vtbl->_setExtensionHeaderFields(_iNetwork, domain, path, fields) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult unsetExtensionHeaderFields(const CERawDataBuffer* domain, const CERawDataBuffer* path)	{ return _iNetwork ? _iNetwork->_vtbl->_unsetExtensionHeaderFields(_iNetwork, domain, path) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult enableExtensionHeaderFields()	{ return _iNetwork ? _iNetwork->_vtbl->_enableExtensionHeaderFields(_iNetwork) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult disableExtensionHeaderFields()	{ return _iNetwork ? _iNetwork->_vtbl->_disableExtensionHeaderFields(_iNetwork) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult submit(bool toReload, bool isMainResource, unsigned long identifier, void* webCoreJob, struct ICENetworkEventListener* webCoreClient, const CERawDataBuffer* protocol, CENetworkHttpMethod method, const CERawDataBuffer* host, UINT16 port, const CERawDataBuffer* path, const CERawDataBuffer* query, const CERawDataBuffer* hash, char* customHeader, char* postData, UINT32 postSize, bool isUpload, const CERawDataBuffer* filePaths, bool* contextCreated, void* userArg)	{ return _iNetwork ? _iNetwork->_vtbl->_submit(_iNetwork, toReload, isMainResource, identifier, webCoreJob, webCoreClient, protocol, method, host, port, path, query, hash, customHeader, postData, postSize, isUpload, filePaths, contextCreated, userArg) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult syncSubmit(bool isMainResource, unsigned long identifier, void* webCoreJob, struct ICENetworkEventListener* webCoreClient, const CERawDataBuffer* protocol, CENetworkHttpMethod method, const CERawDataBuffer* host, UINT16 port, const CERawDataBuffer* path, const CERawDataBuffer* query, const CERawDataBuffer* hash, char* customHeader, char* postData, UINT32 postSize, bool isUpload, const CERawDataBuffer* filePaths, bool* contextCreated, void* userArg)	{ return _iNetwork ? _iNetwork->_vtbl->_syncSubmit(_iNetwork, isMainResource, identifier, webCoreJob, webCoreClient, protocol, method, host, port, path, query, hash, customHeader, postData, postSize, isUpload, filePaths, contextCreated, userArg) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult cancel(unsigned long identifier, void* userArg)	{ return _iNetwork ? _iNetwork->_vtbl->_cancel(_iNetwork, identifier, userArg) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult didFinishLoading(unsigned long identifier, const CERawDataBuffer* metaTagInfo)	{ return _iNetwork ? _iNetwork->_vtbl->_didFinishLoading(_iNetwork, identifier, metaTagInfo) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult replyBadCertMessage(unsigned long identifier, bool sslContinue)	{ return _iNetwork ? _iNetwork->_vtbl->_replyBadCertMessage(_iNetwork, identifier, sslContinue) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult replyAuthMessage(unsigned long identifier, const CERawDataBuffer* site, const CERawDataBuffer* realm, const CERawDataBuffer* username, const CERawDataBuffer* password, bool authContinue, bool updateSavedFlag)	{ return _iNetwork ? _iNetwork->_vtbl->_replyAuthMessage(_iNetwork, identifier, site, realm, username, password, authContinue, updateSavedFlag) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setMaxConnections(unsigned short maxConns)	{ return _iNetwork ? _iNetwork->_vtbl->_setMaxConnections(_iNetwork, maxConns) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setMaxConnectionsPerHost(unsigned short maxConns)	{ return _iNetwork ? _iNetwork->_vtbl->_setMaxConnectionsPerHost(_iNetwork, maxConns) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setMaxRequestsPerConnection(unsigned short maxReqs)	{ return _iNetwork ? _iNetwork->_vtbl->_setMaxRequestsPerConnection(_iNetwork, maxReqs) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setMaxConcurrentRequests(unsigned int maxReqs)	{ return _iNetwork ? _iNetwork->_vtbl->_setMaxConcurrentRequests(_iNetwork, maxReqs) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setRecvBufferSize(unsigned int recvBytes)	{ return _iNetwork ? _iNetwork->_vtbl->_setRecvBufferSize(_iNetwork, recvBytes) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setPoolSizeForHTTP(unsigned long poolsize)	{ return _iNetwork ? _iNetwork->_vtbl->_setPoolSizeForHTTP(_iNetwork, poolsize) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setPoolSizeForSSL(unsigned long poolsize)	{ return _iNetwork ? _iNetwork->_vtbl->_setPoolSizeForSSL(_iNetwork, poolsize) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setTimeOut(unsigned int timeout_sec)	{ return _iNetwork ? _iNetwork->_vtbl->_setTimeOut(_iNetwork, timeout_sec) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setApplicationInterface(struct ICEUnknown* app)	{ return _iNetwork ? _iNetwork->_vtbl->_setApplicationInterface(_iNetwork, app) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult deleteBasicAuthenticationCache()	{ return _iNetwork ? _iNetwork->_vtbl->_deleteBasicAuthenticationCache(_iNetwork) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setPermissiveDirectoryList(CEUStringList* directoryArray)	{ return _iNetwork ? _iNetwork->_vtbl->_setPermissiveDirectoryList(_iNetwork, directoryArray) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult deleteCertForCredential(void* cert)	{ return _iNetwork ? _iNetwork->_vtbl->_deleteCertForCredential(_iNetwork, cert) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICENetwork* _iNetwork;
};

/*! @}
 * end of CEComICENetworkRef
 */

/*! \defgroup CEComICENetworkEventListenerRef CEComICENetworkEventListenerRef
 * @{
 */

class CEComICENetworkEventListenerRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICENetworkEventListenerRef() : _iNetworkEventListener(0) {}
	CEComICENetworkEventListenerRef(ICENetworkEventListener* iOther) : _iNetworkEventListener(0)
	{
		if (iOther)
		{
			_iNetworkEventListener = iOther;
			_iNetworkEventListener->_vtbl->_addRef(_iNetworkEventListener);
		}
	}

	CEComICENetworkEventListenerRef(const CEComICENetworkEventListenerRef& other) : _iNetworkEventListener(0)
	{
		if (other._iNetworkEventListener)
		{
			_iNetworkEventListener = other._iNetworkEventListener;
			_iNetworkEventListener->_vtbl->_addRef(_iNetworkEventListener);
		}
	}

	~CEComICENetworkEventListenerRef()
	{
		if (_iNetworkEventListener)
		{
			ICENetworkEventListener* tmp = _iNetworkEventListener;
			_iNetworkEventListener = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICENetworkEventListener* iNetworkEventListener;
				void* _ptr;
			} uIntf;
			uIntf.iNetworkEventListener = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICENetworkEventListener, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iNetworkEventListener)
				{
					ICENetworkEventListener* tmp = _iNetworkEventListener;
					_iNetworkEventListener = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iNetworkEventListener = uIntf.iNetworkEventListener;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICENetworkEventListener* object() const	{ return _iNetworkEventListener; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICENetworkEventListener* iOther)
	{
		if (_iNetworkEventListener)
		{
			ICENetworkEventListener* tmp = _iNetworkEventListener;
			_iNetworkEventListener = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iNetworkEventListener = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICENetworkEventListener* detach()
	{
		ICENetworkEventListener* iIntf = _iNetworkEventListener;
		_iNetworkEventListener = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICENetworkEventListener* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iNetworkEventListener;
		if (_iNetworkEventListener)
		{
			_iNetworkEventListener->_vtbl->_addRef(_iNetworkEventListener);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICENetworkEventListener*() const	{ return _iNetworkEventListener; }
	FORCEINLINE_WITHOUT_DEBUG ICENetworkEventListener& operator*() const	{ return *_iNetworkEventListener; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iNetworkEventListener == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICENetworkEventListener* iOther) const	{ return (_iNetworkEventListener != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICENetworkEventListener* iOther) const	{ return (_iNetworkEventListener == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICENetworkEventListenerRef& operator=(const CEComICENetworkEventListenerRef& other)	{ return operator=(other._iNetworkEventListener); }

	CEComICENetworkEventListenerRef& operator=(const ICENetworkEventListener* iOther)
	{
		if (_iNetworkEventListener != iOther)
		{
			if (_iNetworkEventListener)
			{
				ICENetworkEventListener* tmp = _iNetworkEventListener;
				_iNetworkEventListener = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iNetworkEventListener = const_cast<ICENetworkEventListener*>(iOther);
			if (_iNetworkEventListener)
			{
				_iNetworkEventListener->_vtbl->_addRef(_iNetworkEventListener);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICENetworkEventListener** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICENetworkEventListenerRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iNetworkEventListener && "has a com object reference. clear first.");
		return & _iNetworkEventListener; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICENetworkEventListenerRef& other)
	{
		bool result = false;
		if (_iNetworkEventListener)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iNetworkEventListener);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iNetworkEventListener ? _iNetworkEventListener->_vtbl->_queryInterface(_iNetworkEventListener, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iNetworkEventListener) { _iNetworkEventListener->_vtbl->_addRef(_iNetworkEventListener); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iNetworkEventListener) { _iNetworkEventListener->_vtbl->_release(_iNetworkEventListener); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult didReceiveResponse(const void* webCoreJob, const char* responseHeaderBuffer, UINT32 responseHeaderSize, const char* url, UINT32 httpCode, UINT32 expectedContentLength, bool noContext, bool syncRequest, UINT32 sslError)	{ return _iNetworkEventListener ? _iNetworkEventListener->_vtbl->_didReceiveResponse(_iNetworkEventListener, webCoreJob, responseHeaderBuffer, responseHeaderSize, url, httpCode, expectedContentLength, noContext, syncRequest, sslError) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult didReceiveData(const void* webCoreJob, UINT32 httpCode, const char* url, const void* data, UINT32 size, bool syncRequest)	{ return _iNetworkEventListener ? _iNetworkEventListener->_vtbl->_didReceiveData(_iNetworkEventListener, webCoreJob, httpCode, url, data, size, syncRequest) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult didFinishLoading(const void* webCoreJob, bool syncRequest, void* userArg)	{ return _iNetworkEventListener ? _iNetworkEventListener->_vtbl->_didFinishLoading(_iNetworkEventListener, webCoreJob, syncRequest, userArg) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult didFinishLoadingAllInOne(const void* webCoreJob, const char* responseHeaderBuffer, UINT32 responseHeaderSize, const char* url, UINT32 httpCode, UINT32 expectedContentLength, bool noContext, const void* data, UINT32 size, bool syncRequest, void* userArg)	{ return _iNetworkEventListener ? _iNetworkEventListener->_vtbl->_didFinishLoadingAllInOne(_iNetworkEventListener, webCoreJob, responseHeaderBuffer, responseHeaderSize, url, httpCode, expectedContentLength, noContext, data, size, syncRequest, userArg) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult didFail(const void* webCoreJob, const char* url, INT32 lowLevelError, CEHResult resultCode, bool noContext, bool syncRequest, void* userArg)	{ return _iNetworkEventListener ? _iNetworkEventListener->_vtbl->_didFail(_iNetworkEventListener, webCoreJob, url, lowLevelError, resultCode, noContext, syncRequest, userArg) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult willSendReqeust(const void* webCoreJob, const char* url, bool syncRequest, void* userArg)	{ return _iNetworkEventListener ? _iNetworkEventListener->_vtbl->_willSendReqeust(_iNetworkEventListener, webCoreJob, url, syncRequest, userArg) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult badCertificate(unsigned long identifier, UINT32 errCode, const char* iIssuedTo, const char* iIssuedBy, double issuedDate, double expireDate, const char* iSerialNumber, const char* iFingerPrint, void* userArg, bool withServerCert)	{ return _iNetworkEventListener ? _iNetworkEventListener->_vtbl->_badCertificate(_iNetworkEventListener, identifier, errCode, iIssuedTo, iIssuedBy, issuedDate, expireDate, iSerialNumber, iFingerPrint, userArg, withServerCert) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult authentication(unsigned long identifier, const char* site, const char* realm, char* defaultUser, char* defaultPass, bool wasSaved, void* userArg)	{ return _iNetworkEventListener ? _iNetworkEventListener->_vtbl->_authentication(_iNetworkEventListener, identifier, site, realm, defaultUser, defaultPass, wasSaved, userArg) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setCertificate(const void* webCoreJob, void* cert)	{ return _iNetworkEventListener ? _iNetworkEventListener->_vtbl->_setCertificate(_iNetworkEventListener, webCoreJob, cert) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult uploadStartNotification(INT32 total)	{ return _iNetworkEventListener ? _iNetworkEventListener->_vtbl->_uploadStartNotification(_iNetworkEventListener, total) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult uploadProgressNotification(INT32 progress)	{ return _iNetworkEventListener ? _iNetworkEventListener->_vtbl->_uploadProgressNotification(_iNetworkEventListener, progress) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult uploadFinishedNotification()	{ return _iNetworkEventListener ? _iNetworkEventListener->_vtbl->_uploadFinishedNotification(_iNetworkEventListener) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getWindowAlive(bool* isAlive)	{ return _iNetworkEventListener ? _iNetworkEventListener->_vtbl->_getWindowAlive(_iNetworkEventListener, isAlive) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICENetworkEventListener* _iNetworkEventListener;
};

/*! @}
 * end of CEComICENetworkEventListenerRef
 */

/*! \defgroup CEComICENetworkPromptHandlerRef CEComICENetworkPromptHandlerRef
 * @{
 */

class CEComICENetworkPromptHandlerRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICENetworkPromptHandlerRef() : _iNetworkPromptHandler(0) {}
	CEComICENetworkPromptHandlerRef(ICENetworkPromptHandler* iOther) : _iNetworkPromptHandler(0)
	{
		if (iOther)
		{
			_iNetworkPromptHandler = iOther;
			_iNetworkPromptHandler->_vtbl->_addRef(_iNetworkPromptHandler);
		}
	}

	CEComICENetworkPromptHandlerRef(const CEComICENetworkPromptHandlerRef& other) : _iNetworkPromptHandler(0)
	{
		if (other._iNetworkPromptHandler)
		{
			_iNetworkPromptHandler = other._iNetworkPromptHandler;
			_iNetworkPromptHandler->_vtbl->_addRef(_iNetworkPromptHandler);
		}
	}

	~CEComICENetworkPromptHandlerRef()
	{
		if (_iNetworkPromptHandler)
		{
			ICENetworkPromptHandler* tmp = _iNetworkPromptHandler;
			_iNetworkPromptHandler = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICENetworkPromptHandler* iNetworkPromptHandler;
				void* _ptr;
			} uIntf;
			uIntf.iNetworkPromptHandler = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICENetworkPromptHandler, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iNetworkPromptHandler)
				{
					ICENetworkPromptHandler* tmp = _iNetworkPromptHandler;
					_iNetworkPromptHandler = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iNetworkPromptHandler = uIntf.iNetworkPromptHandler;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICENetworkPromptHandler* object() const	{ return _iNetworkPromptHandler; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICENetworkPromptHandler* iOther)
	{
		if (_iNetworkPromptHandler)
		{
			ICENetworkPromptHandler* tmp = _iNetworkPromptHandler;
			_iNetworkPromptHandler = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iNetworkPromptHandler = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICENetworkPromptHandler* detach()
	{
		ICENetworkPromptHandler* iIntf = _iNetworkPromptHandler;
		_iNetworkPromptHandler = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICENetworkPromptHandler* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iNetworkPromptHandler;
		if (_iNetworkPromptHandler)
		{
			_iNetworkPromptHandler->_vtbl->_addRef(_iNetworkPromptHandler);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICENetworkPromptHandler*() const	{ return _iNetworkPromptHandler; }
	FORCEINLINE_WITHOUT_DEBUG ICENetworkPromptHandler& operator*() const	{ return *_iNetworkPromptHandler; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iNetworkPromptHandler == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICENetworkPromptHandler* iOther) const	{ return (_iNetworkPromptHandler != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICENetworkPromptHandler* iOther) const	{ return (_iNetworkPromptHandler == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICENetworkPromptHandlerRef& operator=(const CEComICENetworkPromptHandlerRef& other)	{ return operator=(other._iNetworkPromptHandler); }

	CEComICENetworkPromptHandlerRef& operator=(const ICENetworkPromptHandler* iOther)
	{
		if (_iNetworkPromptHandler != iOther)
		{
			if (_iNetworkPromptHandler)
			{
				ICENetworkPromptHandler* tmp = _iNetworkPromptHandler;
				_iNetworkPromptHandler = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iNetworkPromptHandler = const_cast<ICENetworkPromptHandler*>(iOther);
			if (_iNetworkPromptHandler)
			{
				_iNetworkPromptHandler->_vtbl->_addRef(_iNetworkPromptHandler);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICENetworkPromptHandler** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICENetworkPromptHandlerRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iNetworkPromptHandler && "has a com object reference. clear first.");
		return & _iNetworkPromptHandler; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICENetworkPromptHandlerRef& other)
	{
		bool result = false;
		if (_iNetworkPromptHandler)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iNetworkPromptHandler);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iNetworkPromptHandler ? _iNetworkPromptHandler->_vtbl->_queryInterface(_iNetworkPromptHandler, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iNetworkPromptHandler) { _iNetworkPromptHandler->_vtbl->_addRef(_iNetworkPromptHandler); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iNetworkPromptHandler) { _iNetworkPromptHandler->_vtbl->_release(_iNetworkPromptHandler); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult requestBadCertificate(UINT32 errCode, const char* iIssuedTo, const char* iIssuedBy, double issuedDate, double expireDate, const char* iSerialNumber, const char* iFingerPrint, void* userArg, bool withServerCert)	{ return _iNetworkPromptHandler ? _iNetworkPromptHandler->_vtbl->_requestBadCertificate(_iNetworkPromptHandler, errCode, iIssuedTo, iIssuedBy, issuedDate, expireDate, iSerialNumber, iFingerPrint, userArg, withServerCert) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICENetworkPromptHandler* _iNetworkPromptHandler;
};

/*! @}
 * end of CEComICENetworkPromptHandlerRef
 */

#endif // __cplusplus

#endif
