///////////////////////////////////////////////////////////////////////////////
// Copyright 2004,2007,2008,2009 Sony Corporation
///////////////////////////////////////////////////////////////////////////////

#ifndef __CESysThread_h__
#define __CESysThread_h__

#include "CESysDefs.h"
#include "CESysHandle.h"

#ifdef __cplusplus
extern "C"
{
#endif

	//! thread type
	enum eCESysThreadType
	{
		eCESysThreadType_UI,		 //!< window system UI events dispatching
		eCESysThreadType_HTML,		 //!< HTML interpretation (parse, JS execute, layout)
		eCESysThreadType_URI_http,       //!< network access from http/https
		eCESysThreadType_URI_http_recv,       //!< network response from http/https
		eCESysThreadType_WTF,
		eCESysThreadType_Mithril_Server,		 //!< Mithril Server thread.
		eCESysThreadType_Mithril_Client,		 //!< Mithril Client thread.

		// temporary thread types:
#if CE_OS(LINUX) || CE_OS(ANDROID)
		eCESysThreadType_EpollServer,   //!< EpollServer (epoll supported platform only)
#endif

#if CE_OS(WIN32) || CE_OS(HYDRA)
		eCESysThreadType_RawPipe_Server,   //!< RawPipe server-side worker
		eCESysThreadType_RawPipe_Client,   //!< RawPipe client-side worker
#endif

		eCESysThreadType_enums
	};

	//! thread handle
	//! \note The thread handle is used to call thread manipulation APIs.
	//!       It is unique within the created process.
	typedef CESysHandle CESysThreadHandle;

	//! thread id
	//! \note The thread id is unique throughout the system.
#if CE_HAVE(NATIVE_THREAD_ID)
	typedef UINT32 CESysThreadId;
#else
	typedef UINT64 CESysThreadId;
#endif

	//! thread procdure type
	typedef CEHResult (*CESysThreadProc)(void* initarg);

	//! get the reserved stack size
	//! \param[in] type      type of the thread
	CE_DLL_EXPORT size_t CESysThreadClassGetStackSize(eCESysThreadType type);

	//! get the priority for the thread class
	//! \param[in] type      type of the thread
	CE_DLL_EXPORT INT32 CESysThreadClassGetPriority(eCESysThreadType type);

	//! Create a new thread
	//! Return 0 if failed.
	//! Return generated thread's CESysThreadHandle if succeeded.
	//! Generated CESysThreadHandle should be released by CESysThreadCloseHandle or CESysWaitForThreadExit.
	//! \param[in] type  type of the thread
	//! \param[in] proc  the procedure to be called on the created thread
	//! \param[in] arg   the arbitrary data to be passed to the thread proc
	//! \param[out] IdOut pointer to the variable to receive generated thread's CESysThreadId. Or NULL.
	CE_DLL_EXPORT CESysThreadHandle CESysCreateThread(eCESysThreadType type, CESysThreadProc proc, void* arg, CESysThreadId* IdOut);

	//! Close CESysThreadHandle.
	CE_DLL_EXPORT void CESysThreadCloseHandle(CESysThreadHandle hThread);

	//! wait for the thread exit
	//! \param[in] hThread  specifies the thread handle to wait for
	//! \param[out] exitValueOut  pointer to the variable to receive the exit value from the thread
	//! \note you can specify NULL to exitValueOut if you don't want to know it.
	CE_DLL_EXPORT CEHResult CESysWaitForThreadExit(CESysThreadHandle hThread, CEHResult* exitValueOut);	

	//! get the current thread id
	CE_DLL_EXPORT CESysThreadId CESysGetCurrentThreadId();

	//! sleep the current thread
	//! \param[in] microsec  specifies the duration of sleep
	CE_DLL_EXPORT void CESysSleep(UINT32 microsec);


	//! get the stack address of the current thread
	//! \param[out] stackBase   Pointer to the variable to receive the start address of the current thread stack segment
	//! \param[out] stackLimit  Pointer to the variable to receive the end address of the current thread stack segment
	//! \note In the platform that the stack segment is used from higher address to lower, such as win32 or x86/linux,
	//!       'stackBase' is larger than 'stackLimit'.
	CE_DLL_EXPORT CEHResult CESysGetCurrentThreadStackInfo(void** stackBase, void** stackLimit);


#ifdef __cplusplus
}
#endif

#endif // __CESysThread_h__
