///////////////////////////////////////////////////////////////////////////////
// Copyright 2004,2005,2006,2007,2008,2009 Sony Corporation
///////////////////////////////////////////////////////////////////////////////

#ifndef CECom_h
#define CECom_h

///////////////////////////////////////////////////////////////////////////////
// Includes
///////////////////////////////////////////////////////////////////////////////

#include "CESysDefs.h"
#include "CENew.h"

#include "CESysMemory.h"
#include "CEComDebugServices.h"


// <--- CEComStdClassID migration ----
#define CEComStdClassID_CEUString			(CEComGetStdCLSIDForIID(CEComIID_ICEUString))
#define CEComStdClassID_CEUStringReader		(CEComGetStdCLSIDForIID(CEComIID_ICEUStringReader))
#define CEComStdClassID_CEUStringWriter		(CEComGetStdCLSIDForIID(CEComIID_ICEUStringWriter))
#define CEComStdClassID_CEI18nClassFactory	(CEComGetStdCLSIDForIID(CEComIID_ICEI18nClassFactory))

#define CEComStdClassID_CEFSPathnameFactory 0x16d832ba  // TODO: this magic number should be moved to CEComGetStdCLSIDForIID() or around
// --- CEComStdClassID migration ---->


struct ICEApartment;
struct ICEApartmentTransferHandler;

#ifdef __cplusplus
extern "C"
{
#endif

	typedef struct ICEUString ICEUString;
	typedef struct ICEUStringWriter ICEUStringWriter;

///////////////////////////////////////////////////////////////////////////////
// Interfaces
///////////////////////////////////////////////////////////////////////////////

	////////////////////////////////////
	//  Allocator Callback Functions
	////////////////////////////////////
    typedef enum CEComAllocatorErrorCode {
        ALLOCATOR_NO_ERROR = 0,
        ALLOCATOR_WAS_MEMORY_EMPTY_ERROR,
		ALLOCATOR_NOT_INITIALIZED
    } CEComAllocatorErrorCode;

	typedef struct CEComAllocatorRec CEComAllocatorRec;

	typedef void* (*CEComAllocatorMallocProc)         (CEComAllocatorRec* pAllocator, size_t sz );
	typedef void* (*CEComAllocatorReallocProc)        (CEComAllocatorRec* pAllocator, void* ptr, size_t newSize );
	typedef void* (*CEComAllocatorCallocProc)         (CEComAllocatorRec* pAllocator, size_t numElement, size_t szElement );
	typedef void  (*CEComAllocatorFreeProc)           (CEComAllocatorRec* pAllocator, void* ptr );
	typedef USIZE_T (*CEComAllocatorGetAllocedSizeProc)  (CEComAllocatorRec* pAllocator);
	typedef bool  (*CEComAllocatorCheckHeapProc)      (CEComAllocatorRec* pAllocator);
	typedef void  (*CEComAllocatorSetMaxHeapSizeProc) (CEComAllocatorRec* pAllocator, USIZE_T ptr);
	typedef CEComAllocatorErrorCode  (*CEComAllocatorGetErrorCodeProc)   (CEComAllocatorRec* pAllocator);
	typedef void  (*CEComAllocatorResetErrorCodeProc)   (CEComAllocatorRec* pAllocator);
	typedef USIZE_T (*CEComAllocatorGetHeapSizeProc)  (CEComAllocatorRec* pAllocator);

	//! This structure is used to allocate memory.
	//! application can use \a calldata member to store an application specific data pointer.
	struct CEComAllocatorRec
	{
		CEComAllocatorMallocProc			malloc;
		CEComAllocatorReallocProc			realloc;
		CEComAllocatorCallocProc			calloc;
		CEComAllocatorFreeProc				free;
		CEComAllocatorGetAllocedSizeProc	getAllocedSize;
		CEComAllocatorCheckHeapProc			checkHeap;
		CEComAllocatorSetMaxHeapSizeProc	setMaxHeapSize;
		CEComAllocatorGetErrorCodeProc	    getErrorCode;
		CEComAllocatorResetErrorCodeProc	resetErrorCode;
		CEComAllocatorGetHeapSizeProc       getHeapSize;
		void*         calldata;
	};

	////////////////////////////////////
	//  memory allocator options
	////////////////////////////////////
	// allocator returns null when status is low memory.
	#define CECOM_MALLOC_USE_SOFT_LIMIT (1)
	// trigger low memory
	#define CECOM_MALLOC_TRIGGER_LOW_MEMORY (2)
	// used by plugin
	#define CECOM_MALLOC_PLUGIN_USE (4)
	// handle is purgeable
	#define CECOM_MALLOC_PURGEABLE (8)
	// allocate long life gc collectable payload
	#define CECOM_GC_MALLOC_LONGLIFE (16)
	// handle is page
	#define CECOM_MALLOC_PAGE (32)
	// exclude the allocated payload from the leak report
	#define CECOM_MALLOC_EXCLUDE_FROM_LEAK_REPORT (64)

	//////////////////////////////////////////////////////////
	//  DLL_EXPORTed function prototypes for COM class DLL
	//////////////////////////////////////////////////////////

	// Each CECOM DLL should expose this type function as "CEDllGetClassObject"
	// CECOM gets the ICEClassFactory interface from this function.
	typedef CEHResult (*CEDllGetClassObjectProc)(UINT32 clsid, UINT32 iid, CEComAllocatorRec* pAllocatorRec, void** ppv);

	// Each CECOM DLL can expose this type function as "CEDllCanUnloadNow"
	// CECOM knows the timing to release DLLs from this function.
	typedef bool (*CEDllCanUnloadNowProc)();

	// Each CECOM DLL should expose this type function as "CEDllMain"
	// CECOM call this, just after loding DLL and just before unloading DLL.
	// In case of static link, all these entries are called in CEComInitialize() and CEComUninitialize().
	typedef bool (*CEDllMainProc)(CEDllHandle hDLL, UINT32 reason, void* param);


	////////////////////////////////////
	//  COM Class Loader Functions
	////////////////////////////////////

	typedef struct CEComEarlyBindServerRegistryEntry
	{
		UINT32			clsid;
		CEDllGetClassObjectProc	_getClassObject;
		CEDllCanUnloadNowProc	_canUnloadNow;
		CEDllMainProc           _dllMain;
	} CEComEarlyBindServerRegistryEntry;

	typedef struct CEComLateBindServerRegistryEntry
	{
		UINT32      clsid;
		const char* filename;
	} CEComLateBindServerRegistryEntry;


	////////////////////////////////////
	//  Initialize/Uninitialize
	////////////////////////////////////

	//! cecom heap segment's struct
	typedef struct CEComHeapSegment
	{
		void *ptr;		//!< pointer to heap segment, null means CECom allocates heap segment of size by allocator->malloc
		USIZE_T size;	//!< size of heap segment
	} CEComHeapSegment;

	// heap arena configuration which enables general allocator
	#define CECOM_HEAP_ARENA_CONFIG_USE_GENERAL_ALLOCATOR (1)
	// heap arena configuration which enables gc allocator
	#define CECOM_HEAP_ARENA_CONFIG_USE_GC_ALLOCATOR (2)
	// heap arena configuration which enables handle allocator
	#define CECOM_HEAP_ARENA_CONFIG_USE_HANDLE_ALLOCATOR (4)
	// heap arena configuration which declares dynamic heap arena creation which uses handle allocator
	// if you create heap arena which uses handle allocator dynamically, set this flag to one of CEComInitParams's arrayOfHeapArena
	#define CECOM_HEAP_ARENA_CONFIG_DYNAMIC_CREATE_HANDLE_ALLOCATOR (8)
	// heap arena configuration which enables leak collect allocator
	#define CECOM_HEAP_ARENA_CONFIG_USE_LEAK_COLLECT_ALLOCATOR (16)

	// heap arena id which is mandatory
	#define CECOM_HEAP_ARENA_ID_DEFAULT (0)

	//! cecom heap arena's struct
	typedef struct CEComHeapArena
	{
		UINT32 arenaId;							//!< heap arena id
		UINT32 config;							//!< heap arena configuration
		CEComHeapSegment *arrayOfHeapSegment;	//!< array of heap segment
		UINT32 numHeapSegment;					//!< number of heap segment
		const char *name;						//!< name of heap arena
		USIZE_T payloadAlignment;				//!< alignment of payload
	} CEComHeapArena;

	//! cecom initialization parameters' struct
	typedef struct CEComInitParams
	{
		CEComAllocatorRec*  allocator;
		CEComDebugServiceRec* debugService;
		bool enableApartment;
#if defined(STATIC_LINK)
		CEComEarlyBindServerRegistryEntry* registryEntry;
#else // defined(STATIC_LINK)
		CEComLateBindServerRegistryEntry* registryEntry;
#endif // defined(STATIC_LINK)
		UINT32 numOfRegistryEntry;
		CEComHeapArena *arrayOfHeapArena;	//!< array of heap arena
		UINT32 numHeapArena;				//!< number of heap arena
		UINT32 apartmentIid;
	} CEComInitParams;

	//! Initialize CECom environment.
	DLL_EXPORT CEHResult CEComInitialize(CEComInitParams* pParams);

	//! Uninitialize CECom environment.
	DLL_EXPORT CEHResult CEComUninitialize();


	////////////////////////////////////
	//  Memory anagement functions
	////////////////////////////////////
	/* low memory observer function */
	typedef void (*CEComLowMemoryObserverProc)(void *usrdata);

	//! memory handle
	typedef void* CEMemHandle;

	//! memory handle id
	typedef enum
	{
		CEHeapId_Default = 0,			//!< default
		CEHeapId_HttpCache,				//!< http cache
		CEHeapId_DXSurface,				//!< surface pool
		CEHeapId_FreeTypeStreamCache,	//!< free type stream cache
		CEHeapId_ImageCache				//!< image cache
	} CEHeapId;

	DLL_EXPORT void* CEComMalloc(size_t size);
	DLL_EXPORT void* CEComRealloc(void* ptr, size_t newSize);
	DLL_EXPORT void* CEComCalloc(size_t numElement, size_t szElement);
	DLL_EXPORT void  CEComFree(void* ptr);


	//! check wheter the entire heap is good or corrupted
	DLL_EXPORT bool  CEComCheckHeap(UINT32 arenaId);

	//! get the entire alloced size 
	DLL_EXPORT USIZE_T CEComGetAllocedSize(UINT32 arenaId);

	//! get the entire alloced handle size 
	DLL_EXPORT void CEComGetAllocedHandleSize(USIZE_T *purgeable, USIZE_T *nonPurgeable, UINT32 arenaId);

	//! returns pointer to the CEComAllocatorRec.
	DLL_EXPORT CEComAllocatorRec* CEComGetAllocatorRec();

	//! set the upper limit of the entire heap size 
	//! (this is a temporary API. this will be move to some COM I/F later)
	DLL_EXPORT void CEComSetMaxHeapSize(USIZE_T size, UINT32 arenaId);

	//! set the upper limit of the entire heap size 
	//! (this is a temporary API. this will be move to some COM I/F later)
	DLL_EXPORT USIZE_T CEComGetMaxHeapSize(UINT32 arenaId);

	//! set high watermark percantage of low memory
	//! (this is a temporary API. this will be move to some COM I/F later)
	DLL_EXPORT void CEComSetHighWaterMarkPercent(UINT32 percent, UINT32 arenaId);

	//! set low watermark percantage of low memory
	//! (this is a temporary API. this will be move to some COM I/F later)
	DLL_EXPORT void CEComSetLowWaterMarkPercent(UINT32 percent, UINT32 arenaId);

	//! set purge limit percantage of low memory
	//! (this is a temporary API. this will be move to some COM I/F later)
	DLL_EXPORT void CEComSetPurgeLimitPercent(UINT32 percent, UINT32 arenaId);

	//! get high watermark percantage of low memory
	//! (this is a temporary API. this will be move to some COM I/F later)
	DLL_EXPORT UINT32 CEComGetHighWaterMarkPercent(UINT32 arenaId);

	//! get low watermark percantage of low memory
	//! (this is a temporary API. this will be move to some COM I/F later)
	DLL_EXPORT UINT32 CEComGetLowWaterMarkPercent(UINT32 arenaId);

	//! set virtual memory size
	//! (this is a temporary API. this will be move to some COM I/F later)
	DLL_EXPORT CEHResult CEComSetVirtualMemorySize(USIZE_T sizePhysical, USIZE_T sizeVirtual, UINT32 arenaId);

	//! suspend virtual memory lock
	//! (this is a temporary API. this will be move to some COM I/F later)
	DLL_EXPORT CEHResult CEComSuspendVirtualMemoryLock(UINT32 arenaId);

	//! resume virtual memory lock
	//! (this is a temporary API. this will be move to some COM I/F later)
	DLL_EXPORT CEHResult CEComResumeVirtualMemoryLock(UINT32 arenaId);

	//! check whether memory is low
	//! (this is a temporary API. this will be move to some COM I/F later)
	DLL_EXPORT bool CEComIsLowMemory(UINT32 arenaId);

	//! force heap status to be low memory
	//! (this is a temporary API. this will be move to some COM I/F later)
	DLL_EXPORT void CEComForceLowMemory(UINT32 arenaId);

	//! moved from each application class derived from CEApplication 
	DLL_EXPORT bool CEComIsMemoryRecoverd(UINT32 arenaId);

	//! regist low memory observer function
	DLL_EXPORT void CEComRegistLowMemoryObserver(CEComLowMemoryObserverProc proc, void *usrdata, UINT32 arenaId);

	//! unregist low memory observer function
	DLL_EXPORT void CEComUnregistLowMemoryObserver(UINT32 arenaId);

	//! get allocator error code
	//! (this is a temporary API. this will be move to some COM I/F later)
	DLL_EXPORT CEComAllocatorErrorCode CEComGetAllocatorErrorCode(UINT32 arenaId);

	//! reset allocator error code
	//! (this is a temporary API. this will be move to some COM I/F later)
	DLL_EXPORT void CEComResetAllocatorErrorCode(UINT32 arenaId);


	////////////////////////////////////
	//  Get and Release COM Classes
	////////////////////////////////////

	//! Load the required CECOM DLL and get the COM class object
	DLL_EXPORT CEHResult CEComGetClassObject(UINT32 clsid, UINT32 iid, void** ppv);

	//! Unload all unused CECOM DLLs
	DLL_EXPORT void CEComFreeUnusedLibraries();

	//! get the default clsid for the specified iid
	DLL_EXPORT UINT32 CEComGetStdCLSIDForIID(UINT32 iid);

	////////////////////////////////////////////////////
	//  Global Locking (enter/leave neutral apartment)
	////////////////////////////////////////////////////

	//! Acquire the process global lock

#if defined(_DEBUG)||!defined(NDEBUG)
	DLL_EXPORT void CEComGlobalLockEx2(const char* file, int line);
	DLL_EXPORT bool CEComGlobalTryLockEx2(const char* file, int line);
	DLL_EXPORT UINT32 CEComGlobalLockOwnerThreadId();
#define CEComGlobalLock() CEComGlobalLockEx2(__FILE__, __LINE__)
#define CEComGlobalTryLock() CEComGlobalTryLockEx2(__FILE__, __LINE__)
#else //#if defined(_DEBUG)||!defined(NDEBUG)
#if defined(PLATFORM_XAI) || defined(PLATFORM_HYDRA)
	DLL_EXPORT void CEComGlobalLockEx2(const char* file, int line);
	DLL_EXPORT bool CEComGlobalTryLockEx2(const char* file, int line);
#endif
#define CEComGlobalLock() CEComGlobalLockEx()
#define CEComGlobalTryLock() CEComGlobalTryLockEx()
#endif //#if defined(_DEBUG)||!defined(NDEBUG)

	DLL_EXPORT void CEComGlobalLockEx();

	DLL_EXPORT bool CEComGlobalTryLockEx();

	//! Release the process global lock
	DLL_EXPORT void CEComGlobalUnlock();

	///////////////////////////////////
	//  Set/Get caller function info
	///////////////////////////////////

	typedef struct
	{
		const char* _pFuncSig;
		const char* _pFilename;
	} CEComDebugInfo;

	//! Set the current function/line info to associte it with successive API calls
	DLL_EXPORT void CEComSetDebugInfo(const char* pFuncSig, const char* pFilename);

	//! Clear the debug info set by CEComSetDebugInfo
	DLL_EXPORT void CEComClearDebugInfo();

	//! Get the currently-set debug info
	DLL_EXPORT const CEComDebugInfo* CEComGetDebugInfo();


	////////////////////////////////////////////////////
	//  Thread Context functions
	////////////////////////////////////////////////////

	//! Get the caller's thread context interface
	DLL_EXPORT CEHResult CEComGetThreadContext(UINT32 iid, void* *const iOut);

	//! Push the context object onto the context stack associated with the current thread
	DLL_EXPORT CEHResult CEComPushThreadContext(UINT32 iid, void* iUnk);

	//! Pop the context stack associated with the current thread
	DLL_EXPORT void CEComPopThreadContext(UINT32 iid);

	//! Push multiple context objects onto the context stack associated with the current thread
	DLL_EXPORT CEHResult CEComPushThreadContexts(UINT32 numOfIntfs, const UINT32* iids, void** iUnks);

	//! Pop the context stack associated with the current thread
	DLL_EXPORT void CEComPopThreadContexts(UINT32 numOfIntfs, const UINT32* iids);

	//! Push class objects onto the context stack associated with the current thread
	DLL_EXPORT CEHResult CEComPushClassObjectsToThreadContext(UINT32 numOfIntfs, const UINT32* clsids, const UINT32* iids);

	//! attach the caller thread to CECOM's thread context manager
	DLL_EXPORT CEHResult CEComAttachThread(UINT32 iid);

	//! detach the caller thread from CECOM's thread context manager
	DLL_EXPORT CEHResult CEComDetachThread();

	//! detach all the worker threads from CECOM's thread context manager.
	//!
	//! This function should be called very carefully.
	//! When this function is called, the thread context manager frees zombie entries.
	//! Applications must take care that no CE functions (including CEComDetachThread())
	//! are called on worker threads after this function is called.
	//! 
	//! [Bug 8374] 2007/1/24  miyasita
	//! due to silly application implementation on BDRE4G, we cannot expect
	//! all worker threads are properly terminated with calling CEComDetachThread() at
	//! termination. to prevend resource leak, our wrapper layer calls
	//! CEComDetachAllThreads() just before calling CEComUninitialize(),
	DLL_EXPORT CEHResult CEComDetachAllThreads();

	typedef enum
	{
		eCEComSysEvent_NoEvent = 0,

		// the thread termination is requested
		eCEComSysEvent_ThreadTermination = 1,

		// currently used heap size is close to the upper limit
		eCEComSysEvent_LowMemory,

		// the thread must stop the activility as soon as possible
		eCEComSysEvent_Suspend,

		// the threa can restart its activity
		eCEComSysEvent_Resume,

		eCEComSysEvent_Last = eCEComSysEvent_Resume,
		eCEComSysEvent_NumOfEvent = eCEComSysEvent_Last + 1,
	} eCEComSysEvent;

	// receive and consume event
	typedef CEHResult (*CEComThreadEventHandlerProc)(eCEComSysEvent event, void* userData);
	DLL_EXPORT CEHResult CEComSetThreadEventHandler(CEComThreadEventHandlerProc proc, void* userData);
	DLL_EXPORT void CEComNotifyThreadEventConsumed(eCEComSysEvent event);

	// post event and receive consumed
	typedef void (*CEComThreadEventConsumedNotifyProc)();
	DLL_EXPORT CEHResult CEComPostThreadEvent(eCEComSysEvent event, CEComThreadEventConsumedNotifyProc proc);

	typedef enum
	{
		eCEComThreadActivity_Sleep,
		eCEComThreadActivity_Wake,
		eCEComThreadActivity_Timeout,

		eCEComThreadActivity_Last = eCEComThreadActivity_Timeout,
		eCEComThreadActivity_NumOfActivity = eCEComThreadActivity_Last + 1,
	} eCEComThreadActivity;

	// set time stamp for the specified activity
	DLL_EXPORT void CEComSetThreadActivityTimeStamp(eCEComThreadActivity activity, INT64 timeStamp);

	// get time stamp for the specified activity. return true if the time stamp is available
	DLL_EXPORT bool CEComGetThreadActivityTimeStamp(eCEComThreadActivity activity, INT64* timeStampOut);

	// ignore thread activity calling this function
	DLL_EXPORT void CEComIgnoreThreadActivity();
	
#if defined(_DEBUG)
	#define CECOM_PUSH_THREAD_CONTEXT(iid, iUnk) \
		(CEComSetDebugInfo(CEFUNCSIG, __FILE__), CEComPushThreadContext((iid),(iUnk)))

	#define CECOM_POP_THREAD_CONTEXT(iid) \
		(CEComSetDebugInfo(CEFUNCSIG, __FILE__), CEComPopThreadContext((iid)))

	#define CECOM_PUSH_THREAD_CONTEXTS(numOfIntfs, iids, iUnks) \
		(CEComSetDebugInfo(CEFUNCSIG, __FILE__), CEComPushThreadContexts((numOfIntfs), (iids), (iUnks)))

	#define CECOM_POP_THREAD_CONTEXTS(numOfIntfs, iids) \
		(CEComSetDebugInfo(CEFUNCSIG, __FILE__), CEComPopThreadContexts((numOfIntfs), (iids)))

	#define CECOM_PUSH_CLASS_OBJECTS_TO_THREAD_CONTEXT(numOfIntfs, clsids, iids) \
		(CEComSetDebugInfo(CEFUNCSIG, __FILE__), CEComPushClassObjectsToThreadContext((numOfIntfs), (clsids), (iids)))
#else
	#define CECOM_PUSH_THREAD_CONTEXT(iid, iUnk) \
		(CEComPushThreadContext((iid),(iUnk)))

	#define CECOM_POP_THREAD_CONTEXT(iid) \
		(CEComPopThreadContext((iid)))

	#define CECOM_PUSH_THREAD_CONTEXTS(numOfIntfs, iids, iUnks) \
		(CEComPushThreadContexts((numOfIntfs), (iids), (iUnks)))

	#define CECOM_POP_THREAD_CONTEXTS(numOfIntfs, iids) \
		(CEComPopThreadContexts((numOfIntfs), (iids)))

	#define CECOM_PUSH_CLASS_OBJECTS_TO_THREAD_CONTEXT(numOfIntfs, clsids, iids) \
		(CEComPushClassObjectsToThreadContext((numOfIntfs), (clsids), (iids)))
#endif

	////////////////////////////////////////////////////
	//  Apartment operations
	////////////////////////////////////////////////////

	typedef struct ICEApartment ICEApartment;
	typedef struct ICEApartmentTransferHandler ICEApartmentTransferHandler;

	// neutral/multithread apartment: return singleton instance.
	// singlethread apartment: if caller is sta return itself, otherwize return main sta
	DLL_EXPORT ICEApartment* CEComGetApartment(UINT32 iid);

	DLL_EXPORT CEHResult CEComTransferApartment(ICEApartment* iFromApartment, ICEApartment* iToApartment);

	DLL_EXPORT CEHResult CEComTransferInterfacePointer(ICEApartment* iFromApartment, ICEApartment* iToApartment, UINT32 iid, const void* orig, void* *const transferredOut);

	DLL_EXPORT void CEComDiscardInterfacePointer(UINT32 iid, void* interfacePointer);

	typedef CEHResult (*invoke_message_callback) (UINT32 mid, void* param, void* userData);
	DLL_EXPORT CEHResult CEComPostApartmentMessage(ICEApartment* calleeApartment, ICEApartment* callerApartment, UINT32 mid, void* param, void* userData, invoke_message_callback callback);

	DLL_EXPORT void CEComSetApartmentTransferHandler(ICEApartmentTransferHandler* iTransferHandler);

	///////////////////////////////////
	//  Get location of COM Classes
	///////////////////////////////////

	//! Get the location of the COM class object
	//! \note usually iid is CEComIID_ICFSPathname
	DLL_EXPORT CEHResult CEComGetClassLocation(UINT32 clsid, UINT32 iid, void** ppv);

	///////////////////////////////////
	//  Text formatting functions.
	///////////////////////////////////
	DLL_EXPORT CEHResult CEComSnPrintf(UTF16CHAR* outBuff16, UINT32 *const numOfChars16Ref, const char* format, ...);
	DLL_EXPORT CEHResult CEComVSnPrintf(UTF16CHAR* outBuff16, UINT32 *const numOfChars16Ref, const char* format, va_list args);
	DLL_EXPORT CEHResult CEComISPrintf(ICEUString* buff, const char* format, ...);
	DLL_EXPORT CEHResult CEComVISPrintf(ICEUString* buff, const char* format, va_list args);
	DLL_EXPORT CEHResult CEComISWPrintf(ICEUStringWriter* buff, const char* format, ...);
	DLL_EXPORT CEHResult CEComVISWPrintf(ICEUStringWriter* buff, const char* format, va_list args);

	///////////////////////////////////
	//  Hash functions
	///////////////////////////////////

	//! Calculate Bob-Jenkins Hash
	//! \param[in] key   the key (the unaligned variable-length array of bytes)
	//! \param[in] size  the length of the key, counting by bytes
	DLL_EXPORT UINT32 CEComHash(const UINT8* key, UINT32 size);

	///////////////////////////////////
	//  Source informations.
	///////////////////////////////////
	DLL_EXPORT const CHAR8* CEComGetWorkDirectory();
	DLL_EXPORT INT32 CEComGetP4LastChangeNumber();
	DLL_EXPORT const CHAR8* CEComGetP4LastChangeInformation();
	DLL_EXPORT const CHAR8** CEComGetP4DiffereceFromLastChange();
	DLL_EXPORT const CHAR8** CEComGetP4Opened();

	typedef struct ICEGlobalConfiguration ICEGlobalConfiguration;
	DLL_EXPORT CEHResult CEComSetGlobalConfiguration(ICEGlobalConfiguration* iGlobalConfiguration);
	DLL_EXPORT ICEGlobalConfiguration* CEComGetGlobalConfiguration();
#ifdef __cplusplus
}
#endif // __cplusplus

#endif  //CECom_h

