# -*- tab-width: 8 -*- 
########################################################################################################
### idlfuncs.pl
###   Functions for IDL
###     - perse IDL
###     - write .h and .cpp
########################################################################################################
### Data structure:
###
###    %IDL = ('file1.idl' => $FILE1_REF, 'file2.idl' => $FILE2_REF, ...);
###
###    %FILE = ('name'      => $NAME,
###             'include'   => $INCLUDES_REF,
###             'prototype' => $PROTOTYPES_REF,
###             'enum'      => $ENUMS_REF,
###             'struct'    => $STRUCTS_REF,
###             'interface' => $INTERFACES_REF,
###             'pragma'    => $PRAGMAS_REF,
###             'footer'    => $FOOTER_REF);
###
###    @INCLUDES = ($INCLUDE_REF1, $INCLUDE_REF2, ...);
###
###    %INCLUDE = ('name'  => $NAME,
###                'scope' => 'system' or 'user');
###
###    @PROTOTYPES = ($PROTOTYPE1, $PROTOTYPE2, ...);
###
###    @ENUMS = ($ENUM_REF1, $ENUM_REF2, ...);
###
###    %ENUM = ('name'    => $NAME,
###             'comment' => $COMMENT,
###             'body'    => $BODY);
###
###    @STRUCTS = ($STRUCT_REF1, $STRUCT_REF2, ...);
###
###    %STRUCT = ('name'    => $NAME,
###               'comment' => $COMMENT,
###               'body'    => $BODY);
###
###    @INTERFACES = ($INTERFACE_REF1, $INTERFACE_REF2, ...);
###
###    %INTERFACE = ('name'        => $NAME,
###                  'comment'     => $COMMENT,
###                  'inheritance' => $INHERITANCE,
###                  'function'    => $FUNCTIONS_REF);
###
###    @FUNCTIONS = ($FUNCTION_REF1, $FUNCTION_REF2, ...);
###
###    %FUNCTION = ('name'       => $NAME,
###                 'comment'    => $COMMENT,
###                 'param'      => $PARAM,
###                 'return'     => $RETURN);
###
###    @PRAGMAS = ($PRAGMA_STRING1, $PRAGMA_STRING2, ...);
###
###    %ERRORCODE = ('type'                 => 'CEHResult' or 'CEError',
###                  'return'               => 'CEAPI_RETURN(err)' or 'err',
###                  'code_OperationFailed' => 'CE_SILK_ERR_OPERATION_FAILED' or 'CEAPI_ERR_OPERATION_FAILED',
###                  'code_Success'         => 'CE_S_OK' or 'CEAPI_SUCCESS',
###                  'code_NotFound'        => 'CE_SILK_ERR_NOTFOUND' or 'CEAPI_ERR_NOTFOUND',
###                  'code_BadArgs'         => 'CE_SILK_ERR_BADARGS' or 'CEAPI_ERR_BADARGS');
###
###
###               Note) @OBJECTS = @$OBJECTS_REF, %OBJECT = %$OBJECT_REF
###
########################################################################################################

use strict;

if (&getOSName() eq 'MSWin32')
{
    require 'mswinutil.pl';
}

my($DEBUG) = 0;
my($RECORD_OVERHEAD) = 0;
my($FND_TRANS_FLAG) = 0;

########################################################################################################
### read section
########################################################################################################

sub readIdlFile
{
    my($IDL_REF, $IDL_FULLPATH, @INCLUDEPATHS) = @_;

    ### delete path for databese.
    my($IDL_FILE) = &deletePath($IDL_FULLPATH);
    if ($IDL_REF->{$IDL_FILE})
    {
	### this IDL file is still reading.
	return;
    }

#    print(STDERR "Info: Read file '$IDL_FILE'.\n");

    my(@LINES);
    if ($IDL_FULLPATH)
    {
	open(FP, $IDL_FULLPATH) || die("Error: Can not open file '$IDL_FULLPATH' as input\n");
	@LINES = <FP>;
	close(FP);

	$IDL_REF->{$IDL_FILE}->{'name'} = $IDL_FULLPATH;
    }

    my($LINE_NO) = 0;
    my($LINE);

    my($MODE);
    my(@MODES);
    push(@MODES, 'normal');

    my($ENUM_NAME, $STRUCT_NAME);
    my($BODY);
    my($COMMENT) = ("");
    my($INTERFACE_NO, $STRUCT_NO, $ENUM_NO) = (0, 0, 0);

    ### parse file.
    while (@LINES)
    {
	### update parse mode.
	$MODE = $MODES[$#MODES];

	### increment line number.
	$LINE_NO++;

	### get line.
	$LINE = shift(@LINES);

	### If starting comment.
        if ($LINE =~ /^\s*\/\/\!/ or $LINE =~ /^\s*\/\*\@/)
        {
	    $COMMENT = $COMMENT . $LINE;
        }
	elsif ($LINE =~ /^\s*\/\*\!/)
	{
	    @MODES = &pushMode('comment', $IDL_FILE, $LINE_NO, @MODES);
	    $COMMENT = $COMMENT . "\n" . $LINE;
	}
	### if ending comment.
	elsif ($LINE =~ /\*\// && $MODE eq 'comment')
	{
	    $COMMENT = $COMMENT . $LINE;
	    pop(@MODES);
	}

	### if function decralation.
	elsif ($LINE =~ /^\s*(\w+)\s+(\w+)\s*\(([^\)]*)\)\s*\;\s*$/ && ($MODE eq 'interface'))
	{
	    # [Bug 5741] 2006/2/1 miyasita
	    # replace interface names into struct names.
	    # this is necessary to make the header file accepted by C compiler (not C++).
	    my($REPLACED_PARAM) = &replaceInterfaceWithStruct($3);

	    my($FUNCTION_REF) = {'name' => $2, 'comment' => $COMMENT, 'return' => $1, 'param' => $REPLACED_PARAM};
	    push(@{$IDL_REF->{$IDL_FILE}->{'interface'}->[$INTERFACE_NO]->{'function'}}, $FUNCTION_REF);

	    $COMMENT = "";
	}

	### if starting interface.
	elsif ($LINE =~ /^(interface)\s+(\w+)\s*$/)
	{
	    @MODES = &pushMode('interface', $IDL_FILE, $LINE_NO, @MODES);

	    my($INTERFACE_REF) = {'name' => $2, 'comment' => $COMMENT};
	    push(@{$IDL_REF->{$IDL_FILE}->{'interface'}}, $INTERFACE_REF);

	    $COMMENT = "";
	}
	elsif ($LINE =~ /^(interface)\s+(\w+)\s*\:\s*(\w+)\s*$/)
	{
	    my($INHERITANCE) = $3;
	    if (&getInterfaceHash($IDL_REF, $INHERITANCE))
	    {
		@MODES = &pushMode('interface', $IDL_FILE, $LINE_NO, @MODES);

		my($INTERFACE_REF) = {'name' => $2, 'comment' => $COMMENT, 'inheritance' => $INHERITANCE};
		push(@{$IDL_REF->{$IDL_FILE}->{'interface'}}, $INTERFACE_REF);

		$COMMENT = "";
	    }
	    else
	    {
		die("Error: $IDL_FILE:$LINE_NO: Can not find the interface '$INHERITANCE'.\n");
	    }
	}

	### if starting structure.
	elsif ($LINE =~ /^(struct)\s+(\w+)\s*$/)
	{
	    @MODES = &pushMode('struct', $IDL_FILE, $LINE_NO, @MODES);

	    my($STRUCT_REF) = {'name' => $2};
	    push(@{$IDL_REF->{$IDL_FILE}->{'struct'}}, $STRUCT_REF);

	    $BODY = "";
	}

	### if starting enumerator.
	elsif ($LINE =~ /^(enum)\s+(\w+)\s*$/)
	{
	    @MODES = &pushMode('enum', $IDL_FILE, $LINE_NO, @MODES);

	    my($ENUM_REF) = {'name' => $2};
	    push(@{$IDL_REF->{$IDL_FILE}->{'enum'}}, $ENUM_REF);

	    $BODY = "";
	}

	### if prototype.
	elsif ($LINE =~ /^(struct)\s+(\w+)\s*\;\s*$/)
	{
	    push(@{$IDL_REF->{$IDL_FILE}->{'prototype'}}, $2);
	}

	### if including user header.
	elsif ($LINE =~ /^(\#include)\s+\"(\w+\.(idl|h))\"\s*$/)
	{
	    my($INCLUDE) = $2;
	    if ($INCLUDE =~ /^(\w+).idl$/)
	    {
		my($WANT_FILE) = &findIncludingFile($INCLUDE, @INCLUDEPATHS);
		if ($WANT_FILE)
		{
		    &readIdlFile($IDL_REF, $WANT_FILE, @INCLUDEPATHS);
		}
		else
		{
		    die("Error: $IDL_FILE:$LINE_NO: Can not find file '$INCLUDE' as input.\n");
		}
	    }

	    my($INCLUDE_REF) = {'name' => $INCLUDE, 'scope' => 'user'};
	    push(@{$IDL_REF->{$IDL_FILE}->{'include'}}, $INCLUDE_REF);
	}

	### if including system header.
	elsif ($LINE =~ /^(\#include)\s+\<([\w\/]+\.h)\>\s*$/)
	{
	    my($INCLUDE_REF) = {'name' => $2, 'scope' => 'system'};
	    push(@{$IDL_REF->{$IDL_FILE}->{'include'}}, $INCLUDE_REF);
	}

	### if ending interface, structure or enumerator.
	elsif ($LINE =~ /^\}\s*\;\s*$/ && ($MODE eq 'interface' || $MODE eq 'struct' || $MODE eq 'enum'))
	{
	    ### if ending interface.
	    if ($MODE eq 'interface')
	    {
		$INTERFACE_NO++;
		pop(@MODES);
	    }
	    ### if ending structure.
	    elsif ($MODE eq 'struct')
	    {
		$BODY = $BODY . $LINE;

		$IDL_REF->{$IDL_FILE}->{'struct'}->[$STRUCT_NO]->{'body'} = $BODY;
		$IDL_REF->{$IDL_FILE}->{'struct'}->[$STRUCT_NO]->{'comment'} = $COMMENT;

		$BODY = "";
		$COMMENT = "";

		$STRUCT_NO++;
		pop(@MODES);
	    }
	    ### if ending enumerator.
	    elsif ($MODE eq 'enum')
	    {
		$BODY = $BODY . $LINE;

		$IDL_REF->{$IDL_FILE}->{'enum'}->[$ENUM_NO]->{'body'} = $BODY;
		$IDL_REF->{$IDL_FILE}->{'enum'}->[$ENUM_NO]->{'comment'} = $COMMENT;

		$BODY = "";
		$COMMENT = "";

		$ENUM_NO++;
		pop(@MODES);
	    }
	}

	### if pragma    e.g.   @pragma(  external_ref_def("ICESymbol_RefDef_only.h")  )
	###                               <----------------------------------------->
	###                                         |
	###                                         | $1
	###                                         V
	###                                   <----------->
	elsif ($LINE =~ /^\s*\@pragma\s*\(\s*(\w+\([^\)]*\))\s*\)\s*$/)
	{
	    my($PRAGMA) = $1;
	    push(@{$IDL_REF->{$IDL_FILE}->{'pragma'}}, $PRAGMA);
	}

	else
	{
	    if ($MODE eq 'comment')
	    {
		$COMMENT = $COMMENT . $LINE;
	    }
	    elsif ($MODE eq 'struct' || $MODE eq 'enum')
	    {
		$BODY = $BODY . $LINE;
	    }
	}
    }

    my($FOOTER_REF) = {'comment' => $COMMENT};
    push(@{$IDL_REF->{$IDL_FILE}->{'footer'}}, $FOOTER_REF);

}

sub pushMode
{
    my($NEW, $IDL_FILE, $LINE_NO, @MODES) = @_;
    my($ERROR) = 1;
    my($MODE) = $MODES[$#MODES];

    if ($MODE eq 'normal')
    {
	if ($NEW ne 'normal')
	{
	    $ERROR = 0;
	}
    }
    elsif ($MODE eq 'interface' || $MODE eq 'struct' || $MODE eq 'enum' || $MODE eq 'class')
    {
	if ($NEW eq 'comment')
	{
	    $ERROR = 0;
	}
    }
    else
    {
    }

    if ($ERROR)
    {
	die("Error: $IDL_FILE:$LINE_NO: Parse error. Can not change the read mode for $MODE to $NEW\n");
    }
    else
    {
	push(@MODES, $NEW);
    }

    return @MODES;
}

# make dummy idl hash
# for ccd wizard
sub makeDummyIdlFile
{
    my($IDL_REF, $IDL_FULLPATH, @INCLUDEPATHS) = @_;

    ### delete path for databese.
    my($IDL_FILE) = &deletePath($IDL_FULLPATH);
    if ($IDL_REF->{$IDL_FILE})
    {
	### this IDL file is still reading.
	return;
    }

    if ($IDL_FILE =~ /^(.*)\.idl$/)
    {
        my($INTERFACE_REF) = {'name' => $1, 'comment' => 'dummy'};
        push(@{$IDL_REF->{$IDL_FILE}->{'interface'}}, $INTERFACE_REF);
    }
    
}

########################################################################################################
### write section
########################################################################################################

sub writeCopyright
{
    my($sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $isdst) = localtime(time);
    $year += 1900;

    &writeString("///////////////////////////////////////////////////////////////////////////////\n");
    &writeString("// Copyright $year Sony Corporation\n");
    &writeString("///////////////////////////////////////////////////////////////////////////////\n");
}

sub writeHeader
{
    &writeCopyright();
    &writeString("\n");

    my($H_FILE) = @_;
    my(@PATHS) = split(/[\/\\]/, $H_FILE);
    my($HEADER_MACRO) = pop(@PATHS);
    $HEADER_MACRO =~ tr/\./_/;
    &writeString("#ifndef $HEADER_MACRO\n");
    &writeString("#define $HEADER_MACRO\n");
}

sub writeDefine
{
}

sub writeIncludeList
{
    my(@INCLUDES) = @_;
    my($INCLUDE);
    foreach $INCLUDE (@INCLUDES)
    {
	if ($INCLUDE =~ /^([\w\/]+).(idl|h|ccd|csd)$/)
	{
	    &writeInclude("$1.h", 'user');
	}
	
	if ($FND_TRANS_FLAG && $INCLUDE =~ /^([\w\/]+).(idl)$/)
	{
	    my($FND_IDL_FILE_NAME) = &replaceIdlFilePrefixForPublic($1);
	    &writeInclude("$FND_IDL_FILE_NAME.h", 'user');
	}
    }

    ### adhoc
    if ($FND_TRANS_FLAG)
    {
	&writeInclude('ICETextSupport.h', 'user');
	&writeInclude('ceulinearlists.h', 'user');
	&writeString("\n");
    }
}

sub writeOrgIncludeList
{
    my(@INCLUDE_REFS) = @_;
    my($INCLUDE_REF);
    foreach $INCLUDE_REF (@INCLUDE_REFS)
    {
	my($INCLUDE) = $INCLUDE_REF->{'name'};
	if ($INCLUDE =~ /^ICEApx.*idl$/ || $INCLUDE =~ /^ICEMss.*idl$/)
	{
	    $INCLUDE = &replaceIdlFilePrefixForPublic($INCLUDE);
	}
	&writeInclude($INCLUDE, 'user');
    }
}

sub writeScopedIncludeList
{
    my(@INCLUDE_REFS) = @_;
    my($INCLUDE_REF);
    foreach $INCLUDE_REF (@INCLUDE_REFS)
    {
	if ($INCLUDE_REF->{'name'} =~ /^([\w\/]+).(idl|h|ccd|csd)$/)
	{
	    &writeInclude("$1.h", $INCLUDE_REF->{'scope'});
	}
    }
}

sub writeInclude
{
    my($NAME, $SCOPE) = @_;
    if ($SCOPE eq 'user')
    {
	&writeString("#include \"$NAME\"\n");
    }
    else
    {
	&writeString("#include \<$NAME\>\n");
    }
}

sub writePrototypeList
{
    my(@PROTOTYPES) = @_;
    my($PROTOTYPE);
    foreach $PROTOTYPE (@PROTOTYPES)
    {
	&writeString("struct $PROTOTYPE;\n");
    }
}

sub writeOrgPrototypeList
{
    my(@PROTOTYPES) = @_;
    my($PROTOTYPE);
    foreach $PROTOTYPE (@PROTOTYPES)
    {
	die ("Error: Not implemented yet\n");
	#&writeString("struct $PROTOTYPE;\n");
    }
}

sub writeStructEnumeratorList
{
    my($TYPE, $FILE_REF) = @_;
    my($REF);
    foreach $REF (@{$FILE_REF->{$TYPE}})
    {
	my(%HASH) = %$REF;

	&writeString("$HASH{'comment'}");
	&writeString("typedef $TYPE");

	if ($TYPE eq 'struct')
	{
		&writeString(" $HASH{'name'}");
	}
	&writeString("\n");

	my($BODY) = $HASH{'body'};
	if ($BODY =~ /^([^\}]+)\}\s*;\s*$/)
	{
	    $BODY = $1;
	    &writeString("$BODY} $HASH{'name'};\n");
	}
	else
	{
	    die("Error: Syntax error. \'$HASH{'name'}\'\n");
	}

	&writeString("\n");
    }
}

sub writeOrgStructEnumeratorList
{
    my($TYPE, $FILE_REF) = @_;
    my($REF);
    foreach $REF (@{$FILE_REF->{$TYPE}})
    {
	my(%HASH) = %$REF;

	&writeString("$HASH{'comment'}");
	&writeString("$TYPE $HASH{'name'}\n");

	my($BODY) = $HASH{'body'};
	if ($BODY =~ /^([^\}]+)\}\s*;\s*$/)
	{
	    $BODY = $1;
	    &writeString("$BODY};\n");
	}
	else
	{
	    die("Error: Syntax error. \'$HASH{'name'}\'\n");
	}

	&writeString("\n");
    }
}

sub writeFunctionListWithInheritance
{
    my($IDL_REF, $MODE, $CLASS_REF, $TOPLEVELNAME, $WRITE_SPX, %INTERFACE) = @_;
    if (%INTERFACE)
    {
	my($INHERITANCE) = $INTERFACE{'inheritance'};
	if ($INHERITANCE)
	{
	    my(%INHERITANCE_INTERFACE) = &getInterfaceHash($IDL_REF, $INHERITANCE);
	    if (%INHERITANCE_INTERFACE)
	    {
		if ($FND_TRANS_FLAG && $WRITE_SPX)
		{
		    &writeFunctionListWithInheritance($IDL_REF, $MODE, $CLASS_REF, $TOPLEVELNAME, $WRITE_SPX, %INHERITANCE_INTERFACE);
		}
		else
		{
		    &writeFunctionListWithInheritance($IDL_REF, $MODE, $CLASS_REF, $TOPLEVELNAME, 0, %INHERITANCE_INTERFACE);
		}
	    }
	}
	if ($FND_TRANS_FLAG && $WRITE_SPX)
	{
	    &writeFunctionList($IDL_REF, $MODE, $CLASS_REF, $TOPLEVELNAME, $WRITE_SPX, @{$INTERFACE{'function'}});
	}
	else
	{
	    &writeFunctionList($IDL_REF, $MODE, $CLASS_REF, $TOPLEVELNAME, 0, @{$INTERFACE{'function'}});
	}
    }
}

sub writeFunctionList
{
    my($IDL_REF, $MODE, $CLASS_REF, $TOPLEVELNAME, $WRITE_SPX, @FUNCTIONS) = @_;
    my($FUNCTION_REF);
    foreach $FUNCTION_REF (@FUNCTIONS)
    {
	if ($FND_TRANS_FLAG && $WRITE_SPX)
	{
	    &writeFunction($IDL_REF, $MODE, $CLASS_REF, $TOPLEVELNAME, $WRITE_SPX, %$FUNCTION_REF);
	}
	else
	{
	    &writeFunction($IDL_REF, $MODE, $CLASS_REF, $TOPLEVELNAME, 0, %$FUNCTION_REF);
	}
    }
}

sub convertVariableLengthArgument
{
    my($PARAM) = @_;
    my(@PARAMS) = split(/\,/, $PARAM);
    my($RESULT) = 0;
    if ($PARAMS[$#PARAMS] =~ /\s*\.\.\.\s*/)
    {
	$PARAMS[$#PARAMS] = 'va_list args';
	$RESULT = join(', ', @PARAMS);
    }
    return $RESULT;
};

sub writeFunction
{
    my($IDL_REF, $MODE, $CLASS_REF, $TOPLEVELNAME, $WRITE_SPX, %FUNCTION) = @_;
    my($CLASS) = $CLASS_REF->{'name'};
    if ($CLASS_REF->{'marshal'})
    {
	$CLASS = &getProxyClassName($CLASS_REF);
    }
    my($PREFIX) = $TOPLEVELNAME;
    $PREFIX =~ s/^I/i/;

    my($FUNCTION_NAME) = $FUNCTION{'name'};
    my($PARAM_WO_THIS) = $FUNCTION{'param'};
    my($VAARG_PARAM) = &convertVariableLengthArgument($FUNCTION{'param'});
    if ($VAARG_PARAM)
    {
	$FUNCTION_NAME = 'v' . $FUNCTION{'name'};
	$PARAM_WO_THIS = $VAARG_PARAM;
    }

    my($CFUNCTION_NAME) = "${PREFIX}_$FUNCTION_NAME";

    # class vtable implementation
    # C vtable header
    # class functions implementation
    # C functions header
    # C Wrapper implementation

    # vtable member in *.cpp

    if ($MODE eq 'CLASS_VTABLE_IMPL')
    {
	my($FUNC_IMPL_NAME) = &getImplementationMethodVtblName($CLASS, $FUNCTION_NAME);
	if ($FND_TRANS_FLAG && $WRITE_SPX)
	{
	    $FUNC_IMPL_NAME = &replaceVtableFunctionNamePrefixForPublic($FUNC_IMPL_NAME);
	}
	&writeString("\t$FUNC_IMPL_NAME,\n");
    }
    # vtable member in *.h
    elsif ($MODE eq 'C_VTABLE_HEADER')
    {
	my($SUMMARY) = &getSummary($FUNCTION{'comment'});
	if ($SUMMARY)
	{
	    &writeString("\t$CFUNCTION_NAME\t_$FUNCTION_NAME;\t//!< $SUMMARY\n");
	}
	else
	{
	    &writeString("\t$CFUNCTION_NAME\t_$FUNCTION_NAME;\n");
	}
    }
    # implementation of function in *.cpp
    elsif ($MODE eq 'CLASS_FUNCTION_IMPL')
    {
	my($PARAM) = &addThisParameter($PARAM_WO_THIS, $TOPLEVELNAME, &getInterfaceValueName($TOPLEVELNAME));
	if ($FND_TRANS_FLAG && $WRITE_SPX)
	{
	    $TOPLEVELNAME = &replaceInterfacePrefixForPublic($TOPLEVELNAME);
	    &writeFunctionImplementation($IDL_REF, $CLASS_REF, $PARAM, $FUNCTION_NAME, $TOPLEVELNAME, $WRITE_SPX, %FUNCTION);
	}
	else
	{
	    &writeFunctionImplementation($IDL_REF, $CLASS_REF, $PARAM, $FUNCTION_NAME, $TOPLEVELNAME, 0, %FUNCTION);
	}

    }
    # type define of function in *.h
    elsif ($MODE eq 'C_FUNCTION_HEADER')
    {
	my($PARAM) = &addThisParameter($PARAM_WO_THIS, $TOPLEVELNAME, &getInterfaceValueName($TOPLEVELNAME));
	&writeFunctionComment($FUNCTION{'comment'}, &getInterfaceValueName($TOPLEVELNAME));
	&writeString("typedef $FUNCTION{'return'} (*$CFUNCTION_NAME) ($PARAM);\n");
	&writeString("\n");

    }
    # implementation of function in *Ref.h
    elsif ($MODE eq 'CREF_FUNCTION_IMPL')
    {
	## put it after return type
	my($FORCE_INLINE_GCC) = &getGccInlineOption();

	# TODO: clean this block
	my($THIS) = &getPrivateInterfaceValueName($TOPLEVELNAME);
	my($PARAM) = &addThisParameter($PARAM_WO_THIS, $TOPLEVELNAME, $THIS);
	my(@VALUES) = &getValuesFromAllParameter($PARAM);
	my($VALUE_LIST) = join(', ', @VALUES);

	my($NULL_RETURN);

	if ($FUNCTION{'return'} eq 'CEHResult')
	{
	    $NULL_RETURN = 'CE_SILK_ERR_UNINITIALIZED';
	}
	else
	{
	    $NULL_RETURN = '0';
	}

	&writeString("\t$FORCE_INLINE_GCC $FUNCTION{'return'} $FUNCTION_NAME($PARAM_WO_THIS)");
	if ($FUNCTION{'return'} eq 'void')
	{
	    ## TODO: I will delete null check.
##	    &writeString("\t{ CEASSERT($THIS && \"$THIS is not initialized\"); $THIS->_vtbl->_$FUNCTION_NAME($VALUE_LIST); }\n\n");
	    &writeString("\t{ if ($THIS) { $THIS->_vtbl->_$FUNCTION_NAME($VALUE_LIST); } }\n\n");
	}
	else
	{
	    ## TODO: I will delete null check.
##	    &writeString("\t{ CEASSERT($THIS && \"$THIS is not initialized\"); return $THIS->_vtbl->_$FUNCTION_NAME($VALUE_LIST); }\n\n");
	    &writeString("\t{ return $THIS ? $THIS->_vtbl->_$FUNCTION_NAME($VALUE_LIST) : $NULL_RETURN; }\n\n");
	}

	if ($VAARG_PARAM)
	{
	    # respect original idl's declaration
	    &writeString("\t$FORCE_INLINE_GCC $FUNCTION{'return'} $FUNCTION{'name'}($FUNCTION{'param'})");

	    my($LAST) = $VALUES[$#VALUES - 1];
	    my ($PRE_CALL, $POST_CALL);

	    $PRE_CALL = "\t\tva_list args;\n\t\tva_start(args, $LAST);\n";
	    $POST_CALL = "\t\tva_end(args);\n";

	    my ($RETURN_VALUE) = 'cecomrefReturnValue';
	    &writeString("\n");
	    &writeString("\t{\n");
	    &writeString("\t\t$FUNCTION{'return'} $RETURN_VALUE = $NULL_RETURN;\n");
	    &writeString($PRE_CALL);
	    &writeString("\t\t$RETURN_VALUE = $THIS ? $THIS->_vtbl->_$FUNCTION_NAME($VALUE_LIST) : " . $NULL_RETURN . ";\n");
	    &writeString($POST_CALL);
	    &writeString("\t\treturn $RETURN_VALUE;\n");
	    &writeString("\t}\n");
	}

	if ( $FUNCTION_NAME eq 'dbgToString' )
	{
	    &writeString("#if defined(_DEBUG) || !defined(NDEBUG)\n\tconst UTF16CHAR* dbgToString() { const UTF16CHAR* ret=0; if ($THIS) { dbgToString(&ret); } return ret; }\n#endif // _DEBUG || !NDEBUG\n\n");
	}

    }
    elsif ($MODE eq 'LWSTUB_FUNCTION_HEADER' || $MODE eq 'STDSTUB_FUNCTION_HEADER' || $MODE eq 'IPCSTUB_FUNCTION_HEADER')
    {
	my($STUB_FUNCTION_NAME) = &getStubFunctionName($CLASS_REF->{'marshal'}, $FUNCTION_NAME);
	if ($FUNCTION_NAME ne 'addRef' && $FUNCTION_NAME ne 'release' )
	{
	    my($ORIG_NAME) = &getStubOriginalInterfaceName($TOPLEVELNAME);
	    my(@VALUES) = &getValuesFromAllParameter($PARAM_WO_THIS);
	    my($VALUE_LIST) = join(', ', @VALUES);

	    ## error type is same with interface
	    if ($MODE eq 'LWSTUB_FUNCTION_HEADER')
	    {
		&writeString("\tCEHResult $STUB_FUNCTION_NAME($PARAM_WO_THIS) { return $ORIG_NAME.$FUNCTION_NAME($VALUE_LIST); }\n");
	    }
	    elsif ($MODE eq 'IPCSTUB_FUNCTION_HEADER')
	    {
		if ($FUNCTION_NAME ne 'queryInterface')
		{
		    &writeString("\tCEHResult $STUB_FUNCTION_NAME($PARAM_WO_THIS) { return $ORIG_NAME.$FUNCTION_NAME($VALUE_LIST); }\n");
		}
	    }
	    elsif ($MODE eq 'STDSTUB_FUNCTION_HEADER')
	    {
		my($PARAM) = 'ICEApartment* calleeApartment, ICEApartment* callerApartment';
		if ($PARAM_WO_THIS)
		{
		    $PARAM .=  ', ' . $PARAM_WO_THIS;
		}
		&writeString("\tCEHResult $STUB_FUNCTION_NAME($PARAM);\n");
	    }
	    else
	    {
		die "unknown function header mode:$MODE\n"
	    }
	}
    }
    elsif ($MODE eq 'PROXY_FUNCTION_IMPL')
    {
	my($PARAM) = &addThisParameter($PARAM_WO_THIS, $TOPLEVELNAME, &getInterfaceValueName($TOPLEVELNAME));
	if ($FND_TRANS_FLAG && $WRITE_SPX)
	{
	    &writeFunctionImplementation($IDL_REF, $CLASS_REF, $PARAM, $FUNCTION_NAME, $TOPLEVELNAME, $WRITE_SPX, %FUNCTION);
	}
	else
	{
	    &writeFunctionImplementation($IDL_REF, $CLASS_REF, $PARAM, $FUNCTION_NAME, $TOPLEVELNAME, 0, %FUNCTION);
	}
    }
    elsif ($MODE eq 'MARSHAL_FACTORY_FUNCTION_HEADER')
    {
	if ($FUNCTION_NAME eq 'queryInterface' || $FUNCTION_NAME eq 'addRef' || $FUNCTION_NAME eq 'release' )
	{
	    ## do nothing
	}
	elsif ($FUNCTION_NAME eq 'createInstance')
	{
	    ## deside appropriate error
	    &writeString("\tCEHResult $FUNCTION_NAME($PARAM_WO_THIS) { return CE_SILK_ERR_OPERATION_FAILED; }\n");
	}
	else
	{
	    &writeString("\tCEHResult $FUNCTION_NAME($PARAM_WO_THIS);\n");
	}
    }
    elsif ($MODE eq 'MARSHAL_FACTORY_FUNCTION_IMPL')
    {
	if ($FUNCTION_NAME eq 'createProxy')
	{
	    my($APARTMENT_VALUE, $IID_VALUE, $ORIG_VALUE, $PROXYOUT_VALUE) = &getValuesFromAllParameter($PARAM_WO_THIS);
	    &writeString("CEHResult $CLASS_REF->{'name'}\:\:$FUNCTION_NAME($PARAM_WO_THIS)\n");
	    &writeString("{\n");
	    &writeString("\tCEHResult hr = CE_SILK_ERR_BADARGS;\n");
	    &writeString("\tif ($ORIG_VALUE && $PROXYOUT_VALUE)\n");
	    &writeString("\t{\n");
	    &writeString("\t\tswitch($IID_VALUE)\n");
	    &writeString("\t\t{\n");
	    my($CREATABLE_REF);
	    foreach $CREATABLE_REF (@{$CLASS_REF->{'creatable'}})
	    {
		my($INTERFACE_REF) = $CREATABLE_REF->{'interface'}->[0];
		my($INTERFACE_NAME) = $INTERFACE_REF->{'name'};
		my($IID) = &getIIDName($INTERFACE_NAME);
		&writeString("\t\tcase $IID:\n");
		&writeString("\t\t{\n");
		&writeString("\t\t\t$CREATABLE_REF->{'name'}* proxyObject = new $CREATABLE_REF->{'name'}(this, reinterpret_cast<$INTERFACE_NAME*>($ORIG_VALUE), $APARTMENT_VALUE);\n");
		&writeString("\t\t\tif (proxyObject)\n");
		&writeString("\t\t\t{\n");

		&writeString("\t\t\t\t$INTERFACE_REF->{'name'}* proxyInterface = $CREATABLE_REF->{'name'}\:\:to$INTERFACE_NAME(proxyObject);\n");
		&writeString("\t\t\t\tproxyInterface->_vtbl->_addRef(proxyInterface);\n");
		&writeString("\t\t\t\t*$PROXYOUT_VALUE = proxyInterface;\n");
		&writeString("\t\t\t\thr = CE_S_OK;\n");
		&writeString("\t\t\t}\n");
		&writeString("\t\t\tbreak;\n");
		&writeString("\t\t}\n");
	    }
	    &writeString("\t\tdefault:\n");
	    &writeString("\t\t{\n");
	    &writeString("\t\t\thr = CE_SILK_ERR_NOTFOUND;\n");
	    &writeString("\t\t\tbreak;\n");
	    &writeString("\t\t}\n");
	    &writeString("\t\t}\n");
	    &writeString("\t}\n");
	    &writeString("\treturn hr;\n");
	    &writeString("}\n");
	}
    }
    else
    {
	die("Fatal: Unknown writing mode '$MODE'.\n");
    }
}

sub addThisParameter
{
    my($PARAM_LIST, $TYPE, $VALUE) = @_;

    my(@PARAMS) = split(/, /, $PARAM_LIST);
    unshift(@PARAMS, "$TYPE* $VALUE");
    return join(', ', @PARAMS);
}

sub writeFunctionImplementation
{
    my($IDL_REF, $CLASS_REF, $PARAM_LIST, $OVERWRITE_FUNC_NAME, $TOPLEVEL_INTERFACE_NAME, $WRITE_SPX, %FUNCTION) = @_;
    my($CLASS) = $CLASS_REF->{'name'};
    if ($CLASS_REF->{'marshal'})
    {
	$CLASS = &getProxyClassName($CLASS_REF);
    }

    my($FUNCTION_NAME) = $FUNCTION{'name'};
    if ($OVERWRITE_FUNC_NAME)
    {
	$FUNCTION_NAME = $OVERWRITE_FUNC_NAME;
    }
    my($FUNC_IMPL_NAME) = &getImplementationMethodVtblName($CLASS, $FUNCTION_NAME);

    my($ORG_PARAM_LIST)  = 0;
    my($FND_TRANS)       = $CLASS_REF->{'fnd-trans'};
    my(@ICEUSTRING_VARS) = ();
    my(@CERECT_VARS)     = ();
    my(@CERECTF_VARS)     = ();
    my(@CERAWDATABUFFER_VARS)  = ();
    my(@CEUSTRINGLIST_VARS) = ();
    my(@CEUUINT32LIST_VARS) = ();
    my(@VALUES)          = ();
    my($FUNCTION_PREFIX) = 'Apm';
    if ($CLASS_REF->{'function-prefix'})
    {
	$FUNCTION_PREFIX = $CLASS_REF->{'function-prefix'};
    }

    if ($FND_TRANS_FLAG)
    {
	$ORG_PARAM_LIST  = $PARAM_LIST;
	&fndTransFunctionParam($PARAM_LIST, $FND_TRANS, \@ICEUSTRING_VARS, \@CERECT_VARS, \@CERECTF_VARS, \@CERAWDATABUFFER_VARS, \@CEUSTRINGLIST_VARS, \@CEUUINT32LIST_VARS);
	$PARAM_LIST = &replaceInterfacePrefixForPublic($PARAM_LIST);
	if ($WRITE_SPX)
	{
	    $FUNC_IMPL_NAME = &replaceVtableFunctionNamePrefixForPublic($FUNC_IMPL_NAME);
	    &writeString("static $FUNCTION{'return'} $FUNC_IMPL_NAME($PARAM_LIST)\n");
	    if ($FND_TRANS eq ':trans-to-fnd')
	    {
		@VALUES = &getValuesFromAllParameter($ORG_PARAM_LIST);
	    }
	    elsif ($FND_TRANS eq ':trans-from-fnd')
	    {
		@VALUES = &getValuesFromAllParameter($ORG_PARAM_LIST);
	    }
	    else
	    {
		die("Error: Wrong type $FND_TRANS, stopped");
	    }
	}
	else
	{
	    &writeString("static $FUNCTION{'return'} $FUNC_IMPL_NAME($ORG_PARAM_LIST)\n");
	    if ($FND_TRANS eq ':trans-to-fnd')
	    {
		@VALUES = &getValuesFromAllParameter($ORG_PARAM_LIST);
	    }
	    elsif ($FND_TRANS eq ':trans-from-fnd')
	    {
		@VALUES = &getValuesFromAllParameter($PARAM_LIST);
	    }
	    else
	    {
		die("Error: Wrong type $FND_TRANS, stopped");
	    }
	}
    }
    else
    {
	&writeString("static $FUNCTION{'return'} $FUNC_IMPL_NAME($PARAM_LIST)\n");
	@VALUES = &getValuesFromAllParameter($PARAM_LIST);
    }
    &writeString("{\n");

    my($THREAD_APARTMENT_VALUE) = 'threadApartment';
    my($THIS)                   = shift(@VALUES);
    my($VALUE_LIST)             = join(', ', @VALUES);

    if ($FND_TRANS_FLAG)
    {
	if ($WRITE_SPX)
	{
	    $THIS = &replaceFunctionParamVariablePrefixForPublic($THIS);
	}
	else
	{
	    $THIS = &replaceFunctionParamVariablePrefixForPrivate($THIS);
	}
    }

    if ($FUNCTION_NAME eq 'queryInterface')
    {
	if (!$CLASS_REF->{'marshal'} || $CLASS_REF->{'marshal'} eq ':ipc-marshal')
	{
	    &writeString("\tCEHResult hr = _$FUNCTION_NAME(${CLASS}::toInstance($THIS), $VALUE_LIST);\n");
	}
	else
	{
	    my($PROXY_APARTMENT_VALUE) = 'proxyaPartment';
	    my($IID) = $VALUES[0];
	    my($ORIG_OUT) = $VALUES[1];
	    &writeString("\tICEApartment* $PROXY_APARTMENT_VALUE = ${CLASS}::toInstance($THIS)->getApartment();\n");
	    &writeString("\tCEComICEApartmentRef $THREAD_APARTMENT_VALUE = 0;\n");
	    &writeString("\tCEHResult hr = CEComGetThreadContext(CEComIID_ICEApartment, reinterpret_cast<void**>(&$THREAD_APARTMENT_VALUE));\n");
	    &writeString("\tif (CESucceeded(hr))\n");
	    &writeString("\t{\n");

	    &writeString("\t\tbool isMarshalNecessary = ($PROXY_APARTMENT_VALUE != 0) && ($PROXY_APARTMENT_VALUE != $THREAD_APARTMENT_VALUE);\n");
	    &writeString("\t\tif (isMarshalNecessary)\n");
	    &writeString("\t\t{\n");
	    &writeString("\t\t\tCEComTransferApartment($THREAD_APARTMENT_VALUE, $PROXY_APARTMENT_VALUE);\n");
	    &writeString("\t\t}\n");
	    if ($CLASS_REF->{'marshal'} eq ':lw-marshal')
	    {
		&writeString("\t\thr = _$FUNCTION_NAME(${CLASS}::toInstance($THIS), $VALUE_LIST);\n");
	    }
	    elsif ($CLASS_REF->{'marshal'} eq ':std-marshal')
	    {
		&writeString("\t\thr = _$FUNCTION_NAME($THREAD_APARTMENT_VALUE, ${CLASS}::toInstance($THIS), $VALUE_LIST);\n");
	    }
	    else
	    {
		die "unknown marshalling type $CLASS_REF->{'marshal'}\n";
	    }

	    &writeString("\t\tif (isMarshalNecessary)\n");
	    &writeString("\t\t{\n");
	    &writeString("\t\t\tCEComTransferApartment($PROXY_APARTMENT_VALUE, $THREAD_APARTMENT_VALUE);\n");
	    &writeString("\t\t\tif (CESucceeded(hr))\n");
	    &writeString("\t\t\t{\n");
	    &writeCreateHook($IDL_REF, "\t\t\t\t", $PROXY_APARTMENT_VALUE, $THREAD_APARTMENT_VALUE, $IID, $ORIG_OUT, 'void');
	    &writeString("\t\t\t}\n");
	    &writeString("\t\t}\n");

	    &writeString("\t}\n");
	}
	&writeString("\treturn hr;\n");
    }
    elsif ($FUNCTION_NAME eq 'addRef' || $FUNCTION_NAME eq 'release')
    {
	## addRef/release proxy's refCount
	&writeString("\t_$FUNCTION_NAME(${CLASS}::toInstance($THIS));\n");
    }
    else
    {
	&writeString("\t//CEASSERT($THIS && \"$THIS is 0\");\n");
	&writeString("\n");

	&writeString("\tCEHResult hr = CE_SILK_ERR_OPERATION_FAILED;\n");
	&writeString("\n");

	my($THIS_VALUE) = &getPointerValueName($CLASS);
	my($PROXY_APARTMENT_VALUE) = 'proxyApartment';

	if ($CLASS_REF->{'marshal'})
	{
# 	    my($PROP_NAME, $PROP_OUT);
# 	    if ($CLASS_REF->{'marshal-cachable'} && &isCachableFunction(\%FUNCTION))
# 	    {
# 		$PROP_NAME = &extractPropertyName($FUNCTION_NAME);
# 		$PROP_OUT = &extractPropertyOut(\%FUNCTION);
# 		&writeCacheReturn($THIS_VALUE, $PROP_NAME, $PROP_OUT);
# 	    }

	    if ($CLASS_REF->{'marshal'} eq ':lw-marshal' || $CLASS_REF->{'marshal'} eq ':std-marshal')
	    {
		&writeString("\t$CLASS* $THIS_VALUE = ${CLASS}::toInstance($THIS);\n");
		&writeString("\tICEApartment* $PROXY_APARTMENT_VALUE = $THIS_VALUE->getApartment();\n");
		&writeString("\tCEComICEApartmentRef $THREAD_APARTMENT_VALUE = 0;\n");
		&writeString("\thr = CEComGetThreadContext(CEComIID_ICEApartment, reinterpret_cast<void**>(&$THREAD_APARTMENT_VALUE));\n");

		my($ADDITIONAL_CONDITION) = '';
		my(@PARAMS) = split(/\,/, $PARAM_LIST);
		my($THIS_PARAM) = $PARAMS[0];
		my($THIS_TYPE) = &getType($THIS_PARAM);
		if ($THIS_TYPE =~ /(.*)\*\s*$/)
		{
		    my($THIS_INTERFACE) = $1;
		    if (&getInterfaceHash($IDL_REF, $THIS_INTERFACE))
		    {
			if ($THIS_INTERFACE eq 'ICEURL' || $THIS_INTERFACE eq 'ICEUString')
			{
			    if ($FUNCTION_NAME ne 'isImmutable')
			    {
				&writeString("\tbool isImmutable = false;\n");
				&writeString("\tif (CESucceeded(hr))\n");
				&writeString("\t{\n");
				&writeString("\t\thr = $THIS->_vtbl->_isImmutable($THIS, &isImmutable);\n");
				&writeString("\t}\n");

				$ADDITIONAL_CONDITION = '&& !isImmutable';
			    }
			}
		    }
		}
		&writeString("\tif (CESucceeded(hr))\n");
		&writeString("\t{\n");
		&writeString("\t\tbool isMarshalNecessary = ($PROXY_APARTMENT_VALUE != 0) && ($PROXY_APARTMENT_VALUE != $THREAD_APARTMENT_VALUE) $ADDITIONAL_CONDITION;\n");
		&writeString("\n");
	    }
	    elsif ($CLASS_REF->{'marshal'} eq ':ipc-marshal')
	    {
		&writeString("\thr = CE_S_OK;\n");
	    }
	    else
	    {
		die "unknown marshaller type $CLASS_REF->{'marshal'}";
	    }


	    my(%INTERFACE_OUTS);
	    my(%INTERFACE_INS);

	    ## at first parse param
	    my(@PARAMS) = split(/\s*,\s*/, $PARAM_LIST);
	    ## remove this
	    shift(@PARAMS);
	    my($PARAM);
	    while (@PARAMS)
	    {
		$PARAM = shift(@PARAMS);
		if ($PARAM =~ /(ICE\w*)\*\s*(\*const|\*|)\s+(\w+)/)
		{
		    my($INTERFACE_CANDIDATE) = $1;
		    my($IS_OUTPUT) = $2;
		    my($VALUE_CANDIDATE) = $3;
		    if (&getInterfaceHash($IDL_REF, $INTERFACE_CANDIDATE))
		    {
			my(%INTERFACE_HASH) = &getInterfaceHash($IDL_REF, $INTERFACE_CANDIDATE);
			if ($IS_OUTPUT)
			{
			    $INTERFACE_OUTS{$VALUE_CANDIDATE} = \%INTERFACE_HASH;
			}
			else
			{
			    $INTERFACE_INS{$VALUE_CANDIDATE} = \%INTERFACE_HASH;
			}
		    }
		    else
		    {
			if ($INTERFACE_CANDIDATE ne 'ICEUString')
			{
			    die "Error: param seems interface. but unknown: $INTERFACE_CANDIDATE in param: $PARAM in function: $FUNCTION_NAME($PARAM_LIST)";
			}
		    }
		}
	    }

	    ## just declare transferred pointer to release later
	    my($VALUE, $INTERFACE_REF);
	    while (($VALUE, $INTERFACE_REF) = each(%INTERFACE_INS))
	    {
		if ($CLASS_REF->{'marshal'} eq ':lw-marshal' || $CLASS_REF->{'marshal'} eq ':std-marshal')
		{
		    &writeString("\t\tCE_MAY_ALIAS_TYPE($INTERFACE_REF->{'name'})* proxy${VALUE}Out = 0;\n");2
		}
		elsif ($CLASS_REF->{'marshal'} eq ':ipc-marshal')
		{
		    my($STUB_NAME) = &getStubClassName($CLASS_REF);
		    &writeString("\t\tCE_MAY_ALIAS_TYPE($STUB_NAME)* stub${VALUE}Out = 0;\n");2
		}
		else
		{
		    die "unknown marshal type $CLASS_REF->{'marshal'}";
		}

	    }

	    my($DBG_TEXT) = &getDebugText($CLASS_REF->{'name'}, $FUNCTION_NAME);
	    my($DBG_LEN) = &getDebugLen($CLASS_REF->{'name'}, $FUNCTION_NAME);

	    my($CYCLS_VALUE_START) = 'cyclesStart';
	    my($CYCLS_VALUE_END) = 'cyclesEnd';
	    if ($RECORD_OVERHEAD)
	    {
		if (&getOSName() ne 'MSWin32')
		{
		    &writeString("\t\ttypedef union _LARGE_INTEGER {\n");
		    &writeString("\t\t\tstruct {\n");
		    &writeString("\t\t\t\tINT32 LowPart;\n");
		    &writeString("\t\t\t\tINT32 HighPart;\n");
		    &writeString("\t\t\t};\n");
		    &writeString("\t\t\tUINT64 QuadPart;\n");
		    &writeString("\t\t} LARGE_INTEGER;\n");
		}
		&writeString("\t\tLARGE_INTEGER $CYCLS_VALUE_START;\n");
		&writeString("\t\tLARGE_INTEGER $CYCLS_VALUE_END;\n");
	    }
	    if ($CLASS_REF->{'marshal'} eq ':lw-marshal' || $CLASS_REF->{'marshal'} eq ':std-marshal')
	    {
		&writeString("\t\tif (isMarshalNecessary)\n");
	    }
	    &writeString("\t\t{\n");
	    if ($RECORD_OVERHEAD)
	    {
		&writeClockCount($CYCLS_VALUE_START);
	    }
	    if ($CLASS_REF->{'marshal'} eq ':lw-marshal' || $CLASS_REF->{'marshal'} eq ':std-marshal')
	    {
		&writeString("\t\t\tCEComTransferApartment($THREAD_APARTMENT_VALUE, $PROXY_APARTMENT_VALUE);\n");
	    }

	    ## transfer incoming pointer
	    while (($VALUE, $INTERFACE_REF) = each(%INTERFACE_INS))
	    {
		my($INTERFACE_NAME) = $INTERFACE_REF->{'name'};
		my($IID) = &getIIDName($INTERFACE_NAME);
		&writeString("\t\t\tif (CESucceeded(hr) && $VALUE)\n");
		&writeString("\t\t\t{\n");
		if ($CLASS_REF->{'marshal'} eq ':lw-marshal' || $CLASS_REF->{'marshal'} eq ':std-marshal')
		{
		    &writeString("\t\t\t\thr = CEComTransferInterfacePointer($THREAD_APARTMENT_VALUE, $PROXY_APARTMENT_VALUE, $IID, $VALUE, reinterpret_cast<void**>(&proxy${VALUE}Out));\n");
		    &writeString("\t\t\t\tif (CESucceeded(hr) && proxy${VALUE}Out)\n");
		    &writeString("\t\t\t\t{\n");
		    &writeString("\t\t\t\t\t$VALUE = proxy${VALUE}Out;\n");
		    &writeString("\t\t\t\t}\n");
		}
		elsif ($CLASS_REF->{'marshal'} eq ':ipc-marshal')
		{
		    &writeString("\t\t\t\thr = CERpcGetStub(${CLASS}::toInstance($THIS)->getSession(), $IID, reinterpret_cast<UINT_PTR>($VALUE), reinterpret_cast<UINT_PTR*>(&stub${VALUE}Out));\n");

		    &writeString("\t\t\t\tif (CESucceeded(hr))\n");
		    &writeString("\t\t\t\t{\n");
		    &writeString("\t\t\t\t\thr = CE_S_OK;\n");
		    &writeString("\t\t\t\t\t$VALUE = reinterpret_cast<$INTERFACE_NAME*>(stub${VALUE}Out);\n");
		    &writeString("\t\t\t\t}\n");
#		    &writeString("else if (CE_SILK_ERR_NOTFOUND == err) hr = CE_S_OK; // successfully not found\n");
		}
		else
		{
		    die "unknown marshalling type $CLASS_REF->{'marshal'}\n";
		}

		&writeString("\t\t\t}\n");

	    }

	    if ($CLASS_REF->{'marshal'} eq ':lw-marshal' || $CLASS_REF->{'marshal'} eq ':std-marshal')
	    {
		&writeString("\t\t}\n");
	    }

	    ## call actual interface
	    my($STUB_FUNC_NAME) = &getStubFunctionName($CLASS_REF->{'marshal'}, $FUNCTION_NAME);
	    &writeString("\t\tif (CESucceeded(hr))\n");
	    &writeString("\t\t{\n");

	    if ($CLASS_REF->{'marshal'} eq ':lw-marshal')
	    {
		&writeString("\t\t\thr = $THIS_VALUE->getStub().$STUB_FUNC_NAME($VALUE_LIST);\n");
	    }
	    elsif ($CLASS_REF->{'marshal'} eq ':std-marshal')
	    {
		my($STUB_NAME) = &getStubClassName($CLASS_REF);
		my($METHOD_PARAM) = "$STUB_NAME\:\:" . &getMethodParamName($FUNCTION_NAME);
		&writeString("\t\t\t$METHOD_PARAM param = { $VALUE_LIST };\n");

		my($METHOD_ID) = "$STUB_NAME\:\:" . &getMethodIdName($FUNCTION_NAME);
		&writeString("\t\t\t// only c style cast is allowed by vs.net\n");
		&writeString("\t\t\thr = CEComPostApartmentMessage($PROXY_APARTMENT_VALUE, $THREAD_APARTMENT_VALUE, (UINT32)$METHOD_ID, &param, &$THIS_VALUE->getStub(), $STUB_NAME\:\:invoke_message_callback);\n");
		&writeString("\t\t\t\n");
	    }
	    elsif ($CLASS_REF->{'marshal'} eq ':ipc-marshal')
	    {
		## eek. to get current interface
		my(@PARAMS) = split(/\s*,\s*/, $PARAM_LIST);
		my($THIS_TYPE) = &getType($PARAMS[0]);
		chop($THIS_TYPE);
		$THIS_TYPE =~ s/\*$//;
		my($IFACE_PREFIX) = $THIS_TYPE . '_';

		my($INTERFACES_REF) = $CLASS_REF->{'interface'};
		my($IID) = &getIIDName($THIS_TYPE);
		my($STUB_NAME) = &getStubClassName($CLASS_REF);
		my($METHOD_PARAM) = "$STUB_NAME\:\:" . $IFACE_PREFIX . &getMethodParamName($FUNCTION_NAME);


		my(@SIZES, @VALUES, @TYPES);


		shift(@PARAMS); # skip this

		my(@GET_ADDITIONAL_BUFFER_SIZE);
		push(@GET_ADDITIONAL_BUFFER_SIZE, "\t\t\tUINT32 additionalBufferSize = 0;\n");

		while (@PARAMS)
		{
		    my($NAME_VALUE) = shift(@PARAMS);
		    my($PARAM_TYPE) = &getType($NAME_VALUE);
		    if ($PARAM_TYPE =~ "STR_BUF\s*")
		    {
			my($LEN_PARAM) = shift(@PARAMS);
			my($LEN) = &getValue($LEN_PARAM);
			my($BUF) = &getValue($NAME_VALUE);

			push(@SIZES, "sizeof(STR_LEN)");
			push(@VALUES, $LEN);
			push(@TYPES, 'STR_LEN');

			push(@SIZES, 'sizeof(STR_BUF*)');
			push(@VALUES, "&$BUF");
			push(@TYPES, 'STR_BUF*');

			push(@SIZES, "$LEN * sizeof(STR_BUF)");
			push(@VALUES, $BUF);
			push(@TYPES, $PARAM_TYPE);
		    }
		    elsif ($PARAM_TYPE =~ "DATA_BUF\s*")
		    {
			my($LEN_PARAM) = shift(@PARAMS);
			my($LEN) = &getValue($LEN_PARAM);
			my($BUF) = &getValue($NAME_VALUE);

			push(@SIZES, "sizeof(DATA_LEN)");
			push(@VALUES, $LEN);
			push(@TYPES, 'DATA_LEN');

			push(@SIZES, 'sizeof(DATA_BUF*)');
			push(@VALUES, "&$BUF");
			push(@TYPES, 'DATA_BUF*');

			push(@SIZES, "$LEN * sizeof(DATA_BUF)");
			push(@VALUES, $BUF);
			push(@TYPES, $PARAM_TYPE);
		    }
		    elsif ($PARAM_TYPE =~ "STRARRAY_BUF\s*")
		    {
			my($LEN_PARAM) = shift(@PARAMS);
			my($LEN) = &getValue($LEN_PARAM);
			my($BUF) = &getValue($NAME_VALUE);

			push(@SIZES, "sizeof(STRARRAY_LEN)");
			push(@VALUES, $LEN);
			push(@TYPES, 'STRARRAY_LEN');

			push(@SIZES, 'sizeof(STRARRAY_BUF*)');
			push(@VALUES, "&$BUF");
			push(@TYPES, 'STRARRAY_BUF*');

			push(@SIZES, "$LEN * sizeof(STRARRAY_BUF)");
			push(@VALUES, $BUF);
			push(@TYPES, $PARAM_TYPE);

			push(@GET_ADDITIONAL_BUFFER_SIZE, "\t\t\t{\n");
			push(@GET_ADDITIONAL_BUFFER_SIZE, "\t\t\t\tUINT32 index;\n");
			push(@GET_ADDITIONAL_BUFFER_SIZE, "\t\t\t\tfor(index=0; index < $LEN; index++) {\n");
			push(@GET_ADDITIONAL_BUFFER_SIZE, "\t\t\t\t\tadditionalBufferSize += $BUF\[index\].len * sizeof(STR_BUF);\n");
			push(@GET_ADDITIONAL_BUFFER_SIZE, "\t\t\t\t}\n");
			push(@GET_ADDITIONAL_BUFFER_SIZE, "\t\t\t}\n");
		    }
		    elsif ($PARAM_TYPE =~ "UINT32ARRAY_BUF\s*")
		    {
			my($LEN_PARAM) = shift(@PARAMS);
			my($LEN) = &getValue($LEN_PARAM);
			my($BUF) = &getValue($NAME_VALUE);

			push(@SIZES, "sizeof(UINT32ARRAY_LEN)");
			push(@VALUES, $LEN);
			push(@TYPES, 'UINT32ARRAY_LEN');

			push(@SIZES, 'sizeof(UINT32ARRAY_BUF*)');
			push(@VALUES, "&$BUF");
			push(@TYPES, 'UINT32ARRAY_BUF*');

			push(@SIZES, "$LEN * sizeof(UINT32ARRAY_BUF)");
			push(@VALUES, $BUF);
			push(@TYPES, $PARAM_TYPE);
		    }
		    else
		    {
			my($SIZE) = "sizeof($PARAM_TYPE)";
			my($VAL) = &getValue($NAME_VALUE);
			push(@SIZES, $SIZE);
			push(@VALUES, $VAL);
			push(@TYPES, $PARAM_TYPE);
		    }
		}
		my($SIZES_SIZE);
		my($SIZES_SIZE_IS_ZERO) = "false";
		$SIZES_SIZE = @SIZES;
		
		if ($SIZES_SIZE == 0)
		{
#		    push(@SIZES, '0');
		    $SIZES_SIZE_IS_ZERO = "true";
		    &writeString("\t\t\tUINT32 paramLength = 0;\n");
		    &writeString("\t\t\tUINT8* paramBuffer = 0;\n");
		    &writeString("\t\t\t{\n");
		}
		else
		{
		    my($STR);
		    foreach $STR (@GET_ADDITIONAL_BUFFER_SIZE)
		    {
			&writeString("$STR");
		    }

		    my($SIZE_TOTAL) = join('+', @SIZES);
		    &writeString("\t\t\tUINT32 paramLength = $SIZE_TOTAL + additionalBufferSize;\n");

		    if ($CLASS_REF->{'no-allocator'})
		    {
			&writeString("\t\t\tUINT8* paramBuffer = new UINT8[paramLength];\n");
		    }
		    else
		    {
			&writeString("\t\t\tUINT8* paramBuffer = CENEW_ARRAY(UINT8, paramLength);\n");
		    }

		    &writeString("\t\t\tif (paramBuffer)\n");
		    &writeString("\t\t\t{\n");
		    &writeString("\t\t\t\tint offset = 0;\n");
		}

		my($VALUE);
		my($STRARRAY_LEN);

		foreach $VALUE (@VALUES)
		{
		    my($SIZE) = shift(@SIZES);
		    my($TYPE) = shift(@TYPES);
		    if ($TYPE =~ "STR_BUF\s*" || $TYPE =~ "DATA_BUF\s*" || $TYPE =~ "UINT32ARRAY_BUF\s*")
		    {
			&writeString("\t\t\t\tmemcpy(&paramBuffer[offset], $VALUE, $SIZE);\n");
			&writeString("\t\t\t\toffset += $SIZE;\n");
		    }
		    elsif ($TYPE =~ "STRARRAY_BUF\s*")
		    {
			&writeString("\t\t\t\tmemcpy(&paramBuffer[offset], $VALUE, $SIZE);\n");
			&writeString("\t\t\t\toffset += $SIZE;\n");

			if ($VALUE !~ "&")
			{
			    &writeString("\t\t\t\t{\n");
			    &writeString("\t\t\t\t\tUINT32 index;\n");
			    &writeString("\t\t\t\t\tfor(index=0; index < $STRARRAY_LEN; index++) {\n");
			    &writeString("\t\t\t\t\t\tmemcpy(&paramBuffer[offset], $VALUE\[index\].buf, $VALUE\[index\].len * sizeof(STR_BUF));\n");
			    &writeString("\t\t\t\t\t\toffset += $VALUE\[index\].len * sizeof(STR_BUF);\n");
			    &writeString("\t\t\t\t\t}\n");
			    &writeString("\t\t\t\t}\n");
			}
		    }
		    else
		    {
			&writeString("\t\t\t\tmemcpy(&paramBuffer[offset], &$VALUE, $SIZE);\n");
			&writeString("\t\t\t\toffset += $SIZE;\n");
		    }

		    if ($TYPE =~ "STRARRAY_LEN")
		    {
			$STRARRAY_LEN = "$VALUE";
		    }
		}


		my($PROXY_NAME) = &getProxyClassName($CLASS_REF);
		my($METHOD_ID) = "$STUB_NAME\:\:" . $IFACE_PREFIX . &getMethodIdName($FUNCTION_NAME);
		&writeString("\t\t\t\t// only c style cast is allowed by vs.net\n");
		&writeString("\t\t\t\tCE_MAY_ALIAS_TYPE(UINT_PTR) stub = 0;\n");
		&writeString("\t\t\t\thr = CERpcGetStub(${CLASS}::toInstance($THIS)->getSession(), $IID, reinterpret_cast<UINT_PTR>($THIS), reinterpret_cast<UINT_PTR*>(&stub));\n");
		&writeString("\t\t\t\tif (CESucceeded(hr))\n");

		&writeString("\t\t\t\t{\n");
		&writeString("\t\t\t\t\t#if defined(MARSHALLER_DEBUG_PRINT)\n");
		&writeString("\t\t\t\t\tMARSHALLER_DEBUG_PRINT(\"$THIS_TYPE\:\:$FUNCTION_NAME(proxy:%p  stub:%p)\\n\", $THIS, (void*)stub);\n");
		&writeString("\t\t\t\t\t#endif //defined(MARSHALLER_DEBUG_PRINT)\n");
		&writeString("\t\t\t\t\thr = CERpcInvokeRemote(${CLASS}::toInstance($THIS)->getSession(), stub, $IID, (UINT32)$METHOD_ID, paramLength, paramBuffer);\n");
		&writeString("\t\t\t\t}\n");
		&writeString("\t\t\t\t\n");
		if ($SIZES_SIZE_IS_ZERO eq "false")
		{
		    if ($CLASS_REF->{'no-allocator'})
		    {
			&writeString("\t\t\t\tdelete [] paramBuffer;\n");
		    }
		    else
		    {
			&writeString("\t\t\t\tCEDELETE_ARRAY(paramBuffer);\n");
		    }
		}
		&writeString("\t\t\t}\n");
	    }

	    &writeString("\t\t}\n");

	    if ($CLASS_REF->{'marshal'} eq ':lw-marshal' || $CLASS_REF->{'marshal'} eq ':std-marshal')
	    {
		&writeString("\t\tif (isMarshalNecessary)\n");
		&writeString("\t\t{\n");
		## release transferred pointer
		while (($VALUE, $INTERFACE_REF) = each(%INTERFACE_INS))
		{
		    &writeString("\t\t\tif (proxy${VALUE}Out) reinterpret_cast<ICEUnknown*>(proxy${VALUE}Out)->_vtbl->_release(reinterpret_cast<ICEUnknown*>(proxy${VALUE}Out));\n");
		}
		&writeString("\t\t\t//ignore error\n");
		&writeString("\t\t\tCEComTransferApartment($PROXY_APARTMENT_VALUE, $THREAD_APARTMENT_VALUE);\n");
	    }
	    if ($RECORD_OVERHEAD)
	    {
		&writeClockCount($CYCLS_VALUE_END);
		if ($DBG_LEN gt 1)
		{
		    &writeString("\t\t\tUINT64 result = reinterpret_cast<UINT64&>($CYCLS_VALUE_END.QuadPart) - reinterpret_cast<UINT64&>($CYCLS_VALUE_START.QuadPart);\n");
		    &writeString("\t\t\t$PROXY_APARTMENT_VALUE->_vtbl->_log($PROXY_APARTMENT_VALUE, $DBG_TEXT, $DBG_LEN, result);\n");
		}
	    }

	    if ($FUNCTION_NAME eq 'createInstance')
	    {
		my($IID) = $VALUES[1];
		my($ORIG_OUT) = $VALUES[2];
		&writeString("\t\t\tif (CESucceeded(hr))\n");
		&writeString("\t\t\t{\n");
		&writeCreateHook($IDL_REF, "\t\t\t\t", $PROXY_APARTMENT_VALUE, $THREAD_APARTMENT_VALUE, $IID, $ORIG_OUT, 'void');
		&writeString("\t\t\t}\n");
	    }
	    else
	    {
		while (($VALUE, $INTERFACE_REF) = each(%INTERFACE_OUTS))
		{
		    my($IID) = &getIIDName($INTERFACE_REF->{'name'});
		    my($ORIG_OUT) = $VALUE;
		    &writeString("\t\t\tif (CESucceeded(hr) && $ORIG_OUT)\n");
		    &writeString("\t\t\t{\n");
		    &writeCreateHook($IDL_REF, "\t\t\t\t", $PROXY_APARTMENT_VALUE, $THREAD_APARTMENT_VALUE, $IID, $ORIG_OUT, $INTERFACE_REF->{'name'});
		    &writeString("\t\t\t}\n");
		}
	    }

	    if ($CLASS_REF->{'marshal'} eq ':lw-marshal' || $CLASS_REF->{'marshal'} eq ':std-marshal')
	    {
		&writeString("\t\t}\n");
	    }
	    &writeString("\t}\n");
	}
	else
	{
	    my($STR)     = "Str";
	    my($LEN)     = "Len";
	    my($BUF)     = "Buf";
	    my($ALC)     = "Alc";
	    my($X)       = "_x";
	    my($Y)       = "_y";
	    my($W)       = "_width";
	    my($H)       = "_height";
	    my($REF)     = "Ref";
	    my($DATALEN) = "_length";
	    my($DATABUF) = "_data";
	    my(@ALLOCATED_VARS);

	    if ($FND_TRANS_FLAG)
	    {
		if ($WRITE_SPX)
		{
		    ### variable
		    foreach my $CERECT_VAR (@CERECT_VARS)
		    {
			&writeString("\tCERect\* $CERECT_VAR = 0;\n");
		    }
		    foreach my $CERECTF_VAR (@CERECTF_VARS)
		    {
			&writeString("\tCERectF\* $CERECTF_VAR = 0;\n");
		    }
		    foreach my $ICEUSTRING_VAR (@ICEUSTRING_VARS)
		    {
			&writeString("\tCEComICEUStringRef $ICEUSTRING_VAR;\n");
		    }
		    foreach my $CERAWDATABUFFER_VAR (@CERAWDATABUFFER_VARS)
		    {
			&writeString("\tCERawDataBuffer\* $CERAWDATABUFFER_VAR = 0;\n");
		    }
		    foreach my $CEUSTRINGLIST_VAR (@CEUSTRINGLIST_VARS)
		    {
			&writeString("\tCEUStringList\* $CEUSTRINGLIST_VAR = 0;\n");
		    }
		    foreach my $CEUUINT32LIST_VAR (@CEUUINT32LIST_VARS)
		    {
			&writeString("\tCEUUINT32List\* $CEUUINT32LIST_VAR = 0;\n");
		    }

		    &writeString("\n");

		    &writeString("\thr = CE_S_OK;\n");

		    ### RECT_X rect_x, RECT_Y rect_y, RECT_W rect_width, RECT_H rect_height ---> CERect* rect
		    foreach my $CERECT_VAR (@CERECT_VARS)
		    {
			&writeString("\tif (CESucceeded(hr))\n");
			&writeString("\t{\n");
			&writeString("\t\t$CERECT_VAR = (CERect*)CEComMalloc(sizeof(CERect));\n");
			&writeString("\t\tif ($CERECT_VAR)\n");
			&writeString("\t\t{\n");
			&writeString("\t\t\t$CERECT_VAR->$X      = $CERECT_VAR$X;\n");
			&writeString("\t\t\t$CERECT_VAR->$Y      = $CERECT_VAR$Y;\n");
			&writeString("\t\t\t$CERECT_VAR->$W  = $CERECT_VAR$W;\n");
			&writeString("\t\t\t$CERECT_VAR->$H = $CERECT_VAR$H;\n");
			&writeString("\t\t\thr = CE_S_OK;\n");
			&writeString("\t\t}\n");
			&writeString("\t\telse\n");
			&writeString("\t\t{\n");
			&writeString("\t\t\thr = CE_SILK_ERR_MEMERR;\n");
			&writeString("\t\t}\n");
			&writeString("\t}\n");
			&writeString("\n");
		    }

		    ### RECTF_X rectf_x, RECTF_Y rectf_y, RECTF_W rectf_width, RECTF_H rectf_height ---> CERectF* rectf
		    foreach my $CERECTF_VAR (@CERECTF_VARS)
		    {
			&writeString("\tif (CESucceeded(hr))\n");
			&writeString("\t{\n");
			&writeString("\t\t$CERECTF_VAR = (CERectF*)CEComMalloc(sizeof(CERectF));\n");
			&writeString("\t\tif ($CERECTF_VAR)\n");
			&writeString("\t\t{\n");
			&writeString("\t\t\t$CERECTF_VAR->$X      = $CERECTF_VAR$X;\n");
			&writeString("\t\t\t$CERECTF_VAR->$Y      = $CERECTF_VAR$Y;\n");
			&writeString("\t\t\t$CERECTF_VAR->$W  = $CERECTF_VAR$W;\n");
			&writeString("\t\t\t$CERECTF_VAR->$H = $CERECTF_VAR$H;\n");
			&writeString("\t\t\thr = CE_S_OK;\n");
			&writeString("\t\t}\n");
			&writeString("\t\telse\n");
			&writeString("\t\t{\n");
			&writeString("\t\t\thr = CE_SILK_ERR_MEMERR;\n");
			&writeString("\t\t}\n");
			&writeString("\t}\n");
			&writeString("\n");
		    }

		    ### STR_BUF* strStr, STR_LEN strLen ---> struct ICEUString* str
		    foreach my $ICEUSTRING_VAR (@ICEUSTRING_VARS)
		    {
			&writeString("\tif (CESucceeded(hr))\n");
			&writeString("\t{\n");
			&writeString("\t\tif ($ICEUSTRING_VAR$STR == NULL)\n");
			&writeString("\t\t{\n");
			&writeString("\t\t\t$ICEUSTRING_VAR = NULL;\n");
			&writeString("\t\t}\n");
			&writeString("\t\telse\n");
			&writeString("\t\t{\n");
			&writeString("\t\t\thr = ICEUStringCreate(CEComStdClassID_CEUString, $ICEUSTRING_VAR);\n");
			&writeString("\t\t\tif (CESucceeded(hr))\n");
			&writeString("\t\t\t{\n");
			&writeString("\t\t\t\thr = $ICEUSTRING_VAR.initWithByteArray($ICEUSTRING_VAR$STR, eICEI18nEncoding_utf_8, $ICEUSTRING_VAR$LEN);\n");
			&writeString("\t\t\t}\n");
			&writeString("\t\t}\n");
			&writeString("\t}\n");
			&writeString("\n");
		    }

		    ### DATA_LEN dataLen, DATA_BUF dataBuf ---> CERawDataBuffer data
		    foreach my $CERAWDATABUFFER_VAR (@CERAWDATABUFFER_VARS)
		    {
			&writeString("\tif (CESucceeded(hr))\n");
			&writeString("\t{\n");
			&writeString("\t\t$CERAWDATABUFFER_VAR = (CERawDataBuffer*)CEComMalloc(sizeof(CERawDataBuffer));\n");
			&writeString("\t\tif ($CERAWDATABUFFER_VAR)\n");
			&writeString("\t\t{\n");
			&writeString("\t\t\t$CERAWDATABUFFER_VAR->$DATABUF = $CERAWDATABUFFER_VAR$BUF;\n");
			&writeString("\t\t\t$CERAWDATABUFFER_VAR->$DATALEN = $CERAWDATABUFFER_VAR$LEN;\n");
			&writeString("\t\t\thr = CE_S_OK;\n");
			&writeString("\t\t}\n");
			&writeString("\t\telse\n");
			&writeString("\t\t{\n");
			&writeString("\t\t\thr = CE_SILK_ERR_MEMERR;\n");
			&writeString("\t\t}\n");
			&writeString("\t}\n");
			&writeString("\n");
		    }

		    ### STRARRAY_BUF* arrayBuf, STRARRAY_LEN arrayLen ---> CEUStringList *list
		    foreach my $CEUSTRINGLIST_VAR (@CEUSTRINGLIST_VARS)
		    {
			&writeString("\tif (CESucceeded(hr))\n");
			&writeString("\t{\n");
			&writeString("\t\t$CEUSTRINGLIST_VAR = new CEUStringList();\n");
			&writeString("\t\tif ($CEUSTRINGLIST_VAR)\n");
			&writeString("\t\t{\n");
			&writeString("\t\t\thr = $CEUSTRINGLIST_VAR->initWithSTRARRAY($CEUSTRINGLIST_VAR$BUF, $CEUSTRINGLIST_VAR$LEN);\n");
			&writeString("\t\t}\n");
			&writeString("\t\telse\n");
			&writeString("\t\t{\n");
			&writeString("\t\t\thr = CE_SILK_ERR_MEMERR;\n");
			&writeString("\t\t}\n");
			&writeString("\t}\n");
			&writeString("\n");
		    }

		    ### UINT32ARRAY_BUF* arrayBuf, UINT32ARRAY_LEN arrayLen ---> CEUUINT32List *list
		    foreach my $CEUUINT32LIST_VAR (@CEUUINT32LIST_VARS)
		    {
			&writeString("\tif (CESucceeded(hr))\n");
			&writeString("\t{\n");
			&writeString("\t\t$CEUUINT32LIST_VAR = new CEUUINT32List();\n");
			&writeString("\t\tif ($CEUUINT32LIST_VAR)\n");
			&writeString("\t\t{\n");
			&writeString("\t\t\thr = $CEUUINT32LIST_VAR->initWithUINT32ARRAY($CEUUINT32LIST_VAR$BUF, $CEUUINT32LIST_VAR$LEN);\n");
			&writeString("\t\t}\n");
			&writeString("\t\telse\n");
			&writeString("\t\t{\n");
			&writeString("\t\t\thr = CE_SILK_ERR_MEMERR;\n");
			&writeString("\t\t}\n");
			&writeString("\t}\n");
			&writeString("\n");
		    }

		    ### adhoc
		    #&writeString("//$VALUE_LIST\n");
		    #&writeString("//$PARAM_LIST\n");
		    #&writeString("//$ORG_PARAM_LIST\n");
		    $VALUE_LIST = &replaceReinterpretCastTo(0, $VALUE_LIST, $PARAM_LIST);

		    ### original calls
		    &writeString("\tif (CESucceeded(hr))\n");
		    &writeString("\t{\n");

		    if ($FND_TRANS eq ':trans-to-fnd')
		    {
			&writeString("\t\t$CLASS* $THIS_VALUE = ${CLASS}::toInstance($THIS);\n");
			&writeString("\t\thr = $THIS_VALUE->object()->$FUNCTION_NAME($VALUE_LIST);\n");
		    }
		    elsif ($FND_TRANS eq ':trans-from-fnd')
		    {
			my($REF) = "Ref";
			my($POINTER_VALUE_NAME) = &replaceInterfaceValueNameToPointerValueNameForPublic($THIS);
			&writeString("\t\t//$CLASS* $THIS_VALUE = ${CLASS}::toInstance($THIS);\n");
			&writeString("\t\t//hr = $THIS_VALUE->$POINTER_VALUE_NAME$REF().$FUNCTION_NAME($VALUE_LIST);\n");
		    }
		    else
		    {
			die("Error: Wrong type $FND_TRANS, stopped");
		    }
		    &writeString("\t}\n");

		    ### free CEComMalloc
		    foreach my $CERECT_VAR (@CERECT_VARS)
		    {
			&writeString("\n");
			&writeString("\tif ($CERECT_VAR)\n");
			&writeString("\t{\n");
			&writeString("\t\tCEComFree($CERECT_VAR);\n");
			&writeString("\t\t$CERECT_VAR = 0;\n");
			&writeString("\t}\n");
		    }
		    foreach my $CERAWDATABUFFER_VAR (@CERAWDATABUFFER_VARS)
		    {
			&writeString("\n");
			&writeString("\tif ($CERAWDATABUFFER_VAR)\n");
			&writeString("\t{\n");
			&writeString("\t\tCEComFree($CERAWDATABUFFER_VAR);\n");
			&writeString("\t\t$CERAWDATABUFFER_VAR = 0;\n");
			&writeString("\t}\n");
		    }
		    foreach my $CEUSTRINGLIST_VAR (@CEUSTRINGLIST_VARS)
		    {
			&writeString("\n");
			&writeString("\tif ($CEUSTRINGLIST_VAR)\n");
			&writeString("\t{\n");
			&writeString("\t\t$CEUSTRINGLIST_VAR->disposeAllElements();\n");
			&writeString("\t\tdelete $CEUSTRINGLIST_VAR;\n");
			&writeString("\t\t$CEUSTRINGLIST_VAR = 0;\n");
			&writeString("\t}\n");
		    }
		    foreach my $CEUUINT32LIST_VAR (@CEUUINT32LIST_VARS)
		    {
			&writeString("\n");
			&writeString("\tif ($CEUUINT32LIST_VAR)\n");
			&writeString("\t{\n");
			&writeString("\t\t$CEUUINT32LIST_VAR->disposeAllElements();\n");
			&writeString("\t\tdelete $CEUUINT32LIST_VAR;\n");
			&writeString("\t\t$CEUUINT32LIST_VAR = 0;\n");
			&writeString("\t}\n");
		    }

		}
		else
		{
		    if ($FND_TRANS eq ':trans-to-fnd')
		    {
			&writeString("\t$CLASS* $THIS_VALUE = ${CLASS}::toInstance($THIS);\n");
			&writeString("\thr = $THIS_VALUE->object()->$FUNCTION_NAME($VALUE_LIST);\n");
		    }
		    elsif ($FND_TRANS eq ':trans-from-fnd')
		    {
			### CERect* rect ---> RECT_X rect_x, RECT_Y rect_y, RECT_W rect_width, RECT_H rect_height
			foreach my $CERECT_VAR (@CERECT_VARS)
			{
			    &writeString("\tCEASSERT($CERECT_VAR && \"$CERECT_VAR is 0\");\n");
			    &writeString("\tRECT_X $CERECT_VAR$X      = $CERECT_VAR->$X;\n");
			    &writeString("\tRECT_Y $CERECT_VAR$Y      = $CERECT_VAR->$Y;\n");
			    &writeString("\tRECT_W $CERECT_VAR$W  = $CERECT_VAR->$W;\n");
			    &writeString("\tRECT_H $CERECT_VAR$H = $CERECT_VAR->$H;\n");
			    &writeString("\n");
			}

			### CERectF* rectf ---> RECTF_X rectf_x, RECTF_Y rectf_y, RECTF_W rectf_width, RECTF_H rectf_height
			foreach my $CERECTF_VAR (@CERECTF_VARS)
			{
			    &writeString("\tCEASSERT($CERECTF_VAR && \"$CERECTF_VAR is 0\");\n");
			    &writeString("\tRECTF_X $CERECTF_VAR$X      = $CERECTF_VAR->$X;\n");
			    &writeString("\tRECTF_Y $CERECTF_VAR$Y      = $CERECTF_VAR->$Y;\n");
			    &writeString("\tRECTF_W $CERECTF_VAR$W  = $CERECTF_VAR->$W;\n");
			    &writeString("\tRECTF_H $CERECTF_VAR$H = $CERECTF_VAR->$H;\n");
			    &writeString("\n");
			}

			### STR_BUF* strStr, STR_LEN strLen ---> struct ICEUString* str
			foreach my $ICEUSTRING_VAR (@ICEUSTRING_VARS)
			{
			    &writeString("\tSTR_BUF*        $ICEUSTRING_VAR$STR = 0;\n");
			    &writeString("\tSTR_LEN         $ICEUSTRING_VAR$LEN = 0;\n");
			    &writeString("\tCEComAllocatorRec* $ICEUSTRING_VAR$ALC = 0;\n");
			    &writeString("\n");
			}

			### CERawDataBuffer data --> DATA_LEN dataLen, DATA_BUF dataBuf
			foreach my $CERAWDATABUFFER_VAR (@CERAWDATABUFFER_VARS)
			{
			    &writeString("\tCEASSERT($CERAWDATABUFFER_VAR && \"$CERAWDATABUFFER_VAR is 0\");\n");
			    &writeString("\tDATA_BUF* $CERAWDATABUFFER_VAR$BUF = $CERAWDATABUFFER_VAR->$DATABUF;\n");
			    &writeString("\tDATA_LEN $CERAWDATABUFFER_VAR$LEN = $CERAWDATABUFFER_VAR->$DATALEN;\n");
			    &writeString("\n");
			}

			### CEUStringList *array --> STRARRAY_BUF *arrayBuf, STRARRAY_LEN arrayLen
			foreach my $CEUSTRINGLIST_VAR (@CEUSTRINGLIST_VARS)
			{
			    &writeString("\tSTRARRAY_BUF* $CEUSTRINGLIST_VAR$BUF = $CEUSTRINGLIST_VAR->createArray();\n");
			    &writeString("\tSTRARRAY_LEN $CEUSTRINGLIST_VAR$LEN = $CEUSTRINGLIST_VAR->size();\n");
			    push(@ALLOCATED_VARS, $CEUSTRINGLIST_VAR);
			}

			### CEUUINT32List *array --> UINT32ARRAY_BUF *arrayBuf, UINT32ARRAY_LEN arrayLen
			foreach my $CEUUINT32LIST_VAR (@CEUUINT32LIST_VARS)
			{
			    &writeString("\tUINT32ARRAY_BUF* $CEUUINT32LIST_VAR$BUF = $CEUUINT32LIST_VAR->createArray();\n");
			    &writeString("\tUINT32ARRAY_LEN $CEUUINT32LIST_VAR$LEN = $CEUUINT32LIST_VAR->size();\n");
			    push(@ALLOCATED_VARS, $CEUUINT32LIST_VAR);
			}

			&writeString("\thr = CE_S_OK;\n");

			foreach my $ICEUSTRING_VAR (@ICEUSTRING_VARS)
			{
			    &writeString("\tif (CESucceeded(hr))\n");
			    &writeString("\t{\n");
			    &writeString("\t\tif ($ICEUSTRING_VAR == NULL)\n");
			    &writeString("\t\t{\n");
			    &writeString("\t\t\t$ICEUSTRING_VAR$STR = NULL;\n");
			    &writeString("\t\t}\n");
			    &writeString("\t\telse\n");
			    &writeString("\t\t{\n");
			    &writeString("\t\t\t$ICEUSTRING_VAR$ALC = CEComGetAllocatorRec();\n");
			    &writeString("\t\t\tif ($ICEUSTRING_VAR$ALC)\n");
			    &writeString("\t\t\t{\n");
			    &writeString("\t\t\t\tCEComICEUStringRef $ICEUSTRING_VAR$REF(const_cast<ICEUString*>($ICEUSTRING_VAR));\n");
			    &writeString("\t\t\t\thr = $ICEUSTRING_VAR$REF.getBytesWithAlloc(eICEI18nEncoding_utf_8, $ICEUSTRING_VAR$ALC, &$ICEUSTRING_VAR$STR, &$ICEUSTRING_VAR$LEN);\n");
			    &writeString("\t\t\t}\n");
			    &writeString("\t\t\telse\n");
			    &writeString("\t\t\t{\n");
			    &writeString("\t\t\t\thr = CE_SILK_ERR_MEMERR;\n");
			    &writeString("\t\t\t}\n");
			    &writeString("\t\t}\n");
			    &writeString("\t}\n");
			    &writeString("\n");
			}

			### adhoc
			#&writeString("//$VALUE_LIST\n");
			#&writeString("//$PARAM_LIST\n");
			#&writeString("//$ORG_PARAM_LIST\n");
			$VALUE_LIST = &writeQueryInterfaceTo($VALUE_LIST, $ORG_PARAM_LIST);

			### fnd trans calls
			&writeString("\tif (CESucceeded(hr))\n");
			&writeString("\t{\n");
			my($REF) = "Ref";
			my($SPX_INTERFACE_VALUE_NAME) = &replaceFunctionParamVariablePrefixForPublic($THIS);
			my($SPX_POINTER_VALUE_NAME) = &replaceInterfaceValueNameToPointerValueNameForPublic($SPX_INTERFACE_VALUE_NAME);

			if ($FUNCTION_NAME eq 'getSubstance')
			{
			    $SPX_POINTER_VALUE_NAME = 'unknown';
			    $FUNCTION_NAME = 'queryInterface';
			}
			&writeString("\t\t$CLASS* $THIS_VALUE = ${CLASS}::toInstance($THIS);\n");
			&writeString("\t\thr = $THIS_VALUE->$SPX_POINTER_VALUE_NAME$REF().$FUNCTION_NAME($VALUE_LIST);\n");
			&writeString("\t}\n");

			### free CEComAllocatorRec
			foreach my $ICEUSTRING_VAR (@ICEUSTRING_VARS)
			{
			    &writeString("\n");
			    &writeString("\tif ($ICEUSTRING_VAR$ALC && $ICEUSTRING_VAR$STR)\n");
			    &writeString("\t{\n");
			    &writeString("\t\t$ICEUSTRING_VAR$ALC->free($ICEUSTRING_VAR$ALC, $ICEUSTRING_VAR$STR);\n");
			    &writeString("\t\t$ICEUSTRING_VAR$ALC = 0;\n");
			    &writeString("\t\t$ICEUSTRING_VAR$STR = 0;\n");
			    &writeString("\t}\n");
			}
		    }
		    else
		    {
			die("Error: Wrong type $FND_TRANS, stopped");
		    }
		}
	    }
	    else
	    {
		&writePreFunctionCall($TOPLEVEL_INTERFACE_NAME);
		&writeString("\t//$TOPLEVEL_INTERFACE_NAME\n");
		&writeString("\t$CLASS* $THIS_VALUE = ${CLASS}::toInstance($THIS);\n");
		&writeString("\thr = $THIS_VALUE->object()->$FUNCTION_NAME($VALUE_LIST);\n");
		&writePostFunctionCall($TOPLEVEL_INTERFACE_NAME);
	    }

	    my($ALLOCATED_VAR);
	    foreach $ALLOCATED_VAR (@ALLOCATED_VARS)
	    {
		&writeString("\tif($ALLOCATED_VAR$BUF)  CEFREE($ALLOCATED_VAR$BUF);\n");
	    }
	}

	&writeString("\n");
	&writeString("\treturn hr;\n");
    }

    &writeString("}\n");
    &writeString("\n");

    #for JNI blocks.
    if ($CLASS_REF->{'java-to-c'})
    {
	if (($FUNCTION{'name'} ne "addRef") && ($FUNCTION{'name'} ne "release"))
	{
	    my($ICE_NAME) = 0;
	    my($idx) = index($PARAM_LIST, '*');

	    &writeString("inline $FUNCTION{'return'} $FUNC_IMPL_NAME");
	    $ICE_NAME = substr($PARAM_LIST, 0, $idx);
	    if ($FUNCTION{'name'} eq "queryInterface")
	    {
		&writeString("_$ICE_NAME");
	    }

	    my(@PARAMS) = split(/, /, $PARAM_LIST);
	    my($PARAM_CONV) = shift(@PARAMS);
	    my($P_TYPE) = &getType($PARAM_CONV);
	    my($P_VAL);
	    my($PARAM_CONV_WK);
	    my($PARAM_OL);

	    $P_TYPE = "struct ${P_TYPE}";
	    $P_VAL = &getValue($PARAM_CONV);
	    $PARAM_CONV_WK = &convertType($P_TYPE, $CLASS_REF);
	    $PARAM_CONV_WK = &convertNativeType($PARAM_CONV_WK);
	    $PARAM_OL = "${PARAM_CONV_WK} ${P_VAL}";

	    foreach $PARAM_CONV (@PARAMS)
	    {
		$P_TYPE = &getType($PARAM_CONV);
		$P_VAL = &getValue($PARAM_CONV);
		$PARAM_CONV_WK = &convertType($P_TYPE, $CLASS_REF);
		$PARAM_CONV_WK = &convertNativeType($PARAM_CONV_WK);
		$PARAM_OL = "${PARAM_OL}, ${PARAM_CONV_WK} ${P_VAL}";
	    }

	    &writeString("_JNI(JNIEnv *env, jobject o, $PARAM_OL)\n");
	    &writeString("{\n");
	    &writeString("\tCEHResult hr = CE_S_OK;\n");

	    my($WRAP_PARAM);
	    my(@PARAMS_WK) = split(/, /, $PARAM_LIST);
	    my(@VALUES_WK) = split(/, /, $VALUE_LIST);
	    my(@IFACE_VAR) = split(/ /, $PARAMS_WK[0]);
	    my($IFACE_IID) = &getIIDName($ICE_NAME);
	    my($IFACE_REF) = &getInterfaceWrapperName($ICE_NAME);
	    my($TABS);

	    &writeString("\tUINT_PTR ifaceOut = ApmProxyStub_proxyToInterface($IFACE_IID, static_cast<UINT_PTR>(${IFACE_VAR[1]}));\n");
	    &writeString("\tif (ifaceOut)\n\t{\n");
	    &writeString("\t\t${IFACE_REF} ${IFACE_VAR[1]}Ref = reinterpret_cast<$ICE_NAME*>(ifaceOut);\n");
	    &writeString("\t\tif ($IFACE_VAR[1]". "Ref)\n\t\t{\n");
	    if ($VALUE_LIST ne "")
	    {
		my($PARAM_CONV);
		my($VALUE_TMP) = shift(@VALUES_WK);
		shift(@PARAMS_WK);
		foreach $PARAM_CONV (@PARAMS_WK)
		{
		    my($PARAM_CONV_WK);
		    my($JAVA_INTERFACE_FLAG);
		    $PARAM_CONV_WK = &convertType($PARAM_CONV, $CLASS_REF);
		    $JAVA_INTERFACE_FLAG = &isJavaInterface($PARAM_CONV_WK);
		    if($JAVA_INTERFACE_FLAG)
		    {
			my(@PARAMS) = split(/ /, $PARAM_CONV);
			my $IFACE_NAME = $PARAMS[1];
			my($IFACE_IID_NAME);
			&writeString("\t\t\tjobject stubOut = NULL;\n");
			$IFACE_NAME =~ s/\*//;
			$IFACE_IID_NAME = &getIIDName($IFACE_NAME);
			&writeString("\t\t\tApmGetInternalJNI(${IFACE_IID_NAME}, ${VALUE_TMP}, &stubOut);\n");
			&writeString("\t\t\tif(stubOut)\n");
			&writeString("\t\t\t{\n");
			$WRAP_PARAM = "$WRAP_PARAM, (${PARAMS[1]})stubOut";
			$TABS = "\t";
		    }
		    elsif ($PARAM_CONV_WK eq "String")
		    {
			$WRAP_PARAM = $WRAP_PARAM . ", " . "($VALUE_TMP == NULL ? NULL : (STR_BUF *)(env->GetStringUTFChars ((jstring)$VALUE_TMP, NULL)))";
		    }
		    elsif ($PARAM_CONV_WK eq "byte[]")
		    {
			my($CAST_PARAM) = &getType(${PARAM_CONV});
			$CAST_PARAM =~ s/ $//;
			$WRAP_PARAM = $WRAP_PARAM . ", " . "($VALUE_TMP == NULL ? NULL : (${CAST_PARAM})(env->GetPrimitiveArrayCritical ($VALUE_TMP, NULL)))";
		    }
		    else
		    {
			my($CAST_PARAM) = &getType(${PARAM_CONV});
			$CAST_PARAM =~ s/ $//;
			$WRAP_PARAM = "$WRAP_PARAM, (${CAST_PARAM})${VALUE_TMP}";
		    }
		    $VALUE_TMP = shift(@VALUES_WK);
		}
		$WRAP_PARAM =~ s/,//;
	    }
	    my($FUNCTION_NAME) = $FUNCTION{'name'};
	    my($ERR_OPER_FAILED) = "hr = CE_SILK_ERR_OPERATION_FAILED\;";
	    &writeString("${TABS}\t\t\t$IFACE_VAR[1]". "Ref.$FUNCTION_NAME" . "($WRAP_PARAM);\n");
	    if(${TABS} ne "")
	    {
		&writeString("\t\t\t}\n");
		&writeString("\t\t\telse\n");
		&writeString("\t\t\t{\n");
		&writeString("\t\t\t\t$ERR_OPER_FAILED\n");
		&writeString("\t\t\t}\n");
	    }
	    &writeString("\t\t}\n");
	    &writeString("\t\telse\n\t\t{\n\t\t\t$ERR_OPER_FAILED\n\t\t}\n\t}\n");
	    &writeString("\telse\n\t{\n\t\t$ERR_OPER_FAILED\n\t}\n");
	    if ($FUNCTION{'return'} ne "void")
	    {
		&writeString("\treturn hr;\n");
	    }
	    &writeString("}\n");
	    &writeString("\n");
	}
    }
}

sub writeCreateHook
{
    ### void** should be handled specially. queryInterface and createInstance
    my($IDL_REF, $INDENT, $FROM_APARTMENT_VALUE, $TO_APARTMENT_VALUE, $IID_VALUE, $ORIGIN_OUT, $TYPE, $SKIP_RELEASE) = @_;

    &writeString($INDENT . "CE_MAY_ALIAS_TYPE($TYPE)* proxyOut = 0;\n");
    &writeString($INDENT . "hr = CEComTransferInterfacePointer($FROM_APARTMENT_VALUE, $TO_APARTMENT_VALUE ,$IID_VALUE, *$ORIGIN_OUT, reinterpret_cast<void**>(&proxyOut));\n");

    &writeString($INDENT . "if (CESucceeded(hr))\n");
    &writeString($INDENT . "{\n");
    &writeString($INDENT . "\tif (proxyOut)\n");
    &writeString($INDENT . "\t{\n");
    if (!$SKIP_RELEASE)
    {
	if (!&getInterfaceHash($IDL_REF, $TYPE) && $TYPE eq 'void')
	{
	    &writeString($INDENT . "\t\tif (*$ORIGIN_OUT) { static_cast<ICEUnknown*>(*$ORIGIN_OUT)->_vtbl->_release(static_cast<ICEUnknown*>(*$ORIGIN_OUT)); }\n");
	}
	else
	{
	    &writeString($INDENT . "\t\tif (*$ORIGIN_OUT) { (*$ORIGIN_OUT)->_vtbl->_release(*$ORIGIN_OUT); }\n");
	}
    }
    else
    {
	## eek.
	&writeString($INDENT . "\t\t//! eek!. don't release here. release later temporarliy!!\n");
    }
    &writeString($INDENT . "\t\t*$ORIGIN_OUT = reinterpret_cast<$TYPE*>(proxyOut);\n");
    &writeString($INDENT . "\t}\n");
    &writeString($INDENT . "}\n");
    &writeString($INDENT . "else\n");
    &writeString($INDENT . "{\n");
    &writeString($INDENT . "\t*$ORIGIN_OUT = 0;\n");
    &writeString($INDENT . "}\n");
}

sub writeFunctionComment
{
    my($COMMENT, $VALUE) = @_;

    if ($COMMENT)
    {
	my(@LINES) = split(/\n/, $COMMENT);
	if (@LINES)
	{
	    my($LINE);
	    while (@LINES)
	    {
		$LINE = shift(@LINES);

		if ($LINE =~ /^(\s*\*\s+)(\\param)\[\w+\](\s+)\w+(\s*).+$/ || $LINE =~ /^(\s*\*\s+)(\\param)(\s+)\w+(\s+).+$/)
		{
		    &writeString("$1\\param\[in\]$3$VALUE$4Specifies interface pointer\n");
		    &writeString("$LINE\n");
		    last;
		}
		elsif ($LINE =~ /^(\s*\*)(\s+)(\\return)\s+.+$/)
		{
		    &writeString("$1$2\\param\[in\]\t$VALUE\tSpecifies interface pointer\n");
		    &writeString("$1\n");
		    &writeString("$LINE\n");
		    last;
		}
		else
		{
		    &writeString("$LINE\n");
		}
	    }

	    &writeString(join("\n", @LINES) . "\n");
	}
    }
}

sub isCallingWebKit()
{
    my($INTERFACE_NAME) = @_;

    if ($INTERFACE_NAME =~ /ICEWebKit/ || 
        $INTERFACE_NAME =~ /ICEHtmlFocusNavigatorPeer/)
    {
        return 1;
    }
    return 0;
}

sub isCalledByWebKit()
{
    my($INTERFACE_NAME) = @_;

    if ($INTERFACE_NAME =~ /ICEHtmlWebKit/ ||
        $INTERFACE_NAME =~ /ICEPluginModuleLoader/ ||
        $INTERFACE_NAME =~ /ICEHtmlWebRequestMediator/ ||
        $INTERFACE_NAME =~ /ICEHtmlFocusNavigator/ ||
	$INTERFACE_NAME =~ /ICENetworkEventListener/)
    {
        return 1;
    }
    return 0;
}

sub writePreFunctionCall()
{
    my($INTERFACE_NAME) = @_;
    if (&isCallingWebKit($INTERFACE_NAME))
    {
        &writeString("\tCEComGlobalUnlock();\n")
    }
    elsif (&isCalledByWebKit($INTERFACE_NAME))
    {
        &writeString("\tCEComGlobalLock();\n")
    }
}

sub writePostFunctionCall()
{
    my($INTERFACE_NAME) = @_;
    if (&isCallingWebKit($INTERFACE_NAME))
    {
        &writeString("\tCEComGlobalLock();\n")
    }
    elsif (&isCalledByWebKit($INTERFACE_NAME))
    {
        &writeString("\tCEComGlobalUnlock();\n")
    }
}

sub debugMode
{
    my($FLAG) = @_;

    $DEBUG = $FLAG;
}

sub writeString
{
    my($STRING) = @_;
    &writeStringDispatch($STRING,'FPW');
}
sub writeStringDispatch
{
    my($STRING,$FILEHANDLE) = @_;

    ### debug info
    my($FRAME_NUM) = 0;
    my($DEBUG_INFO) = "\t\t\t\t\// ";
    while ($FRAME_NUM lt $DEBUG)
    {
	my($NEXT_FRAME) = $FRAME_NUM + 1;
	my($PKG0, $FILE0, $LINE0, $SUBROUTINE0) = caller($FRAME_NUM);
	my($PKG1, $FILE1, $LINE1, $SUBROUTINE1) = caller($NEXT_FRAME);
	$DEBUG_INFO .= ' <=== ' . $SUBROUTINE1 . ' at ' . $FILE0 . ':' . $LINE0 . ' in ' . $PKG0;
	$FRAME_NUM = $NEXT_FRAME;
    }
    $DEBUG_INFO .= "\n";

    if ($DEBUG)
    {
	$STRING =~ s/\n/$DEBUG_INFO/g;
    }
    
    ### newline code:
    ###   Win32 active Perl -> No care
    ###   Win32 cygwin perl -> change to CRLF
    ###   Linux perl        -> change to LF
    if (&getOSName() ne 'MSWin32')
    {
	$STRING =~ s/\r//g;
	if (&getOSName() ne 'linux')
	{
	    $STRING =~ s/\n/\r\n/g;
	}
    }

    if ($FILEHANDLE eq "FPW")
    {
	if (fileno(FPW))
	{
	    print(FPW $STRING);
	}
	else
	{
	    print($STRING);
	}
    }
    elsif ($FILEHANDLE eq "FPC")#output .java-file(class)
    {
	if (fileno(FPC))
	{
	    print(FPC $STRING);
	}
	else
	{
	    print($STRING);
	}
    }
    elsif ($FILEHANDLE eq "FPI")#output .java-file(interface)
    {
	if (fileno(FPI))
	{
	    print(FPI $STRING);
	}
	else
	{
	    print($STRING);
	}
    }
    else
    {
	print($STRING);
    }
}

sub writeDepMakeFileFromIDLWin
{
    my($IDL_REF, $IDL_FULLPATH, $OUT_FILE, $CWDIR, $RULE_FILE_OUTDIR) = @_;

    my(@ABSTARGETES) = mswinutil::toAbsPath($OUT_FILE, $CWDIR);

    my(@DEPENDS);
    my($t);

    foreach $t (keys(%{$IDL_REF}))
    {
	push(@DEPENDS, $IDL_REF->{$t}->{'name'});
    }

    foreach $t (keys(%INC))
    {
	push(@DEPENDS, $INC{$t});
    }

    push(@DEPENDS, &getProgramName());


    ### delete path for databese.
    my($IDL_FILE) = &deletePath($IDL_FULLPATH);
    my($P_FILE) = &getFileName($OUT_FILE, 0, 'Pi.mak');

    $P_FILE = $RULE_FILE_OUTDIR . "/" . &deletePath($P_FILE);

    &mswinutil::writeDepMakeFile($CWDIR, $RULE_FILE_OUTDIR,
				 $P_FILE,
				 \@ABSTARGETES, \@DEPENDS);
}

sub writeRuleFileFromIDL
{
    my($IDL_REF, $IDL_FULLPATH, $OUT_FILE) = @_;

    ### delete path for databese.
    my($IDL_FILE) = &deletePath($IDL_FULLPATH);
    my($P_FILE) = &getFileName($OUT_FILE, 0, 'Pi');
    if ($P_FILE)
    {
	open(FPW, ">$P_FILE") || die("Error: Can not open file $P_FILE as output\n");
    }

    &writeString("$OUT_FILE: \\\n");
    
    my($FILE_NAME);
    foreach $FILE_NAME (keys(%{$IDL_REF}))
    {
	my($IDL_FILE_REF) = $IDL_REF->{$FILE_NAME};
	&writeString("  $IDL_FILE_REF->{'name'} \\\n");
    }

    my($INC_FILE);
    foreach $INC_FILE (keys(%INC))
    {
	&writeString("  $INC{$INC_FILE} \\\n");
    }

    my($PROGRAM) = &getProgramName();
    &writeString("  $PROGRAM\n");

    close(FPW);
}

########################################################################################################
### others
########################################################################################################

sub findIncludingFile
{
    my($INCLUDE, @INCLUDEPATHS) = @_;
    unshift(@INCLUDEPATHS, '.');

    my($INCLUDEPATH);
    foreach $INCLUDEPATH (@INCLUDEPATHS)
    {
	my($FILE) = $INCLUDEPATH . '/'. $INCLUDE;

	if (-f $FILE)
	{
	    return $FILE;
	}
    }
}

sub getSummary
{
    my($COMMENT) = @_;
    if ($COMMENT)
    {
	my(@LINES) = split(/\n/, $COMMENT);
	if (@LINES)
	{
	    my($LINE);
	    while (@LINES)
	    {
		$LINE = shift(@LINES);
		if ($LINE =~ /^(\s*\*\s+)(<b>Summary:<\/b>)\s*$/)
		{
		    $LINE = shift(@LINES);
		    if ($LINE =~ /^(\s*\*\s+)([^<]+)<br>\s*$/)
		    {
			return $2;
		    }
		}
	    }
	}
    }
}

sub getValueTypeMapFromAllParameter
{
    my($PARAM_LIST) = @_;
    my(@PARAMS) = split(/, /, $PARAM_LIST);
    my(%TYPE_VALUE) = map { &getValue($_) => &getType($_) } @PARAMS;

    return %TYPE_VALUE;
}

sub getType
{
    my($PARAM) = @_;
    my($VALUE) = &getValue($PARAM);

    $PARAM =~ s/$VALUE//e;

    return $PARAM;
}

sub getValuesFromAllParameter
{
    my($PARAM_LIST) = @_;
    my(@PARAMS) = split(/, /, $PARAM_LIST);
    my($PARAM);
    my(@VALUES);
    foreach $PARAM (@PARAMS)
    {
	push(@VALUES, &getValue($PARAM));
    }

    return @VALUES;
}

sub getValue
{
    my($PARAM) = @_;
    my(@WORDS) = split(/[ \*]/, $PARAM);
    return $WORDS[$#WORDS];
}

sub getPrivateInterfaceValueName
{
    my($INTERFACE) = @_;
    return '_' . &getInterfaceValueName($INTERFACE);
}

sub getInterfaceValueName
{
    my($TYPE) = @_;
    my($VALUE) = "";
    if ($TYPE =~ /(ICEHtml)(\w+)/)
    {
	$VALUE = 'i' . $2;
    }
    elsif ($TYPE =~ /(ICE)(\w+)/)
    {
	$VALUE = 'i' . $2;
    }
    else
    {
	die("Error: Illegal interface name $TYPE.\n");
    }

    return $VALUE;
}

sub getPointerValueName
{
    my($TYPE) = @_;
    my($VALUE) = "";

#     if ($TYPE =~ /(ICEHtml|CEHtml)(\w+)/)
#     {
# 	$VALUE = $2;
#     }
#     elsif ($TYPE =~ /(ICE|CE)(\w+)/)
#     {
# 	$VALUE = $2;
#     }
#     elsif ($TYPE =~ /(Ph)(\w+)/)
#     {
# 	# For paf files.
# 	$VALUE = $2;
#     }
#     elsif ($TYPE =~ /(URL)/)
#     {
# 	$VALUE = $TYPE;
#     }
#     else
#     {
# 	die("Error: Illegal interface or class name $TYPE.\n");
#     }

     if ($TYPE =~ /(ICEHtml|CEHtml)(\w+)/)
     {
 	$VALUE = $2;
     }
     elsif ($TYPE =~ /(ICE|CE)(\w+)/)
     {
 	$VALUE = $2;
     }
     else
     {
 	$VALUE = $TYPE;
     }

    if ($VALUE =~ /(\w)(\w*)/)
    {
	my($HEAD) = $1;
	$HEAD =~ tr/A-Z/a-z/;
	$VALUE = $HEAD . $2;
    }

    return $VALUE;
}

sub getVtableName
{
    my($INTERFACE) = @_;
    return $INTERFACE . '_vtbl';
}

sub getIIDName
{
    my($INTERFACE) = @_;
    return 'CEComIID_' . $INTERFACE;
}

sub getClsidName
{
    my($CLASS) = @_;
    return 'CEComClassID_' . $CLASS;
}

sub getInterfaceWrapperName
{
    my($INTERFACE) = @_;
    return 'CECom' . $INTERFACE . 'Ref';
}

sub getImplementationMethodVtblName
{
    my($CLASS_NAME, $ORIG_NAME) = @_;
    return '__' . &getPointerValueName($CLASS_NAME) . '_' . $ORIG_NAME;
}

sub getStubClassName
{
    my($CLASS_REF) = @_;
    my($MARSHAL) = $CLASS_REF->{'marshal'};
    my($STUB_NAME);
    if ($MARSHAL)
    {
	$STUB_NAME = $CLASS_REF->{'name'} . 'Stub';
    }
    else
    {
	die("Error: marshal type should be specified.\n");
    }

    return $STUB_NAME;
}

sub getJNIClassName
{
    my($CLASS_REF) = @_;
    my($MARSHAL) = $CLASS_REF->{'marshal'};
    my($STUB_NAME);
    if ($MARSHAL)
    {
	$STUB_NAME = $CLASS_REF->{'name'};
    }
    else
    {
	die("Error: marshal type should be specified.\n");
    }

    return $STUB_NAME;
}


sub getStubFunctionName
{
    my($MARSHAL, $FUNCTION_NAME) = @_;
    my($NAME) = $FUNCTION_NAME;
    if ($MARSHAL eq ':lw-marshal')
    {
	$NAME .= 'Lw';
    }
    elsif ($MARSHAL eq ':std-marshal')
    {
	$NAME .= 'Std';
    }
    elsif ($MARSHAL eq ':ipc-marshal')
    {
	$NAME .= 'Prc';
    }
    else
    {
	die "Error: invalid marshal type: $MARSHAL";
    }
    return $NAME;
}

sub getStubOriginalInterfaceName
{
    my($NAME) = @_;
    my($ORIG_NAME) = '_orig' . $NAME;
    return $ORIG_NAME;
}

sub getProxyClassName
{
    my($CLASS_REF) = @_;
    my($MARSHAL) = $CLASS_REF->{'marshal'};
    my($PROXY_NAME);
    if ($MARSHAL eq ':lw-marshal' || $MARSHAL eq ':std-marshal')
    {
	#$PROXY_NAME = $CLASS_REF->{'name'} . 'Proxy';
	$PROXY_NAME = $CLASS_REF->{'name'};
    }
    elsif ($MARSHAL eq ':ipc-marshal')
    {
	$PROXY_NAME = $CLASS_REF->{'name'} . 'Proxy';
    }
    else
    {
	die("Error: marshal type should be specified.\n");
    }

    return $PROXY_NAME;
}

sub getStubClassName
{
    my($CLASS_REF) = @_;
    my($MARSHAL) = $CLASS_REF->{'marshal'};
    my($STUB_NAME);
    if ($MARSHAL)
    {
	$STUB_NAME = $CLASS_REF->{'name'} . 'Stub';
    }

    return $STUB_NAME;
}

sub getIpcStubName
{
    my($INTERFACE) = @_;

    my($STUB_NAME) = &getStubClassName();
    return $STUB_NAME;
}

sub getMethodIdName
{
    my($FUNCTION_NAME) = @_;
    return 'MethodId_' . $FUNCTION_NAME;
}

sub getMethodParamName
{
    my($FUNCTION_NAME) = @_;
    return $FUNCTION_NAME . '_param';
}

sub getDebugText
{
    my($CLASS_NAME, $FUNCTION_NAME) = @_;
    my($TEXT) = &getBaseDebugText($CLASS_NAME, $FUNCTION_NAME);
    return 'const_cast<char*>("' . $TEXT . '\n")';
}

sub getDebugLen
{
    my($CLASS_NAME, $FUNCTION_NAME) = @_;
    my($TEXT) = &getBaseDebugText($CLASS_NAME, $FUNCTION_NAME);
    return length($TEXT) + 1;
}

sub getBaseDebugText
{
    my($CLASS_NAME, $FUNCTION_NAME) = @_;

    $CLASS_NAME =~ s/LightWeightMarshaller//g;
    $CLASS_NAME =~ s/StandardMarshaller//g;

    if ($CLASS_NAME =~ /CEFS/ || $CLASS_NAME =~ /CEUString/)
    {
	return '';
    }
    else
    {
	return $CLASS_NAME . '::' . $FUNCTION_NAME;
    }
}

sub getFunctionsIncludingInherit
{
    my($IDL_REF, $INTERFACE_REF) = @_;
    my(@INTERFACES_ORDERED_FROM_ROOT);
    while ($INTERFACE_REF)
    {
	unshift(@INTERFACES_ORDERED_FROM_ROOT, $INTERFACE_REF);
	if (&getInterfaceHash($IDL_REF, $INTERFACE_REF->{'inheritance'}))
	{
	    my(%INTERFACE_HASH) = &getInterfaceHash($IDL_REF, $INTERFACE_REF->{'inheritance'});
	    $INTERFACE_REF = \%INTERFACE_HASH;
	}
	else
	{
	    $INTERFACE_REF = 0;
	}
    }

    my(@FUNCTIONS);

    $INTERFACE_REF = shift(@INTERFACES_ORDERED_FROM_ROOT);
    while ($INTERFACE_REF)
    {
	my($FUNCTION_REF);
	foreach $FUNCTION_REF (@{$INTERFACE_REF->{'function'}})
	{
	    push(@FUNCTIONS, $FUNCTION_REF);
	}
	$INTERFACE_REF = shift(@INTERFACES_ORDERED_FROM_ROOT);
    }

    return @FUNCTIONS;
}

sub isChildInterfaceOf
{
    my($IDL_REF, $CHILD_INTERFACE_NAME, $PARENT_INTERFACE_NAME) = @_;

    my(%INTERFACE_HASH) = &getInterfaceHash($IDL_REF, $CHILD_INTERFACE_NAME);
    my($INTERFACE_REF) = \%INTERFACE_HASH;
    while ($INTERFACE_REF && $INTERFACE_REF->{'name'} ne 'ICEUnknown')
    {
	if ($INTERFACE_REF->{'name'} eq $PARENT_INTERFACE_NAME)
	{
	    return 1;
	}
	else
	{
	    if (&getInterfaceHash($IDL_REF, $INTERFACE_REF->{'inheritance'}))
	    {
		my(%INHERITANCE_HASH) = &getInterfaceHash($IDL_REF, $INTERFACE_REF->{'inheritance'});
		$INTERFACE_REF = \%INHERITANCE_HASH;
	    }
	    else
	    {
		$INTERFACE_REF = 0;
	    }
	}
    }
    return 0;
}

sub getInterfaceHash
{
    my($IDL_REF, $NAME) = @_;
    my($IDL_FILE);

    foreach $IDL_FILE (keys(%$IDL_REF))
    {
	my($INTERFACE_REF);
	my($INTERFACES_REF) = $IDL_REF->{$IDL_FILE}->{'interface'};
	foreach $INTERFACE_REF (@$INTERFACES_REF)
	{
	    my(%INTERFACE) = %$INTERFACE_REF;
	    if ($INTERFACE{'name'} eq $NAME)
	    {
		return %INTERFACE;
	    }
	}
    }

    return 0;
}

sub getFileName
{
    my($FILE, $NOW, $NEW, $APPENDAGE) = @_;

    my(@STRINGS) = split(/\./, $FILE);
    if ($NOW && $STRINGS[$#STRINGS] ne $NOW)
    {
	die("Error: $FILE: Suffix of $FILE is not '$NOW'.\n");
    }

    $STRINGS[$#STRINGS] = $NEW;
    $STRINGS[$#STRINGS - 1] .= $APPENDAGE;
    return join('.', @STRINGS);
}

sub uniq
{
    my(@LIST) = @_;
    my(@UNIQ_LIST);

    my($VALUE);
    foreach $VALUE (@LIST)
    {
	my($i);
	for ($i = 0; $i < $#UNIQ_LIST + 1; $i++)
	{
	    if ($UNIQ_LIST[$i] eq $VALUE)
	    {
		last;
	    }
	}

	if ($i >= $#UNIQ_LIST + 1)
	{
	    push(@UNIQ_LIST, $VALUE);
	}
    }

    return @UNIQ_LIST;
}

sub deletePath
{
    my($FULLPATH) = @_;
    my(@STRINGS) = split(/[\/\\]/, $FULLPATH);

    return pop(@STRINGS);
}

sub getPath
{
    my($FULLPATH) = @_;
    my(@STRINGS) = split(/[\/\\]/, $FULLPATH);

    pop(@STRINGS);
    return join('/', @STRINGS);

}

sub getMD5Last32
{
    my($NAME) = @_;
    my($MD5);

    if (&getOSName() eq 'MSWin32')
    {
	my(@STRINGS) = split(/[\/\\]/, &getPerlName());
	$STRINGS[$#STRINGS] = 'md5.exe';
	my($COMMAND) = '"' . join('\\', @STRINGS) . '"' . " -s $NAME";
	my($RESULT) = `$COMMAND`;
	if ($RESULT =~ /([^=]+)=\s*(\w+)\s*$/)
	{
	    $MD5 = $2;
	}
    }
    else
    {
	my(@STRINGS) = split(/[\/\\]/, &getProgramName());
	$STRINGS[$#STRINGS] = 'md5.pl';
	my($COMMAND) = 'perl ' . join('/', @STRINGS) . " $NAME";
	$MD5 = `$COMMAND`;
    }

    if ($MD5 =~ /^(\w{24})(\w{8})$/)
    {
	$MD5 = "0x$2";
    }
    else
    {
	die("Fatal: Cannot generate MD5.\n");
    }

    return $MD5;
}

sub getGccInlineOption
{
    # use definition by sstype.h
    return 'FORCEINLINE_WITHOUT_DEBUG';
}

sub writeClockCount
{
    my($VALUE) = @_;

    if (&getOSName() eq 'MSWin32')
    {
	&writeString("__asm {\n");
	&writeString("\tcpuid\n");
	&writeString("\trdtsc\n");
	&writeString("\tmov $VALUE.LowPart, eax\n");
	&writeString("\tmov $VALUE.HighPart, edx\n");
	&writeString("}\n");
    }
    elsif (&getOSName() eq 'linux')
    {
	&writeString("$VALUE.QuadPart = CEComGetSystemTick();\n");

#  	&writeString("asm volatile(\n");
# 	&writeString("\"rdtsc\\n\"\n");
# 	&writeString("\"movl %%eax, %0\\n\"\n");
#  	&writeString("\"movl %%edx, %1\\n\"\n");
# 	&writeString(":\n");
# 	&writeString(": \"m\"($VALUE.LowPart), \"m\"($VALUE.HighPart)\n");
# 	&writeString(": \"ax\", \"dx\"\n");
#  	&writeString(");\n");

#  	&writeString("__asm__ volatile(\n");
# 	&writeString("\"0:\n\t\"\n");
# 	&writeString("\"mftbu %0\n\t\"\n");
# 	&writeString("\"mftb %1\n\t\"\n");
# 	&writeString("\"mftbu %2\n\t\"\n");
# 	&writeString("\"cmpw %0, %2\n\t\"\n");
# 	&writeString("\"bne- 0b\"\n");
# 	&writeString(": \"=r\"($VALUE.HighPart), \"=r\"($VALUE.LowPart), \"=r\"(tmp)\n");
# 	&writeString(": /* nope */\n");
# 	&writeString(": \"cc\");\n");
    }
    else
    {
	die "unknown os\n";
    }
}

sub writeQueryInterfaceTo
{
    my($VALUE_LIST, $PARAM_LIST) = @_;

    my(@VALUES) = split(/[\s,]+/, $VALUE_LIST);

    my($PARAM);
    foreach $PARAM (split(/[,]/, $PARAM_LIST))
    {
	my(@TOKENS) = split(/[\s\*\&]/, $PARAM);
	my($VALUE) = pop(@TOKENS);
	my($TYPE) = grep(/ICEApx|ICEMss/, @TOKENS);
	if ($TYPE)
	{
	    $TYPE = replaceInterfacePrefixForPublic($TYPE);
	    for (my($i) = 0; $i < $#VALUES + 1; $i++)
	    {
		if ($VALUES[$i] eq $VALUE)
		{
		    &writeString("\tCECom${TYPE}Ref ${VALUE}Ref = 0;\n");
		    &writeString("\tif (CESucceeded(hr) && $VALUE)\n");
		    &writeString("\t{\n");
		    &writeString("\t\tCEComICEGenFoundationTranslatorRef iGenFoundationTranslatorRef;\n");
		    &writeString("\t\thr = iGenFoundationTranslatorRef.initByQueryInterface(${VALUE});\n");
		    &writeString("\t\tif (CESucceeded(hr))\n");
		    &writeString("\t\t{\n");
		    &writeString("\t\t\thr = iGenFoundationTranslatorRef.getSubstance(CEComIID_${TYPE}, reinterpret_cast<void**>(&${VALUE}Ref));\n");
		    &writeString("\t\t}\n");
		    &writeString("\t\telse\n");
		    &writeString("\t\t{\n");
		    &writeString("\t\t\thr = ${VALUE}Ref.initByQueryInterface(${VALUE});\n");
		    &writeString("\t\t}\n");
		    &writeString("\t}\n");
		    &writeString("\n");

		    $VALUES[$i] .= 'Ref';
		}
	    }
	}
    }

    return join(', ', @VALUES);
}

# [Bug 5741] 2006/2/1 miyasita
# replace interface names into struct names.
# this is necessary to make the header file accepted by C compiler (not C++).
#
# e.g.   "ICEFoo* pThis, ICEGoo* pTarget, UINT32 val"  ----> "struct ICEFoo* pThis, struct ICEGoo* pTarget, UINT32 val"
sub replaceInterfaceWithStruct
{
    my($PARAM) = @_;
    $PARAM =~ s/(ICE\w+\s*\*)/struct $1/g;

    return $PARAM;
}

sub replaceInterfaceWithoutStruct
{
    my($PARAM) = @_;
    $PARAM =~ s/struct ICE/ICE/g;

    @_[0] = $PARAM;
}

sub replaceIdlFilePrefixForPublic
{
    my($PARAM) = @_;
    $PARAM =~ s/ICEApx/ICESpx/g;
    $PARAM =~ s/ICEMss/ICEMsf/g;

    return $PARAM;
}

sub replaceInterfacePrefixForPublic
{
    my($PARAM) = @_;
    $PARAM =~ s/ICEApx/ICESpx/g;
    $PARAM =~ s/ICEMss/ICEMsf/g;

    return $PARAM;
}

sub replaceVariablePrefixForPublic
{
    my($PARAM) = @_;
    $PARAM =~ s/_apx/_spx/g;
    $PARAM =~ s/_mss/_msf/g;
    $PARAM =~ s/_mrc/_mrs/g;

    return $PARAM;
}

sub replaceVtableFunctionNamePrefixForPublic
{
    my($PARAM) = @_;
    $PARAM =~ s/__apx/__spx/g;
    $PARAM =~ s/__mss/__msf/g;
    $PARAM =~ s/__mrc/__mrs/g;

    return $PARAM;
}

sub replaceFunctionParamVariablePrefixForPublic
{
    my($PARAM) = @_;
    $PARAM =~ s/iApx/iSpx/g;
    $PARAM =~ s/iMss/iMsf/g;

    return $PARAM;
}

sub replaceFunctionParamVariablePrefixForPrivate
{
    my($PARAM) = @_;
    $PARAM =~ s/iSpx/iApx/g;
    $PARAM =~ s/iMsf/iMss/g;

    return $PARAM;
}

sub replaceInterfaceValueNameToPointerValueNameForPublic
{
    my($PARAM) = @_;
    $PARAM =~ s/iSpx/spx/g;
    $PARAM =~ s/iMsf/msf/g;

    return $PARAM;
}

sub replaceInterfaceValueNameToPointerValueNameForPublic
{
    my($PARAM) = @_;
    $PARAM =~ s/iSpx/spx/g;
    $PARAM =~ s/iMsf/msf/g;

    return $PARAM;
}

sub replaceInterfaceToVarForPublic
{
    my($PARAM) = @_;
    $PARAM =~ s/ICEApx/_iSpx/;
    $PARAM =~ s/ICEMss/_iMsf/;
    $PARAM .= "Ref";

    return $PARAM;
}

sub replaceInterfaceToVarCallMethodForPublic
{
    my($PARAM) = @_;
    $PARAM =~ s/ICEApx/spx/;
    $PARAM =~ s/ICEMss/msf/;
    $PARAM .= "Ref()";

    return $PARAM;
}

sub replaceReinterpretCastTo
{
    my($IS_APX_TO_SPX, $VALUE_LIST, $PARAM_LIST) = @_;

    my($BEFORE);
    my($AFTER);
    if ($IS_APX_TO_SPX)
    {
	$BEFORE = "Apx";
	$AFTER  = "Spx";
    }
    else
    {
	$BEFORE = "Spx";
	$AFTER  = "Apx";
    }

    my($LOOP) = 1;
    while ($LOOP)
    {
	if ($PARAM_LIST =~ /\s*,\s*/)
	{
	    my($FRONT1) = "$`"; #&writeString("//front1 = $FRONT1\n");
	    my($MATCH1) = "$&"; #&writeString("//match1 = $MATCH1\n");
	    my($BACK1)  = "$'"; #&writeString("//back1  = $BACK1\n");

	    if ($FRONT1 =~ /ICE$BEFORE.*?\*/)
	    {
		my($FRONT3) = "$`"; #&writeString("//front3 = $FRONT3\n");
		my($MATCH3) = "$&"; #&writeString("//match3 = $MATCH3\n");
		my($BACK3)  = "$'"; #&writeString("//back3  = $BACK3\n");

		$MATCH3 =~ s/ICE$BEFORE/ICE$AFTER/; #&writeString("//match3 = $MATCH3\n");
		$BACK3  =~ s/\s//g;                 #&writeString("//back3  = $BACK3\n");

		$VALUE_LIST =~ s/$BACK3/reinterpret_cast<$MATCH3>($BACK3)/;
	    }

	    $PARAM_LIST = $BACK1;
	}
	else
	{
	    if ($PARAM_LIST =~ /ICE$BEFORE.*?\*/)
	    {
		my($FRONT2) = "$`"; #&writeString("//front2 = $FRONT2\n");
		my($MATCH2) = "$&"; #&writeString("//match2 = $MATCH2\n");
		my($BACK2)  = "$'"; #&writeString("//back2  = $BACK2\n");

		$MATCH2 =~ s/ICE$BEFORE/ICE$AFTER/; #&writeString("//match2 = $MATCH2\n");
		$BACK2  =~ s/\s//g;                 #&writeString("//back2  = $BACK2\n");

		$VALUE_LIST =~ s/$BACK2/reinterpret_cast<$MATCH2>($BACK2)/;
	    }

	    $LOOP = 0;
	}
    }

    return $VALUE_LIST;
}

# opposite with above
sub extractInterfaceWithStruct
{
    my($PARAM) = @_;
    $PARAM =~ s/struct\s+(ICE\w+)\s*\*\s*/$1/g;

    return $PARAM;
}

sub getOSName
{
    my($OS) = $;
    $OS =~ s/\s//g;

    return $OS;
}

sub getProgramName
{
    return $0;
}

sub getPerlName
{
    return $^X;
}

1; ### for required call.

########################################################################################################
### fnd trans
########################################################################################################
sub fndTransFromICEUString
{
    my($PARAM) = @_;
    my(@VARIABLES) = ();

    ### struct ICEUString* str ---> STR_BUF* strStr, STR_LEN strLen
    my($FLAG_ICEUSTRING) = 1;
    while ($FLAG_ICEUSTRING)
    {
	if ($PARAM =~ /struct ICEUString\s*\*\s*/)
	{
	    my($NEW_PARAM);

	    my($STR_BUF)    = "STR_BUF* ";
	    my($STR_LEN)    = "STR_LEN ";
	    my($STR)        = "Str";
	    my($LEN)        = "Len";

	    my($FRONT1) = "$`"; #&writeString("front1 = $FRONT1\n");
	    my($MATCH1) = "$&"; #&writeString("match1 = $MATCH1\n");
	    my($BACK1)  = "$'"; #&writeString("back1  = $BACK1\n");

	    if ($BACK1 =~ /\s*,\s*/)
	    {
		my($FRONT2) = "$`"; #&writeString("front2 = $FRONT2\n");
		my($MATCH2) = "$&"; #&writeString("match2 = $MATCH2\n");
		my($BACK2)  = "$'"; #&writeString("back2  = $BACK2\n");

		$NEW_PARAM = "$FRONT1$STR_BUF$FRONT2$STR, $STR_LEN$FRONT2$LEN, $BACK2";
		push(@VARIABLES, $FRONT2);
	    }
	    elsif ($BACK1 =~ /\s/)
	    {
		my($FRONT3) = "$`"; #&writeString("front3 = $FRONT3\n");
		my($MATCH3) = "$&"; #&writeString("match3 = $MATCH3\n");
		my($BACK3)  = "$'"; #&writeString("back3  = $BACK3\n");

		$NEW_PARAM = "$FRONT1$STR_BUF$FRONT3$STR, $STR_LEN$FRONT3$LEN";
		push(@VARIABLES, $FRONT3);
	    }
	    else
	    {
		$NEW_PARAM = "$FRONT1$STR_BUF$BACK1$STR, $STR_LEN$BACK1$LEN";
		push(@VARIABLES, $BACK1);
	    }

	    $PARAM = $NEW_PARAM;
	}
	else
	{
	    $FLAG_ICEUSTRING = 0;
	}
    }
    @_[0] = $PARAM;

    return @VARIABLES;
}

sub fndTransFromCERect
{
    my($PARAM) = @_;
    my(@VARIABLES) = ();

    ### CERect* rect ---> RECT_X rect_x, RECT_Y rect_y, RECT_W rect_width, RECT_H rect_height
    my($FLAG_CERECT) = 1;
    while ($FLAG_CERECT)
    {
	if ($PARAM =~ /const\s*CERect\s*\*\s*/ || $PARAM =~ /CERect\s*\*\s*/)
	{
	    my($NEW_PARAM);

	    my($RECT_X) = "RECT_X ";
	    my($RECT_Y) = "RECT_Y ";
	    my($RECT_W) = "RECT_W ";
	    my($RECT_H) = "RECT_H ";
	    my($X)     = "_x";
	    my($Y)     = "_y";
	    my($W)     = "_width";
	    my($H)     = "_height";

	    my($FRONT1) = "$`"; #&writeString("front1 = $FRONT1\n");
	    my($MATCH1) = "$&"; #&writeString("match1 = $MATCH1\n");
	    my($BACK1)  = "$'"; #&writeString("back1  = $BACK1\n");

	    if ($BACK1 =~ /\s*,\s*/)
	    {
		my($FRONT2) = "$`"; #&writeString("front2 = $FRONT2\n");
		my($MATCH2) = "$&"; #&writeString("match2 = $MATCH2\n");
		my($BACK2)  = "$'"; #&writeString("back2  = $BACK2\n");

		$NEW_PARAM = "$FRONT1$RECT_X$FRONT2$X, $RECT_Y$FRONT2$Y, $RECT_W$FRONT2$W, $RECT_H$FRONT2$H, $BACK2";
		push(@VARIABLES, $FRONT2);
	    }
	    elsif ($BACK1 =~ /\s/)
	    {
		my($FRONT3) = "$`"; #&writeString("front3 = $FRONT3\n");
		my($MATCH3) = "$&"; #&writeString("match3 = $MATCH3\n");
		my($BACK3)  = "$'"; #&writeString("back3  = $BACK3\n");

		$NEW_PARAM = "$FRONT1$RECT_X$FRONT3$X, $RECT_Y$FRONT3$Y, $RECT_W$FRONT3$W, $RECT_H$FRONT3$H";
		push(@VARIABLES, $FRONT3);
	    }
	    else
	    {
		$NEW_PARAM = "$FRONT1$RECT_X$BACK1$X, $RECT_Y$BACK1$Y, $RECT_W$BACK1$W, $RECT_H$BACK1$H";
		push(@VARIABLES, $BACK1);
	    }

	    $PARAM = $NEW_PARAM;
	}
	else
	{
	    $FLAG_CERECT = 0;
	}
    }
    @_[0] = $PARAM;

    return @VARIABLES;
}

sub fndTransFromCERectF
{
    my($PARAM) = @_;
    my(@VARIABLES) = ();

    ### CERectF* rectf ---> RECTF_X rectf_x, RECTF_Y rectf_y, RECTF_W rectf_width, RECTF_H rectf_height
    my($FLAG_CERECTF) = 1;
    while ($FLAG_CERECTF)
    {
	if ($PARAM =~ /const\s*CERectF\s*\*\s*/ || $PARAM =~ /CERectF\s*\*\s*/)
	{
	    my($NEW_PARAM);

	    my($RECTF_X) = "RECTF_X ";
	    my($RECTF_Y) = "RECTF_Y ";
	    my($RECTF_W) = "RECTF_W ";
	    my($RECTF_H) = "RECTF_H ";
	    my($X)     = "_x";
	    my($Y)     = "_y";
	    my($W)     = "_width";
	    my($H)     = "_height";

	    my($FRONT1) = "$`"; #&writeString("front1 = $FRONT1\n");
	    my($MATCH1) = "$&"; #&writeString("match1 = $MATCH1\n");
	    my($BACK1)  = "$'"; #&writeString("back1  = $BACK1\n");

	    if ($BACK1 =~ /\s*,\s*/)
	    {
		my($FRONT2) = "$`"; #&writeString("front2 = $FRONT2\n");
		my($MATCH2) = "$&"; #&writeString("match2 = $MATCH2\n");
		my($BACK2)  = "$'"; #&writeString("back2  = $BACK2\n");

		$NEW_PARAM = "$FRONT1$RECTF_X$FRONT2$X, $RECTF_Y$FRONT2$Y, $RECTF_W$FRONT2$W, $RECTF_H$FRONT2$H, $BACK2";
		push(@VARIABLES, $FRONT2);
	    }
	    elsif ($BACK1 =~ /\s/)
	    {
		my($FRONT3) = "$`"; #&writeString("front3 = $FRONT3\n");
		my($MATCH3) = "$&"; #&writeString("match3 = $MATCH3\n");
		my($BACK3)  = "$'"; #&writeString("back3  = $BACK3\n");

		$NEW_PARAM = "$FRONT1$RECTF_X$FRONT3$X, $RECTF_Y$FRONT3$Y, $RECTF_W$FRONT3$W, $RECTF_H$FRONT3$H";
		push(@VARIABLES, $FRONT3);
	    }
	    else
	    {
		$NEW_PARAM = "$FRONT1$RECTF_X$BACK1$X, $RECTF_Y$BACK1$Y, $RECTF_W$BACK1$W, $RECTF_H$BACK1$H";
		push(@VARIABLES, $BACK1);
	    }

	    $PARAM = $NEW_PARAM;
	}
	else
	{
	    $FLAG_CERECTF = 0;
	}
    }
    @_[0] = $PARAM;

    return @VARIABLES;
}

sub fndTransFromCERawDataBuffer
{
    my($PARAM) = @_;
    my(@VARIABLES) = ();

    ### CERawDataBuffer data --> DATA_LEN dataLen, DATA_BUF dataBuf
    my($FLAG_CERAWDATABUFFER) = 1;
    while ($FLAG_CERAWDATABUFFER)
    {
	if ($PARAM =~ /CERawDataBuffer\s*\*\s*/)
	{
	    my($NEW_PARAM);

	    my($DATA_BUF) = "DATA_BUF* ";
	    my($DATA_LEN) = "DATA_LEN ";
	    my($BUF)      = "Buf";
	    my($LEN)      = "Len";

	    my($FRONT1) = "$`"; #&writeString("front1 = $FRONT1\n");
	    my($MATCH1) = "$&"; #&writeString("match1 = $MATCH1\n");
	    my($BACK1)  = "$'"; #&writeString("back1  = $BACK1\n");

	    if ($BACK1 =~ /\s*,\s*/)
	    {
		my($FRONT2) = "$`"; #&writeString("front2 = $FRONT2\n");
		my($MATCH2) = "$&"; #&writeString("match2 = $MATCH2\n");
		my($BACK2)  = "$'"; #&writeString("back2  = $BACK2\n");

		$NEW_PARAM = "$FRONT1$DATA_BUF$FRONT2$BUF, $DATA_LEN$FRONT2$LEN, $BACK2";
		push(@VARIABLES, $FRONT2);
	    }
	    elsif ($BACK1 =~ /\s/)
	    {
		my($FRONT3) = "$`"; #&writeString("front3 = $FRONT3\n");
		my($MATCH3) = "$&"; #&writeString("match3 = $MATCH3\n");
		my($BACK3)  = "$'"; #&writeString("back3  = $BACK3\n");

		$NEW_PARAM = "$FRONT1$DATA_BUF$FRONT3$BUF, $DATA_LEN$FRONT3$LEN";
		push(@VARIABLES, $FRONT3);
	    }
	    else
	    {
		$NEW_PARAM = "$FRONT1$DATA_BUF$BACK1$BUF, $DATA_LEN$BACK1$LEN";
		push(@VARIABLES, $BACK1);
	    }

	    $PARAM = $NEW_PARAM;
	}
	else
	{
	    $FLAG_CERAWDATABUFFER = 0;
	}
    }
    @_[0] = $PARAM;

    return @VARIABLES;
}

sub fndTransFromCEUStringList
{
    my($PARAM) = @_;
    my(@VARIABLES) = ();

    ### CEUStringList* array -->  STRARRAY_BUF* arrayBuf, STRARRAY_LEN arrayLen
    my($FLAG_CEUSTRINGLIST) = 1;
    while ($FLAG_CEUSTRINGLIST)
    {
	if ($PARAM =~ /CEUStringList\s*\*\s*/)
	{
	    my($NEW_PARAM);

	    my($STRARRAY_BUF) = "STRARRAY_BUF* ";
	    my($STRARRAY_LEN) = "STRARRAY_LEN ";
	    my($BUF)      = "Buf";
	    my($LEN)      = "Len";


	    my($FRONT1) = "$`"; #&writeString("front1 = $FRONT1\n");
	    my($MATCH1) = "$&"; #&writeString("match1 = $MATCH1\n");
	    my($BACK1)  = "$'"; #&writeString("back1  = $BACK1\n");

	    if ($BACK1 =~ /\s*,\s*/)
	    {
		my($FRONT2) = "$`"; #&writeString("front2 = $FRONT2\n");
		my($MATCH2) = "$&"; #&writeString("match2 = $MATCH2\n");
		my($BACK2)  = "$'"; #&writeString("back2  = $BACK2\n");

		$NEW_PARAM = "$FRONT1$STRARRAY_BUF$FRONT2$BUF, $STRARRAY_LEN$FRONT2$LEN, $BACK2";
		push(@VARIABLES, $FRONT2);
	    }
	    elsif ($BACK1 =~ /\s/)
	    {
		my($FRONT3) = "$`"; #&writeString("front3 = $FRONT3\n");
		my($MATCH3) = "$&"; #&writeString("match3 = $MATCH3\n");
		my($BACK3)  = "$'"; #&writeString("back3  = $BACK3\n");

		$NEW_PARAM = "$FRONT1$STRARRAY_BUF$FRONT3$BUF, $STRARRAY_LEN$FRONT3$LEN";
		push(@VARIABLES, $FRONT3);
	    }
	    else
	    {
		$NEW_PARAM = "$FRONT1$STRARRAY_BUF$BACK1$BUF, $STRARRAY_LEN$BACK1$LEN";
		push(@VARIABLES, $BACK1);
	    }

	    $PARAM = $NEW_PARAM;
	}
	else
	{
	    $FLAG_CEUSTRINGLIST = 0;
	}
    }
    @_[0] = $PARAM;

    return @VARIABLES;
}

sub fndTransFromCEUUINT32List
{
    my($PARAM) = @_;
    my(@VARIABLES) = ();

    ### CEUUINT32List* array -->  UINT32ARRAY_BUF* arrayBuf, UINT32ARRAY_LEN arrayLen
    my($FLAG_CEUUINT32LIST) = 1;
    while ($FLAG_CEUUINT32LIST)
    {
	if ($PARAM =~ /CEUUINT32List\s*\*\s*/)
	{
	    my($NEW_PARAM);

	    my($UINT32ARRAY_BUF) = "UINT32ARRAY_BUF* ";
	    my($UINT32ARRAY_LEN) = "UINT32ARRAY_LEN ";
	    my($BUF)      = "Buf";
	    my($LEN)      = "Len";


	    my($FRONT1) = "$`"; #&writeString("front1 = $FRONT1\n");
	    my($MATCH1) = "$&"; #&writeString("match1 = $MATCH1\n");
	    my($BACK1)  = "$'"; #&writeString("back1  = $BACK1\n");

	    if ($BACK1 =~ /\s*,\s*/)
	    {
		my($FRONT2) = "$`"; #&writeString("front2 = $FRONT2\n");
		my($MATCH2) = "$&"; #&writeString("match2 = $MATCH2\n");
		my($BACK2)  = "$'"; #&writeString("back2  = $BACK2\n");

		$NEW_PARAM = "$FRONT1$UINT32ARRAY_BUF$FRONT2$BUF, $UINT32ARRAY_LEN$FRONT2$LEN, $BACK2";
		push(@VARIABLES, $FRONT2);
	    }
	    elsif ($BACK1 =~ /\s/)
	    {
		my($FRONT3) = "$`"; #&writeString("front3 = $FRONT3\n");
		my($MATCH3) = "$&"; #&writeString("match3 = $MATCH3\n");
		my($BACK3)  = "$'"; #&writeString("back3  = $BACK3\n");

		$NEW_PARAM = "$FRONT1$UINT32ARRAY_BUF$FRONT3$BUF, $UINT32ARRAY_LEN$FRONT3$LEN";
		push(@VARIABLES, $FRONT3);
	    }
	    else
	    {
		$NEW_PARAM = "$FRONT1$UINT32ARRAY_BUF$BACK1$BUF, $UINT32ARRAY_LEN$BACK1$LEN";
		push(@VARIABLES, $BACK1);
	    }

	    $PARAM = $NEW_PARAM;
	}
	else
	{
	    $FLAG_CEUUINT32LIST = 0;
	}
    }
    @_[0] = $PARAM;

    return @VARIABLES;
}

sub setFndTransFlag
{
    $FND_TRANS_FLAG = @_;
}

sub fndTransFunctionParam
{
    my($PARAM, $FND_TRANS, $ICEUSTRING_VARS, $CERECT_VARS, $CERECTF_VARS, $CERAWDATABUFFER_VARS, $CEUSTRINGLIST_VARS, $CEUUINT32LIST_VARS) = @_;

    if ($FND_TRANS eq ':trans-to-fnd' || $FND_TRANS eq ':trans-from-fnd')
    {
	@{$_[2]} = fndTransFromICEUString($PARAM);
	@{$_[3]} = fndTransFromCERect($PARAM);
	@{$_[4]} = fndTransFromCERectF($PARAM);
	@{$_[5]} = fndTransFromCERawDataBuffer($PARAM);
	@{$_[6]} = fndTransFromCEUStringList($PARAM);
	@{$_[7]} = fndTransFromCEUUINT32List($PARAM);
	$PARAM   = replaceFunctionParamVariablePrefixForPublic($PARAM);
	@_[0]    = $PARAM;
    }
    else
    {
	die("Error: Wrong type $FND_TRANS, stopped");
    }
}

sub getJNIMethodName
{
    my($CLASS_REF, $FUNCTION_NAME) = @_;
    my($MARSHAL) = $CLASS_REF->{'marshal'};
    my($JNI_METHOD_NAME);
    if ($MARSHAL eq ':lw-marshal' || $MARSHAL eq ':std-marshal')
    {
	$JNI_METHOD_NAME = $CLASS_REF->{'name'};
    }
    elsif ($MARSHAL eq ':ipc-marshal')
    {
	$JNI_METHOD_NAME = $CLASS_REF->{'name'} . 'Proxy_' . $FUNCTION_NAME . "_JNI";
	$JNI_METHOD_NAME =~ s/Apt/__apt/;
    }
    else
    {
	die("Error: marshal type should be specified.\n");
    }

    return $JNI_METHOD_NAME;
}

sub getJavaInterfaceInstanceName
{
    my($INTERFACE_NAME) = @_;
    $INTERFACE_NAME =~ s/ICE/ice/;
    return $INTERFACE_NAME;
}

sub getJavaMarshallerInstanceName
{
    my($MAR_NAME) = @_;
    $MAR_NAME =~ s/Spx/spx/;
    return $MAR_NAME;
}

sub isJavaInterface
{
    my($JAVA_CLASS_NAME) = @_;
    my($retVal) = 0;
    if($JAVA_CLASS_NAME =~ /ICE/)
    {
	$retVal = 1;
    }
    else
    {
	$retVal = 0;
    }
    return $retVal;
}
