///////////////////////////////////////////////////////////////////////////////
// Copyright 2009 Sony Corporation
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
// 
///////////////////////////////////////////////////////////////////////////////
#include "WebKitSilkPrefix.h"
#include "CEJSExtVariant.h"
#include "CEJSExtVariantGlue.h"
#include "CEJSExternal.h"

#include <runtime/JSGlobalObject.h>
#include "runtime/JSValue.h"


CEJSExtVariant::CEJSExtVariant()
{
	_refCount = 0;
	_propertyValue = JSC::jsUndefined();
}

CEJSExtVariant::~CEJSExtVariant()
{
}

/////////////////////////////////
// factory
/////////////////////////////////
CEHResult CEJSExtVariant::createFromUndefined(JSC::ExecState* exec, CEJSExtVariant** pVariantOut)
{
	CEHResult hr = CE_SILK_ERR_MEMERR;
	CEJSExtVariant* pVariant = new CEJSExtVariantGlue();
	if (pVariant)
	{
		hr = pVariant->_initWithUndefined(exec);
		if (CESucceeded(hr))
		{
			*pVariantOut = pVariant;
		}
		else
		{
			delete pVariant;
		}
	}
	return hr;
}

CEHResult CEJSExtVariant::createFromNull(JSC::ExecState* exec, CEJSExtVariant** pVariantOut)
{
	CEHResult hr = CE_SILK_ERR_MEMERR;
	CEJSExtVariant* pVariant = new CEJSExtVariantGlue();
	if (pVariant)
	{
		hr = pVariant->_initWithNull(exec);
		if (CESucceeded(hr))
		{
			*pVariantOut = pVariant;
		}
		else
		{
			delete pVariant;
		}
	}
	return hr;
}

CEHResult CEJSExtVariant::createFromBoolean(JSC::ExecState* exec, UINT8 booleanValue, CEJSExtVariant** pVariantOut)
{
	CEHResult hr = CE_SILK_ERR_MEMERR;
	CEJSExtVariant* pVariant = new CEJSExtVariantGlue();
	if (pVariant)
	{
		hr = pVariant->_initWithBoolean(exec, booleanValue);
		if (CESucceeded(hr))
		{
			*pVariantOut = pVariant;
		}
		else
		{
			delete pVariant;
		}
	}
	return hr;
}

CEHResult CEJSExtVariant::createFromInt32(JSC::ExecState* exec, INT32 int32Value, CEJSExtVariant** pVariantOut)
{
	CEHResult hr = CE_SILK_ERR_MEMERR;
	CEJSExtVariant* pVariant = new CEJSExtVariantGlue();
	if (pVariant)
	{
		hr = pVariant->_initWithInt32(exec, int32Value);
		if (CESucceeded(hr))
		{
			*pVariantOut = pVariant;
		}
		else
		{
			delete pVariant;
		}
	}
	return hr;
}

CEHResult CEJSExtVariant::createFromNumber(JSC::ExecState* exec, double doubleValue, CEJSExtVariant** pVariantOut)
{
	CEHResult hr = CE_SILK_ERR_MEMERR;
	CEJSExtVariant* pVariant = new CEJSExtVariantGlue();
	if (pVariant)
	{
		hr = pVariant->_initWithNumber(exec, doubleValue);
		if (CESucceeded(hr))
		{
			*pVariantOut = pVariant;
		}
		else
		{
			delete pVariant;
		}
	}
	return hr;
}

CEHResult CEJSExtVariant::createFromString(JSC::ExecState* exec, const UTF16CHAR* pCharArray16, UINT32 numOfChars16, CEJSExtVariant** pVariantOut)
{
	CEHResult hr = CE_SILK_ERR_MEMERR;
	CEJSExtVariant* pVariant = new CEJSExtVariantGlue();
	if (pVariant)
	{
		hr = pVariant->_initWithString(exec, pCharArray16, numOfChars16);
		if (CESucceeded(hr))
		{
			*pVariantOut = pVariant;
		}
		else
		{
			delete pVariant;
		}
	}
	return hr;
}

CEHResult CEJSExtVariant::createFromPropertyValue(
	JSC::ExecState* exec, JSC::JSValue* pPropertyValue, CEJSExtVariant** pVariantOut)
{
	CEHResult hr = CE_SILK_ERR_BADARGS;
	if (pPropertyValue)
	{
		CEJSExtVariant* pVariant = new CEJSExtVariantGlue();
		if (pVariant)
		{
			pVariant->_propertyValue = *pPropertyValue;
			*pVariantOut = pVariant;
			hr = CE_S_OK;
		}
		else
		{
			hr = CE_SILK_ERR_MEMERR;
		}
	}
	return hr;
}

inline CEHResult CEJSExtVariant::_initWithUndefined(JSC::ExecState* exec)
{
	_propertyValue = JSC::jsUndefined();
	return CE_S_OK;
}

inline CEHResult CEJSExtVariant::_initWithNull(JSC::ExecState* exec)
{
	_propertyValue = JSC::jsNull();
	return CE_S_OK;
}

inline CEHResult CEJSExtVariant::_initWithBoolean(JSC::ExecState* exec, bool booleanValue)
{
	_propertyValue = JSC::jsBoolean(booleanValue);
	return CE_S_OK;
}

inline CEHResult CEJSExtVariant::_initWithInt32(JSC::ExecState* exec, INT32 int32Value)
{
	_propertyValue = JSC::jsNumber(exec, (long)int32Value);
	return CE_S_OK;
}

inline CEHResult CEJSExtVariant::_initWithNumber(JSC::ExecState* exec, double numberValue)
{
	_propertyValue = JSC::jsNumber(exec, numberValue);
	return CE_S_OK;
}

inline CEHResult CEJSExtVariant::_initWithString(JSC::ExecState* exec, const UTF16CHAR* pCharArray16, UINT32 numOfChars16)
{
	_propertyValue = JSC::jsString(exec, JSC::UString((UChar*)pCharArray16, numOfChars16, true));
	return CE_S_OK;
}

CEHResult CEJSExtVariant::getType(CEJSExtVariantType* typeOut)
{
	CEHResult hr = CE_S_OK;
	if( _propertyValue.isUndefined() )
	{
		*typeOut = CEJSVariantType_Undefined;
	}
	else if( _propertyValue.isNull() )
	{
		*typeOut = CEJSVariantType_Null;
	}
	else if( _propertyValue.isObject() )
	{
		*typeOut = CEJSVariantType_Object;
	}
	else if(  _propertyValue.isNumber() )
	{
		*typeOut = CEJSVariantType_Number;
	}
	else if( _propertyValue.isString() )
	{
		*typeOut = CEJSVariantType_String;
	}
	else if( _propertyValue.isBoolean() )
	{
		*typeOut = CEJSVariantType_Boolean;
	}
	else
	{
		CEASSERT(0 && "why unknown type?");
		hr = CE_SILK_ERR_OPERATION_FAILED;
	}
	return hr;
}

CEHResult CEJSExtVariant::toBoolean(UINT8* booleanOut)
{
	if( CEJSExternalAppCallback::getExecState() )
	{
		*booleanOut = (UINT8)_propertyValue.toBoolean(CEJSExternalAppCallback::getExecState());
		return(CE_S_OK);
	}
	return(CE_SILK_ERR_OPERATION_FAILED);
}

CEHResult CEJSExtVariant::toNumber(double* numberOut)
{
	if( CEJSExternalAppCallback::getExecState() )
	{
		*numberOut = (double)_propertyValue.toFloat(CEJSExternalAppCallback::getExecState());
		return(CE_S_OK);
	}
	return(CE_SILK_ERR_OPERATION_FAILED);
}

CEHResult CEJSExtVariant::toInteger(double* integerOut)
{
	if( CEJSExternalAppCallback::getExecState() )
	{
		bool ok = false;
		double val = (double)_propertyValue.toInt32(CEJSExternalAppCallback::getExecState(), ok);
		if ( ok )
		{
			*integerOut = val;
			return(CE_S_OK);
		}
	}
	return(CE_SILK_ERR_OPERATION_FAILED);
}

CEHResult CEJSExtVariant::toInt32(INT32* int32Out)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	if( CEJSExternalAppCallback::getExecState() )
	{
		bool ok = false;
		INT32 val = (INT32)_propertyValue.toInt32(CEJSExternalAppCallback::getExecState(), ok);
		if ( ok )
		{
			*int32Out = val;
			return(CE_S_OK);
		}
	}
	return(CE_SILK_ERR_OPERATION_FAILED);
}

CEHResult CEJSExtVariant::toUint32(UINT32* uint32Out)
{
	if( CEJSExternalAppCallback::getExecState() )
	{
		bool ok = false;
		UINT32 val = (UINT32)_propertyValue.toUInt32(CEJSExternalAppCallback::getExecState(), ok);
		if ( ok )
		{
			*uint32Out = val;
			return(CE_S_OK);
		}
	}
	return(CE_SILK_ERR_OPERATION_FAILED);
}

CEHResult CEJSExtVariant::toUint16(UINT16* uint16Out)
{
	if( CEJSExternalAppCallback::getExecState() )
	{
		bool ok = false;
		UINT16 val = (UINT16)_propertyValue.toUInt32(CEJSExternalAppCallback::getExecState(), ok);
		if ( ok )
		{
			*uint16Out = val;
			return(CE_S_OK);
		}
	}
	return(CE_SILK_ERR_OPERATION_FAILED);
}

CEHResult CEJSExtVariant::toString(const UTF16CHAR **const pCharArray16Out, UINT32 *const numOfChars16Out)
{
	if( CEJSExternalAppCallback::getExecState() )
	{
		_stringValue = _propertyValue.toString(CEJSExternalAppCallback::getExecState());
		*pCharArray16Out = reinterpret_cast<const UTF16CHAR*>(_stringValue.characters());
		*numOfChars16Out = static_cast<UINT32>(_stringValue.length());
		return(CE_S_OK);
	}
	return(CE_SILK_ERR_OPERATION_FAILED);
}
