/*
 * Copyright (C) 2007 Kevin Ollivier  All rights reserved.
 * Copyright     2009 Sony Corporation
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
#include "WebKitSilkPrefix.h"

#include "CEWebKitSettings.h"
#include "CEWebKitFrameWindow.h"
#include "CEWebKitFrameWindowImpl.h"
#include "WebCoreSupport/ChromeClientSilk.h"
#include "WebCoreSupport/ContextMenuClientSilk.h"
#include "WebCoreSupport/EditorClientSilk.h"
#include "WebCoreSupport/DragClientSilk.h"
#include "WebCoreSupport/InspectorClientSilk.h"
#include "FocusController.h"
#include "Settings.h"
#include "DragController.h"
#include "Frame.h"
#include "Editor.h"
#include "Pasteboard.h"
#include "PopupMenu.h"
#include "PopupMenuClient.h"
#include "TypingCommand.h"
#include "KeyboardEvent.h"
#include "PlatformKeyboardEvent.h"
#include "EventNames.h"
#include "FrameView.h"
#include "GCController.h"
#include "HitTestRequest.h"
#include "HitTestResult.h"
#include "FrameView.h"
#include "HTMLNames.h"
#include "HTMLInputElement.h"
#include "HTMLTextAreaElement.h"
#include "RenderTextControlSingleLine.h"
#include "RenderTextControlMultiLine.h"
#include "PluginView.h"
#include "CEWebKitDocumentLoader.h"
#include "ICEHtmlClipboard.h"
#include "HTMLAnchorElement.h"

#include "CEWebKitUtility.h"

#include "CEWebKitFocusImpl.h"

#ifdef HAVE_FLOAT_H
#undef HAVE_FLOAT_H
#endif
#include "CEWebKitImpl.h"
#include "PageCache.h"



#include "CEWebKitWindow.h"
#include "CETLTypes.h"
#include "ICEHtmlBrowserPreference.h"

CEWebKitFrameWindowImpl::CEWebKitFrameWindowImpl() :
	_private(NULL)
	,_mainWindow(NULL)
	,_lockingOutForMemoryLimit(false)
    , _hasInitializedTextInput(false)
	,_skipNotifyScrollPosition(false)
	,_hitTestPadding(0)
	,_imeTargetPlugin(NULL)
	,_isFocusNodeChanged_Mouse(false)
	,_focusedTextInputNode(NULL)
{
	_defaultFrameSize.setX(0);
	_defaultFrameSize.setY(0);
	_defaultFrameSize.setWidth(0);
	_defaultFrameSize.setHeight(0);
}

CEWebKitFrameWindowImpl::~CEWebKitFrameWindowImpl()
{
}

CEHResult CEWebKitFrameWindowImpl::init(ICEHtmlWebKitFrameWindowWidget* iWidget, ICEHtmlWebKitFrameWindowChromeClient* iChromeClient, ICEHtmlFocusNavigator* iFocusNavigator)
{
	CEHResult hr = CE_S_OK;

	_iWidgetRef = iWidget;
	_iChromeClientRef = iChromeClient;

	hr = _focusInit(iFocusNavigator);

	if (CESucceeded(hr))
	{
		CEComICEHtmlBrowserPreferenceRef iBrowserPreference;
		hr = iBrowserPreference.initByQueryInterface(_iChromeClientRef);
		if (CESucceeded(hr))
		{
			hr = iBrowserPreference.getHitTestPadding(&_hitTestPadding);
			if (CESucceeded(hr))
			{
				hr = _privateInit();

				if (CESucceeded(hr) && _iChromeClientRef)
				{
					hr = _iLoaderClientRef.initByQueryInterface(_iChromeClientRef);
					if (CE_SILK_ERR_NOTFOUND == hr) 
					{
						hr = CE_S_OK;
					}
				}
			}
		}
	}

	return hr;
}

CEHResult CEWebKitFrameWindowImpl::shutdown()
{
	CEHResult hr = hr = _privateShutdown();

	_focusShutdown();

	_iWidgetRef = NULL;
	_iChromeClientRef = NULL;
	_iLoaderClientRef = NULL;
	_imeTargetPlugin = NULL;

    if (CESucceeded(hr))
    {
        WebCore::gcController().garbageCollectNow();
    }
	
	return hr;
}

CEHResult CEWebKitFrameWindowImpl::load(ICEUString* iUrl, ICEUString* iReferer)
{
	CEHResult hr = CE_S_OK;

	if (_mainWindow && _mainWindow->object())
	{
		hr = _mainWindow->object()->load(iUrl, iReferer);
	}

	return hr;
}

CEHResult CEWebKitFrameWindowImpl::back(UINT32 amount)
{
	CEHResult hr = CE_S_OK;

	if (_mainWindow && _mainWindow->object())
	{
		hr = _mainWindow->object()->back();
	}

	return hr;
}

CEHResult CEWebKitFrameWindowImpl::forward(UINT32 amount)
{
	CEHResult hr = CE_S_OK;

	if (_mainWindow && _mainWindow->object())
	{
		hr = _mainWindow->object()->forward();
	}

	return hr;
}

CEHResult CEWebKitFrameWindowImpl::reload(bool skipCache)
{
	CEHResult hr = CE_S_OK;

	if (_mainWindow && _mainWindow->object())
	{
		hr = _mainWindow->object()->reload(skipCache);
	}

	return hr;
}

CEHResult CEWebKitFrameWindowImpl::paint(ICEVGContext* iVGC, CERect* dirtyRect)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	if (_mainWindow && _mainWindow->object())
	{
		bool bPaintContents = true;

#if USE(ACCELERATED_COMPOSITING)
		CEMRL_FPS_18278_SAVETIME(0, 4);
		hr = CE_S_OK;
		bPaintContents = (!_private->hasExceptForLayersBackingStore());
#endif //#if USE(ACCELERATED_COMPOSITING)

		if (bPaintContents)
			hr = _mainWindow->object()->paint(iVGC, dirtyRect);

#if USE(ACCELERATED_COMPOSITING)

		CEMRL_FPS_18278_SAVETIME(0, 5);

		if (CESucceeded(hr))
		{
			CERectF dirtyRectF = 
			{
				dirtyRect->_x, 
				dirtyRect->_y, 
				dirtyRect->_width, 
				dirtyRect->_height, 
			};
				
			hr = _private->renderLayerBackingStore(iVGC, &dirtyRectF);
		}
#endif //#if USE(ACCELERATED_COMPOSITING)

	}
	else if (_lockingOutForMemoryLimit)
	{
		hr = CE_S_OK;
	}
	return hr;
}

CEHResult CEWebKitFrameWindowImpl::mouse(CEMouseEventParam* mouseEventParam, bool* consumedOut)
{
	CEHResult hr = CE_SILK_ERR_BADARGS;
	if (mouseEventParam)
	{
		CEWebKitHtmlScrollMode scrollMode;
		hr = _iChromeClientRef.getScrollMode(&scrollMode);
		if (!hr)
		{
			bool consumed = false;
			switch (mouseEventParam->type)
			{
			case eCEMouseEventType_ButtonDown:
			case eCEMouseEventType_ButtonDoubleClicked:
			case eCEMouseEventType_ButtonTripleClicked:
			case eCEMouseEventType_Moved:
			case eCEMouseEventType_Dragged:
				hr = _mouse(mouseEventParam->type, mouseEventParam->point, mouseEventParam->button, mouseEventParam->modifiers, consumed);
				break;
			case eCEMouseEventType_ButtonUp:
			{
				if (_hitTestPadding>0) // Added padding area for hittest.
				{
					if (mouseEventParam->button==eCEMouseButton_Left)
					{
						hr = _mouseLButtonUpRepeatJudging(mouseEventParam->point,  mouseEventParam->modifiers, consumed);
					}
					else
					{
						hr = _mouse(mouseEventParam->type, mouseEventParam->point, mouseEventParam->button, mouseEventParam->modifiers, consumed);
					}
				}
				else
				{
					hr = _mouse(mouseEventParam->type, mouseEventParam->point, mouseEventParam->button, mouseEventParam->modifiers, consumed);
				}
				break;
			}
			case eCEMouseEventType_Entered:
			case eCEMouseEventType_Exited:
				hr = CE_S_OK;
				break;

			default:
				break;
			}

			if (mouseEventParam->modifiers & eCEKeyModifier_CTRL)
			{
				bool doZoomIn = false;
				bool doZoomOut = false;
				switch (mouseEventParam->type)
				{
				case eCEMouseEventType_WheelForward:
					doZoomIn = true;
					break;
				case eCEMouseEventType_WheelBackward:
					doZoomOut = true;
					break;
				case eCEMouseEventType_WheelRotated:
					if (mouseEventParam->wheelRotationY < 0) doZoomIn = true;
					else									 doZoomOut = true;
					break;
				default:
					break;
				}
				if (doZoomIn)
				{
					bool b = false;
					canZoomPageIn(&b);
					if (b)
						hr = zoomPageIn();
				}
				else if (doZoomOut)
				{
					bool b = false;
					canZoomPageOut(&b);
					if (b)
						hr = zoomPageOut();
				}
			}
			else
			{
				CEPointBase scrollPos = {0, 0};
				bool bScroll = true;
				switch (mouseEventParam->type)
				{
				case eCEMouseEventType_WheelForward:
					scrollPos._x = 0;
					scrollPos._y = 40;
					break;
				case eCEMouseEventType_WheelBackward:
					scrollPos._x = 0;
					scrollPos._y = -40;
					break;
				case eCEMouseEventType_WheelRotated:
					scrollPos._x = (-mouseEventParam->wheelRotationX);
					scrollPos._y = (-mouseEventParam->wheelRotationY);
					break;
				default:
					bScroll = false;
					break;
				}
				if (bScroll)
					hr = _scroll(mouseEventParam->point._x, mouseEventParam->point._y, scrollPos._x, scrollPos._y, mouseEventParam->modifiers, consumed);
			}

			if (CESucceeded(hr) && consumedOut)
			{
				*consumedOut = consumed;
			}
		}
	}

	return hr;
}

CEHResult CEWebKitFrameWindowImpl::focus(bool focused)
{
	//TODO: invoke window.onfocus or onblur
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	if (_mainWindow && _mainWindow->object())
	{
		hr = _mainWindow->object()->focus(focused);
	}
	return hr;
}

#define LPCVOID_TO_RESHANDLE(pointer) reinterpret_cast<WebCore::ResourceHandle*>(const_cast<void*>(pointer))

CEHResult CEWebKitFrameWindowImpl::didReceiveResponseCallback(const void* webCoreJob, const char* responseHeaderBuffer, UINT32 responseHeaderSize, const char* url, UINT32 httpCode, UINT32 expectedContentLength, bool noContext, bool syncRequest, UINT32 sslErr)
{
	CEHResult hr = CE_S_OK;

	if ( true == WebCore::protocolIs(LPCVOID_TO_RESHANDLE(webCoreJob)->request().url().string(), "https") )
		LPCVOID_TO_RESHANDLE(webCoreJob)->getInternal()->m_response.setSecureStatus(true, sslErr);

	if (syncRequest == false)
	{
		hr = WebCore::ResourceHandleManager::headerCallback(LPCVOID_TO_RESHANDLE(webCoreJob), responseHeaderBuffer, responseHeaderSize, url, httpCode, expectedContentLength, noContext);
	}
	else
	{
		WebCore::ResourceHandle* handle = LPCVOID_TO_RESHANDLE(webCoreJob);
		if (handle)
		{
			WebCore::ResourceHandleInternal* handleinternal = handle->getInternal();
			if (handleinternal)
			{
				WebCore::WebCoreSynchronousLoader* syncLoader = (WebCore::WebCoreSynchronousLoader*)(handleinternal->client());
				if (syncLoader)
				{
					syncLoader->didReceiveResponse(LPCVOID_TO_RESHANDLE(webCoreJob), responseHeaderBuffer, responseHeaderSize, url, httpCode, expectedContentLength, noContext);
				}
			}
		}
	}
	return hr;
}

CEHResult CEWebKitFrameWindowImpl::didReceiveDataCallback(const void* webCoreJob, UINT32 httpCode, const char* url, const void* data, UINT32 size, bool syncRequest)
{
	CEHResult hr = CE_S_OK;
	if (syncRequest == false)
	{
		hr = WebCore::ResourceHandleManager::DataAvailableCallback(LPCVOID_TO_RESHANDLE(webCoreJob), httpCode, url, const_cast<void*>(data), size);
	}
	else
	{
		WebCore::ResourceHandle* handle = LPCVOID_TO_RESHANDLE(webCoreJob);
		if (handle)
		{
			WebCore::ResourceHandleInternal* handleinternal = handle->getInternal();
			if (handleinternal)
			{
				WebCore::WebCoreSynchronousLoader* syncLoader = (WebCore::WebCoreSynchronousLoader*)(handleinternal->client());
				if (syncLoader)
				{
					syncLoader->didReceiveData(LPCVOID_TO_RESHANDLE(webCoreJob), const_cast<void*>(data), size);
				}
			}
		}
	}
	return hr;
}

CEHResult CEWebKitFrameWindowImpl::didFinishLoadingCallback(const void* webCoreJob, bool syncRequest)
{
	CEHResult hr = CE_S_OK;
	if (syncRequest == false)
	{
		hr = WebCore::ResourceHandleManager::CompleteCallback(LPCVOID_TO_RESHANDLE(webCoreJob));
	}
	return hr;
}

CEHResult CEWebKitFrameWindowImpl::didFinishLoadingAllInOneCallback(const void* webCoreJob, const char* responseHeaderBuffer, UINT32 responseHeaderSize, const char* url, UINT32 httpCode, UINT32 expectedContentLength, bool noContext, const void* data, UINT32 size, bool syncRequest)
{
	CEHResult hr = CE_S_OK;
	if (syncRequest == false)
	{
		hr = WebCore::ResourceHandleManager::allInOneCallback(LPCVOID_TO_RESHANDLE(webCoreJob), responseHeaderBuffer, responseHeaderSize, url, httpCode, expectedContentLength, noContext, const_cast<void*>(data), size);
	}
	return hr;
}

CEHResult CEWebKitFrameWindowImpl::willSendRequestCallback(const void* webCoreJob, const char* url, bool syncRequest)
{
	CEHResult hr = CE_S_OK;
	if (syncRequest == false)
	{
		hr = WebCore::ResourceHandleManager::redirectCallback(LPCVOID_TO_RESHANDLE(webCoreJob), url);
	}
	return hr;
}

CEHResult CEWebKitFrameWindowImpl::didFailCallback(const void* webCoreJob, INT32 lowLevelError, CEHResult resultCode, bool noContext, bool syncRequest)
{
	CEHResult hr = CE_S_OK;
	if (syncRequest == false)
	{
		hr = WebCore::ResourceHandleManager::errorCallback(LPCVOID_TO_RESHANDLE(webCoreJob), lowLevelError, resultCode, noContext);
	}
	return hr;
}

CEHResult CEWebKitFrameWindowImpl::replyAuthCallback(unsigned long identifier, ICEUString* site, ICEUString* realm, ICEUString* username, ICEUString* password, bool authContinue, bool updateSavedFlag)
{
	CEHResult hr = CE_S_OK;
	hr = WebCore::ResourceHandleManager::replyAuthCallback(
		identifier,
		site,
		realm,
		username,
		password,
		authContinue,
		updateSavedFlag
		);
	return hr;
}

CEHResult CEWebKitFrameWindowImpl::replyBadCertCallback(unsigned long identifier, bool sslContinue)
{
	CEHResult hr = CE_S_OK;
	hr = WebCore::ResourceHandleManager::replyBadCertCallback(identifier, sslContinue);
	return hr;
}

CEHResult CEWebKitFrameWindowImpl::setCertificate(const void* webCoreJob, void* cert)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	WebCore::Page* page = core(this);
	const WebCore::Frame* frame = page ? page->mainFrame() : NULL;
	FrameLoaderClientSilk* client = NULL;
	if (frame && frame->loader())
	{
		client = static_cast<FrameLoaderClientSilk*>(frame->loader()->client());
		if (client)
			hr = client->setCertificate(cert);
	}
	return hr;
}

CEHResult CEWebKitFrameWindowImpl::_privateInit()
{
	CEHResult hr = CE_SILK_ERR_MEMERR;
	_private = new CEWebKitFrameWindowPrivate(this);
	if (_private)
	{
		hr = _private->init(_mainWindow);
	}
	return hr;
}

CEHResult CEWebKitFrameWindowImpl::_privateShutdown()
{
	CEHResult hr = CE_S_OK;
	if (_private)
	{
		hr = _private->shutdown(_mainWindow);
		delete _private;
		_private = NULL;
	}
	return hr;
}

CEHResult CEWebKitFrameWindowImpl::_focusInit(ICEHtmlFocusNavigator* iFocusNavigator)
{
	CEHResult hr = CE_S_OK;

	CEWebKitFocus* focus = new CEWebKitFocus();
	if (focus)
	{
		hr = focus->object()->init(this, iFocusNavigator);
		_iFocusNavigatorPeerRef = CEWebKitFocus::toICEHtmlFocusNavigatorPeer(focus);
	}
	else
	{
		hr = CE_SILK_ERR_MEMERR;
	}

	return hr;
}

CEHResult CEWebKitFrameWindowImpl::_focusShutdown()
{
	CEWebKitFocusImpl* focus = CEWebKitFocus::toSubstance(_iFocusNavigatorPeerRef);
	if (focus)
	{
		focus->shutdown();
	}
	_iFocusNavigatorPeerRef = NULL;
	return CE_S_OK;
}

CEHResult CEWebKitFrameWindowImpl::_mouse(eCEMouseEventType type, CEPointBase& pt, eCEMouseButton button, UINT32 modifiers, bool& consumedOut)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	if (_mainWindow && _mainWindow->object())
	{
		_isFocusNodeChanged_Mouse = type == (eCEMouseEventType_ButtonDown) ? true : false;
		hr = _mainWindow->object()->mouse(type, pt, button, modifiers, consumedOut);
		_isFocusNodeChanged_Mouse = false;

		if ( !hr && type == eCEMouseEventType_ButtonDown)
		{
			WebCore::HitTestResult result = WebCore::HitTestResult(WebCore::IntPoint());
			hr = _hitTestResult(pt._x, pt._y, result);
			if (!hr)
			{
				if (result.scrollbar())
				{
					this->inactivateInputMethodState(false);
				}
				else
				{
					this->updateInputMethodState();
				}
			}
		}
	}
	else if (_lockingOutForMemoryLimit)
	{
		hr = CE_S_OK;
	}
	return hr;
}

CEHResult CEWebKitFrameWindowImpl::_drag(CEPointBase& pt)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	if (_private)
	{
		WebCore::IntPoint pos(pt._x, pt._y);
		WebCore::IntPoint globalPos(pt._x, pt._y);
		WebCore::DragData dragData(NULL, pos, globalPos, WebCore::DragOperationEvery);
		hr = _private->drag(dragData);
	}
	return hr;
}

CEHResult CEWebKitFrameWindowImpl::_scroll(INT32 x, INT32 y, INT32 dx, INT32 dy, UINT32 modifiers, bool& consumedOut)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	if (_mainWindow && _mainWindow->object())
	{
		hr = _mainWindow->object()->scroll(x, y, dx, dy, modifiers, consumedOut);
	}
	else if (_lockingOutForMemoryLimit)
	{
		hr = CE_S_OK;
	}
	return hr;
}

bool CEWebKitFrameWindowImpl::_isDocumentCoordinateMode()
{
	bool result = false;

	WebCore::Page* page = core(this);
	const WebCore::Frame* frame = page ? page->mainFrame() : NULL;
	if (frame->view())
	{
		result = frame->view()->platformWidget() != NULL ? true : false;
	}

	return result;
}

CEHResult CEWebKitFrameWindowImpl::_mouseLButtonUpRepeatJudging(CEPointBase& pt, UINT32 modifiers, bool& consumedOut)
{
	CEHResult hr = CE_S_OK;

	bool judgeFlag = true;
	INT32 count = 0;
	INT32 sign = 1;
	CEPointBase nextPt = pt;
	while(!hr && judgeFlag)
	{					
		hr = _mouse(eCEMouseEventType_ButtonUp, nextPt, eCEMouseButton_Left, modifiers, consumedOut);
		if (!hr)
		{
			WebCore::Page* page = core(this);
			if (page && page->focusController())
			{
				WebCore::FocusController* focusController = page->focusController();
				if (focusController->focusedOrMainFrame() && focusController->focusedOrMainFrame()->document())
				{
					WebCore::Node* focusedNode = focusController->focusedOrMainFrame()->document()->focusedNode();
					if (focusedNode)
					{
						//CEComDebugPrintf("focusNode \n");
						judgeFlag = false;
					}
				}
			}
			//CEComDebugPrintf("_mouseLButtonUpRepeatJudging consumedOut:%d \n", consumedOut);
			count++;
			if (!hr && (consumedOut || count>_hitTestPadding*2 || !judgeFlag))
			{
				judgeFlag = false;
			}
			else
			{
				nextPt._y += sign*count;
				sign*=-1;
				hr = _mouse(eCEMouseEventType_ButtonDown, nextPt, eCEMouseButton_Left, modifiers, consumedOut);
			}
		}
	}
	return hr;
}

CEHResult CEWebKitFrameWindowImpl::scroll(INT32 x, INT32 y, INT32 dx, INT32 dy, UINT32 modifiers, bool* consumedOut)
{
	return _scroll(x, y, dx, dy, modifiers, *consumedOut);
}

CEHResult CEWebKitFrameWindowImpl::resize(INT32 width, INT32 height)
{
	CEHResult hr = CE_SILK_ERR_BADARGS;
    WebCore::Page* page = getPrivate()->getPage();
    WebCore::Frame* frame = page->mainFrame();
	if (frame->view())
	{
		CEWebKitHtmlScrollMode scrollMode;
		hr = _iChromeClientRef.getScrollMode(&scrollMode);
		if (!hr)
		{
			if (scrollMode==CEWebKitHtmlScrollMode_DocumentZoomFactor2)
			{
				frame->view()->setFixedLayoutSize(WebCore::IntSize(width, height));
			}
			frame->view()->resize(width, height);
		}
	}
	_defaultFrameSize = frame->view()->frameRect();
    return hr;
}

CEHResult CEWebKitFrameWindowImpl::keyboard(CEKeyboardEventParam* keyboardEventParam, bool* consumedOut)
{
	CEHResult err = CE_S_OK;
	if (keyboardEventParam)
	{
		bool consumed = false;
		if (_mainWindow && _mainWindow->object())
		{
			err = _mainWindow->object()->keyboard(keyboardEventParam, consumed);
		}
		else if (_lockingOutForMemoryLimit)
		{
			err = CE_S_OK;
		}

		if (!err && consumedOut)
		{
			*consumedOut = consumed;
		}
	}
	return err;
}

bool CEWebKitFrameWindowImpl::handleEditingKeyboardEvent(WebCore::KeyboardEvent* evt)
{
    WebCore::Node* node = evt->target()->toNode();
    ASSERT(node);
    WebCore::Frame* frame = node->document()->frame();
    ASSERT(frame);

    const WebCore::PlatformKeyboardEvent* keyEvent = evt->keyEvent();
    if (!keyEvent/* || keyEvent->isSystemKey()*/)  // do not treat this as text input if it's a system key event
        return false;

	bool shift = evt->shiftKey();
	bool ctrl  = evt->ctrlKey();
	bool alt   = evt->altKey();
	int keyCode = evt->keyCode();

	//TODO: translate special keys to commands. for example, delete, bs, cut&coypy ...
    //WebCore::Editor::Command command = frame->editor()->command(interpretKeyEvent(evt));
	if (evt->type() == WebCore::eventNames().keydownEvent)
	{
		const char* commandStr = 0;

		switch (keyCode)
		{
			case eCEVK_BACK_SPACE:
				commandStr = "DeleteBackward";
				break;
				
			case eCEVK_DELETE:
				commandStr = "DeleteForward";
				break;
				
			case eCEVK_LEFT:
				commandStr = shift ? "MoveLeftAndModifySelection" : "MoveLeft";
				break;
				
			case eCEVK_RIGHT:
				commandStr = shift ? "MoveRightAndModifySelection" : "MoveRight";
				break;
				
			case eCEVK_UP:
				commandStr = shift ? "MoveUpAndModifySelection" : "MoveUp";
				break;
				
			case eCEVK_DOWN:
				commandStr = shift ? "MoveDownAndModifySelection" : "MoveDown";
				break;
				
			case eCEVK_PAGE_UP:
				commandStr = shift ? "MovePageDownAndModifySelection" : "MovePageUp";
				break;
				
			case eCEVK_PAGE_DOWN:
				commandStr = shift ? "MovePageUpAndModifySelection" : "MovePageDown";
				break;
				
			case eCEVK_HOME:
				commandStr = shift ? "MoveToBeginningOfLineAndModifySelection" : "MoveToBeginningOfLine";
				break;
				
			case eCEVK_END:
				commandStr = shift ? "MoveToEndOfLineAndModifySelection" : "MoveToEndOfLine";
				break;
				
			case eCEVK_C:
				if (ctrl)
				{
					this->handleCopyEvent();
				}
				commandStr = 0;
				break;

			case eCEVK_V:
				commandStr = ctrl ? "Paste" : 0;
				break;
				
			case eCEVK_X:
				commandStr = ctrl ? "Cut" : 0;
				break;
				
			case eCEVK_Z:
				commandStr = ctrl ? "Undo" : 0;
				break;
				
			default:break;
		}

		if (commandStr)
		{
			if (frame->editor()->command(commandStr).execute()) return true;
		}
	}
	else if (!ctrl && !alt)
	{
		const char* commandStr = 0;
		switch (keyCode)
		{
			case '\r':
				commandStr = shift ? "InsertLineBreak" : "InsertNewLine";
				break;
			default: break;
		}

		if (commandStr)
		{
			return frame->editor()->command(commandStr).execute();
		}
	
		// Don't insert null or control characters as they can result in unexpected behaviour
		if (keyCode < ' ')
			return false;
		
		return frame->editor()->insertText(evt->keyEvent()->text(), evt);
	}

	return false;
}

CEHResult CEWebKitFrameWindowImpl::insertText(ICEUString* iString, CEUUINT32List* styleList, CEUUINT32List* lengthList, INT32 caretPosition, bool* consumedOut)
{
	CEHResult hr = CE_S_OK;
	WebCore::Page* page = core(this);

	if (_imeTargetPlugin)
	{
		WebCore::PluginView* focusedPluginView = static_cast<WebCore::PluginView*>(_imeTargetPlugin);
		if (focusedPluginView)
		{
			CEComAllocatorRec* strAlc = ::CEComGetAllocatorRec();
			if (strAlc)
			{
				CEComICEUStringRef iStringRef = iString;
				UCHAR8* compositionTextUTF8 = 0;
				UINT32 compositionTextLenUTF8 = 0;
				hr = iStringRef.getBytesWithAlloc(eICEI18nEncoding_utf_8, strAlc, &compositionTextUTF8, &compositionTextLenUTF8);
				if (!hr)
				{
					if (!focusedPluginView->setCompositionText(compositionTextUTF8, compositionTextLenUTF8))
					{	
						hr = CE_SILK_ERR_OPERATION_FAILED;
					}
				}

				if (strAlc && compositionTextUTF8)
				{
					strAlc->free(strAlc, compositionTextUTF8);
					strAlc = 0;
					compositionTextUTF8 = 0;
				}
			}
		}
	}
	else if (_hasInitializedTextInput)
	{
		WebCore::String str;
		hr = str.fromICEUString(iString);
		if (!CESucceeded(hr))
		{
			str.fromUTF8("");
		}

		UINT32 listSize = 0;
		if (styleList)
		{
			listSize = styleList->size();
		}

		if (caretPosition < 0 || str.length() < caretPosition)
		{
			caretPosition = str.length();
		}

		UINT32 index, startPos = 0, endPos = 0;
		Vector<WebCore::CompositionUnderline> underlines;
		WebCore::Color ulColor(0x20,0x20,0xFF);

		for (index=0; index<listSize; index++)
		{
			startPos = endPos;
			endPos = startPos + lengthList->get(index)->getData();
			eCETLIMEStyleAttr style = static_cast<eCETLIMEStyleAttr>(styleList->get(index)->getData());
			if (style == eCETLIMEStyleAttr_Converted || style == eCETLIMEStyleAttr_TargetConverted)
			{
				bool thick = (style == eCETLIMEStyleAttr_TargetConverted) ? true : false;
				WebCore::CompositionUnderline ul(startPos, endPos, ulColor, thick);
				underlines.append(ul);
			}
		}
		page->focusController()->focusedOrMainFrame()->editor()->setComposition(str, underlines, caretPosition, 0);
	}
	return hr;
}

CEHResult CEWebKitFrameWindowImpl::endComposition()
{
	CEHResult hr = CE_S_OK;
	WebCore::Page* page = core(this);
	if (_imeTargetPlugin)
	{
		WebCore::PluginView* focusedPluginView = static_cast<WebCore::PluginView*>(_imeTargetPlugin);
		if (focusedPluginView)
		{
			if (focusedPluginView->endComposition())
			{
				hr = CE_S_OK;
			}
		}
	}
	if (_hasInitializedTextInput)
	{
		page->focusController()->focusedOrMainFrame()->editor()->confirmComposition();
	}
	return hr;
}

CEHResult CEWebKitFrameWindowImpl::clearText()
{
	CEHResult hr = CE_S_OK;
	WebCore::Page* page = core(this);
	if (_imeTargetPlugin)
	{
		WebCore::PluginView* focusedPluginView = static_cast<WebCore::PluginView*>(_imeTargetPlugin);
		if (focusedPluginView)
		{
			if (focusedPluginView->clearComponentText())
			{
				hr = CE_S_OK;
			}
		}
	}
	if (_hasInitializedTextInput)
	{
		WebCore::Frame* frame = page->focusController()->focusedOrMainFrame();
		frame->selection()->selectAll();
		WebCore::Editor* editor = frame->editor();
		editor->deleteSelectionWithSmartDelete(editor->canSmartCopyOrDelete());

	}
	return hr;
}

CEHResult CEWebKitFrameWindowImpl::exitIME()
{
	CEHResult hr = CE_S_OK;
	if (_imeTargetPlugin)
	{
		WebCore::PluginView* focusedPluginView = static_cast<WebCore::PluginView*>(_imeTargetPlugin);
		if (focusedPluginView)
		{
			if (focusedPluginView->exitComponent())
			{
				hr = CE_S_OK;
			}
		}
	}
	if (_hasInitializedTextInput)
	{
		WebCore::Page* page = core(this);
		if (page)
		{
			WebCore::FocusController* focusController = page->focusController();
			if (focusController->focusedOrMainFrame())
			{
				focusController->setFocusedNode(0, focusController->focusedOrMainFrame());
			}
		}
	}
	return hr;
}

CEHResult CEWebKitFrameWindowImpl::setSelectedItemsPopupMenu(INT32 selectedIndex, void *popup)
{
	CEHResult hr = CE_S_OK;

	WebCore::PopupMenu *popupMenu = reinterpret_cast<WebCore::PopupMenu *>(popup);
	WebCore::PopupMenuClient *popupMenuClient = popupMenu->client();

	if ( 0<= selectedIndex &&
		selectedIndex < popupMenuClient->listSize() &&
		( !popupMenuClient->itemIsSeparator(selectedIndex) &&
		  !popupMenuClient->itemIsLabel(selectedIndex) &&
		  popupMenuClient->itemIsEnabled(selectedIndex) ) )
	{
		popupMenuClient->valueChanged(selectedIndex);
	}
	popupMenuClient->popupDidHide();
	popupMenu->selected();

	return hr;
}

CEHResult CEWebKitFrameWindowImpl::shrinkHistory(UINT32 minimumSize)
{
	CEHResult hr = CE_S_OK;

	if (_mainWindow && _mainWindow->object())
	{
		hr = _mainWindow->object()->shrinkHistory(minimumSize);
	}

	return hr;
}

CEHResult CEWebKitFrameWindowImpl::setCustomTextEncodingName(ICEUString* iTitle)
{
	CEHResult hr = CE_S_OK;

	if (_mainWindow && _mainWindow->object())
	{
		hr = _mainWindow->object()->setCustomTextEncodingName(iTitle);
	}

	return hr;
}

CEHResult CEWebKitFrameWindowImpl::getSettings(ICEWebKitSettings** iSettingsOut)
{
	CEHResult hr = CE_S_OK;
	CEWebKitFrameWindowPrivate* pPrivate = getPrivate();
	if (pPrivate)
	{
		WebCore::Page* pPage = pPrivate->getPage();
		if (pPage)
		{
			CEWebKitSettings* s = new CEWebKitSettings(pPage->settings());
			if (s)
			{
				CEComICEWebKitSettingsRef ref = s->toICEWebKitSettings(s);
				*iSettingsOut = ref.detach();
			}
			else
			{
				hr = CE_SILK_ERR_MEMERR;
			}
		}
		else
		{
			hr =  CE_SILK_ERR_UNINITIALIZED;
		}
	}
	else
	{
		hr =  CE_SILK_ERR_UNINITIALIZED;
	}
	return hr;
}

CEHResult CEWebKitFrameWindowImpl::stop()
{
	CEHResult hr = CE_S_OK;

	if (_mainWindow && _mainWindow->object())
	{
		hr = _mainWindow->object()->stop();
	}

	return hr;
}

CEHResult CEWebKitFrameWindowImpl::emptyPage()
{
    _lockingOutForMemoryLimit = true;
    
	CEHResult hr = _privateShutdown();

	if (CESucceeded(hr))
	{
		WebCore::pageCache()->releaseAutoreleasedPagesNow();
		WebCore::gcController().garbageCollectNow();
	
		CEWebKitImpl* webKitImpl = CEWebKitImpl::getInstance();
		if (webKitImpl)
		{
			webKitImpl->pruneCache();
			webKitImpl->clearLockingOutInLowMemory();
		}
	}

	if (CESucceeded(hr))
	{
		hr = _privateInit();
	}

	if (CESucceeded(hr) && _iChromeClientRef && getPrivate())
	{
		bool canBack = false, canForward = false;
		WebCore::Page* page = getPrivate()->getPage();
		if (page && page->backForwardList())
		{
			canBack = (page->backForwardList()->backItem() != NULL);
			canForward = (page->backForwardList()->forwardItem() != NULL);
		}
		hr = _iChromeClientRef.setCanBackAndForward(canBack, canForward);
		if (CESucceeded(hr))
		{
			hr = _iChromeClientRef.progressFinishedNotification();
		}
	}

	return hr;
}

CEHResult CEWebKitFrameWindowImpl::isEmpty(bool* isEmptyOut)
{
	CEHResult hr = CE_S_OK;

    if (isEmptyOut)
    {
        if (_lockingOutForMemoryLimit)
        {
            *isEmptyOut = true;
            _lockingOutForMemoryLimit = false;
        }
        else
        {
            *isEmptyOut = false;
        }
        hr = CE_S_OK;
	}
    else
    {
        hr = CE_SILK_ERR_BADARGS;
    }

	return hr;
}

CEHResult CEWebKitFrameWindowImpl::setAllowsScrolling(bool visible)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	if (_private)
	{
		WebCore::Page* page = _private->getPage();
		if (page)
		{
			WebCore::Frame* frame = page->mainFrame();
			if (frame)
			{	
				WebCore::FrameView* view = frame->view();
				if (view)
				{
					view->setCanHaveScrollbars(visible);
					hr = CE_S_OK;
				}
			}
		}
	}
	return hr;
}

CEHResult CEWebKitFrameWindowImpl::getContentsSize(INT32* widthOut, INT32* heightOut)
{	
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	if (_mainWindow && _mainWindow->object())
	{
		hr = _mainWindow->object()->getContentsSize(widthOut, heightOut);
	}
	return hr;
}
CEHResult CEWebKitFrameWindowImpl::setVisibleRect(INT32 x, INT32 y, INT32 width, INT32 height)
{
	CEHResult hr = CE_SILK_ERR_BADSTATE;
	if (_private)
	{
		WebCore::Page* page = _private->getPage();
		if (page)
		{
			WebCore::Frame* frame = page->mainFrame();
			if (frame)
			{	
				WebCore::FrameView* view = frame->view();
				if (view)
				{
					WebCore::IntPoint visiblePoint(x, y);
					CEWebKitHtmlScrollMode scrollMode;
					hr = _iChromeClientRef.getScrollMode(&scrollMode);
					if (!hr)
					{
						if (scrollMode==CEWebKitHtmlScrollMode_DocumentZoomFactor2)
						{
							view->setActualVisibleContentRect(WebCore::IntRect(x,y,width,height));
						}
						// for Bug 18432 when Platform changed scroll position do not notify scroll position.
						_skipNotifyScrollPosition = true;
						view->setScrollPosition(visiblePoint);
					}
					hr = CE_S_OK;
				}
			}
		}
	}
	return hr;
}
CEHResult CEWebKitFrameWindowImpl::setPageSizeMultiplier(float multiplier)
{
	CEHResult hr = CE_SILK_ERR_BADSTATE;
	WebCore::Page* page = getPrivate()->getPage();
	CEU_VALIDATE_PTR(page);
	WebCore::Frame* frame = page->mainFrame();
	CEU_VALIDATE_PTR(frame);
	bool b = false;
	hr = getZoomTextOnly(&b);
	CEU_ENSURE_SUCCESS1(hr);
	frame->setZoomFactor(multiplier, b);
	return CE_S_OK;
}

// FIXME: This code should move into WebCore so it can be shared by all the WebKits.
#define MinimumZoomMultiplier   0.5f
#define MaximumZoomMultiplier   3.0f
#define ZoomMultiplierRatio     1.2f

CEHResult CEWebKitFrameWindowImpl::getPageSizeMultiplier(float *const multiplierOut)
{
	WebCore::Page* page = getPrivate()->getPage();
	CEU_VALIDATE_PTR(page);
	WebCore::Frame* frame = page->mainFrame();
	CEU_VALIDATE_PTR(frame);
	*multiplierOut = frame->zoomFactor();
	return CE_S_OK;
}

CEHResult CEWebKitFrameWindowImpl::canZoomPageIn(bool *const result)
{
	CEU_VALIDATE_PTR(result);
	float multipier = 0.0;
	getPageSizeMultiplier(&multipier);
	*result = (multipier * ZoomMultiplierRatio) < MaximumZoomMultiplier;
	return CE_S_OK;
}

CEHResult CEWebKitFrameWindowImpl::zoomPageIn()
{
	bool canZoomIn = false;
	CEHResult hr = canZoomPageIn(&canZoomIn);
	CEU_ENSURE_SUCCESS1(hr);
	if (canZoomIn)
	{
		float multiplier = 0.0;
		hr = getPageSizeMultiplier(&multiplier);
		if (CESucceeded(hr))
		{
			hr = setPageSizeMultiplier(multiplier * ZoomMultiplierRatio);
		}
	}
	return hr;
}

CEHResult CEWebKitFrameWindowImpl::canZoomPageOut(bool *const result)
{
	CEU_VALIDATE_PTR(result);
	float multipier = 0.0;
	CEHResult hr = getPageSizeMultiplier(&multipier);
	CEU_ENSURE_SUCCESS1(hr);
	*result = multipier / ZoomMultiplierRatio > MinimumZoomMultiplier;
	return hr;
}

CEHResult CEWebKitFrameWindowImpl::zoomPageOut()
{
	bool canZoomOut = false;
	CEHResult hr = canZoomPageOut(&canZoomOut);
	CEU_ENSURE_SUCCESS1(hr);
	if (canZoomOut)
	{
		float multiplier = 0.0;
		hr = getPageSizeMultiplier(&multiplier);
		if (CESucceeded(hr))
		{
			hr = setPageSizeMultiplier(multiplier / ZoomMultiplierRatio);
		}
	}
	return hr;
}

CEHResult CEWebKitFrameWindowImpl::canResetPageZoom(bool *const result)
{
	CEU_VALIDATE_PTR(result);
	float multipier = 0.0;
	CEHResult hr = getPageSizeMultiplier(&multipier);
	CEU_ENSURE_SUCCESS1(hr);
	*result = multipier != 1.0;
	return hr;
}

CEHResult CEWebKitFrameWindowImpl::resetPageZoom()
{
	bool canZoom = false;
	CEHResult hr = canResetPageZoom(&canZoom);
	if (canZoom)
		hr = setPageSizeMultiplier(1.0);
	return hr;
}

CEHResult CEWebKitFrameWindowImpl::getZoomTextOnly(bool *const resultOut)
{
	CEU_VALIDATE_PTR(resultOut);
	CEWebKitFrameWindowPrivate* privateFrameWin = getPrivate();
	CEU_VALIDATE_PTR(privateFrameWin);
	WebCore::Page* page = privateFrameWin->getPage();
	CEU_VALIDATE_PTR(page);
	WebCore::Settings* settings = page->settings();
	CEU_VALIDATE_PTR(settings);
	*resultOut = settings->zoomsTextOnly();
	return CE_S_OK;
}

CEHResult CEWebKitFrameWindowImpl::setZoomTextOnly(bool bZoomTextOnly)
{
	bool b = false;
	CEHResult hr = getZoomTextOnly(&b);
	CEU_ENSURE_SUCCESS1(hr);
	if (bZoomTextOnly != b)
	{
		CEWebKitFrameWindowPrivate* privateFrameWin = getPrivate();
		CEU_VALIDATE_PTR(privateFrameWin);
		WebCore::Page* page = privateFrameWin->getPage();
		CEU_VALIDATE_PTR(page);
		WebCore::Settings* settings = page->settings();
		CEU_VALIDATE_PTR(settings);
		settings->setZoomsTextOnly(bZoomTextOnly);
	}
	return CE_S_OK;
}

CEWebKitFrameWindowPrivate::CEWebKitFrameWindowPrivate(CEWebKitFrameWindowImpl* silkPeer) :
	_silkPeer(silkPeer),
	_page(NULL),
	_scroll_mode_h(WebCore::ScrollbarAuto),
	_scroll_mode_v(WebCore::ScrollbarAuto)
#if USE(ACCELERATED_COMPOSITING)
	, _renderer(0)
	, _syncTimer(this)
#endif //#if USE(ACCELERATED_COMPOSITING)
{
}

CEHResult CEWebKitFrameWindowPrivate::init(CEWebKitWindow*& mainWindow)
{
	CEHResult hr = CE_S_OK;
	if (CESucceeded(hr))
	{
		hr = _initPage();
	}
	if (CESucceeded(hr))
	{
		hr = _initMainFrame(mainWindow);
	}
	if (CESucceeded(hr))
	{
		CEASSERT(_page);
		if (_page && _page->focusController())
		{
			_page->focusController()->setActive(true);
			_page->focusController()->setFocused(true);
		}
	}
	if (CESucceeded(hr))
	{
		CEWebKitHtmlScrollMode scrollMode;
		hr = _silkPeer->getChromeClient().getScrollMode(&scrollMode);
		if (!hr)
		{
			if (scrollMode==CEWebKitHtmlScrollMode_DocumentZoomFactor2)
			{
				_scroll_mode_h = WebCore::ScrollbarAlwaysOff;
				_scroll_mode_v = WebCore::ScrollbarAlwaysOff;
			}
		}
	}
	return hr;
}

CEHResult CEWebKitFrameWindowPrivate::shutdown(CEWebKitWindow*& mainWindow)
{
	CEHResult hr = CE_S_OK;
	if (CESucceeded(hr))
	{
		hr = _shutdownMainFrame(mainWindow);
	}
	if (CESucceeded(hr))
	{
		hr = _shutdownPage();
	}

#if USE(ACCELERATED_COMPOSITING)

	invalidateExceptForLayersBackingStore();
	_renderer = 0;
#endif //#if USE(ACCELERATED_COMPOSITING)

	return hr;
}

CEHResult CEWebKitFrameWindowPrivate::drag(WebCore::DragData& data)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	if (_page && _page->dragController())
	{
		_page->dragController()->dragUpdated(&data);
		hr = CE_S_OK;
	}
	return hr;
}

CEHResult CEWebKitFrameWindowPrivate::_initPage()
{
	CEHResult hr = CE_SILK_ERR_MEMERR;

	ChromeClientSilk* chromeClient = new ChromeClientSilk(_silkPeer);
	ContextMenuClientSilk* contextMenuClient = new ContextMenuClientSilk();
	EditorClientSilk* editorClient = new EditorClientSilk(_silkPeer);
	DragClientSilk* dragClient = new DragClientSilk();
	InspectorClientSilk* inspectorClient = new InspectorClientSilk();
	if (chromeClient && contextMenuClient && editorClient && inspectorClient)
	{
		_page = new WebCore::Page(chromeClient, contextMenuClient, editorClient, dragClient, inspectorClient);
		if (_page)
		{
			hr = CE_S_OK;
			CEWebKitImpl* webKitImpl = CEWebKitImpl::getInstance();
			if (webKitImpl)
			{
				CEComICEHtmlWebKitEnvironmentRef iBrowserEnvironmentRef = webKitImpl->getWebKitEnvironment();
				if (iBrowserEnvironmentRef)
				{
					bool enablePlugin = false;
					hr = iBrowserEnvironmentRef.getEnablePlugin(&enablePlugin);
					if (!hr)
					{
						// todo propagate preference
						WebCore::Settings* settings = _page->settings();
						if (settings)
						{
							settings->setLoadsImagesAutomatically(true);
							settings->setDefaultFixedFontSize(13);
							settings->setDefaultFontSize(16);
							settings->setSerifFontFamily("serif");
							settings->setFixedFontFamily("monospace");
							settings->setSansSerifFontFamily("sans-serif");
							settings->setStandardFontFamily("default");
							settings->setCursiveFontFamily("cursive");
							settings->setFantasyFontFamily("fantasy");
							settings->setJavaScriptEnabled(true);
							settings->setUsesPageCache(true);
							settings->setAllowScriptsToCloseWindows(true);
							settings->setPluginsEnabled(enablePlugin);
							if (enablePlugin)
								hr = webKitImpl->preparePluginInfo();

			#if USE(ACCELERATED_COMPOSITING)
							// TODO: connect UI by using CEWebKitSettings.
							{
								bool bAcceleratedCompositingEnabled = false;

								CEComICEMRLLayerRendererRef t;
								CEHResult r = WebCore::GraphicsLayerSilk::Env::getMRLFactory().createLayerRenderer(&t);
								if (CESucceeded(r))
									bAcceleratedCompositingEnabled = (t != 0);
								
								// for debugging.
								// bAcceleratedCompositingEnabled = false;
								settings->setAcceleratedCompositingEnabled(bAcceleratedCompositingEnabled);
							}
			#endif //#if USE(ACCELERATED_COMPOSITING)
						}
					}
				}
			}
		}
	}

	if (CEFailed(hr))
	{
		delete chromeClient;
		delete contextMenuClient;
		delete editorClient;
		delete dragClient;
		delete inspectorClient;
		delete _page;
		_page = NULL;
	}

	return hr;
}

CEHResult CEWebKitFrameWindowPrivate::_initMainFrame(CEWebKitWindow*& mainWindow)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;

	if (_silkPeer)
	{
		hr = _initScrollMode();
		if (CESucceeded(hr))
		{
			CEWebKitWindow* window = new CEWebKitWindow();
			if (window && window->object())
			{
				hr = window->object()->init(_page, NULL);
				if (CESucceeded(hr))
				{
					mainWindow = window;
				}
			}
			else
			{
				hr = CE_SILK_ERR_MEMERR;
			}
		}
	}
	return hr;
}

CEHResult CEWebKitFrameWindowPrivate::_shutdownPage()
{
	if (_page)
	{
		delete _page;
		_page = NULL;
	}
	return CE_S_OK;
}

CEHResult CEWebKitFrameWindowPrivate::_shutdownMainFrame(CEWebKitWindow*& mainWindow)
{
	CEHResult hr = CE_S_OK;
	if (mainWindow && mainWindow->object())
	{
		hr = mainWindow->object()->shutdown();
		delete mainWindow;
		mainWindow = NULL;
	}
	return hr;
}

CEHResult CEWebKitFrameWindowPrivate::_initScrollMode()
{
	CEHResult hr = CE_S_OK;

	// Get scroll preference
	CEWebKitHtmlFrameScrolling scroll_preference_h;
	CEWebKitHtmlFrameScrolling scroll_preference_v;

	hr = _silkPeer->getWidget().getScrollbarPreference(&scroll_preference_h, &scroll_preference_v);
	if (CESucceeded(hr))
	{
		//Horizontal
		switch (scroll_preference_h)
		{
		case CEWebKitHtmlFrameScrolling_Horizontal_Auto:
			_scroll_mode_h = WebCore::ScrollbarAuto;
			hr = CE_S_OK;
			break;
		case CEWebKitHtmlFrameScrolling_Horizontal_Yes:
			_scroll_mode_h = WebCore::ScrollbarAlwaysOn;
			hr = CE_S_OK;
			break;
		case CEWebKitHtmlFrameScrolling_Horizontal_No:
			_scroll_mode_h = WebCore::ScrollbarAlwaysOff;
			hr = CE_S_OK;
			break;
		default:
			hr = CE_SILK_ERR_BADSTATE;
			break;
		}

		if (CESucceeded(hr))
		{
			//Vertical
			switch (scroll_preference_v)
			{
			case CEWebKitHtmlFrameScrolling_Vertical_Auto:
				_scroll_mode_v = WebCore::ScrollbarAuto;
				hr = CE_S_OK;
				break;
			case CEWebKitHtmlFrameScrolling_Vertical_Yes:
				_scroll_mode_v = WebCore::ScrollbarAlwaysOn;
				hr = CE_S_OK;
				break;
			case CEWebKitHtmlFrameScrolling_Vertical_No:
				_scroll_mode_v = WebCore::ScrollbarAlwaysOff;
				hr = CE_S_OK;
				break;
			default:
				hr = CE_SILK_ERR_BADSTATE;
				break;
			}
		}
	}

	return hr;
}

CEWebKitFrameWindowImpl* kit(WebCore::Page* page)
{
	CEWebKitFrameWindowImpl* kit = NULL;
	if (page && page->chrome() && page->chrome()->client())
	{
		kit = static_cast<ChromeClientSilk*>(page->chrome()->client())->getFrameWindow();
	}
	return kit;
}

WebCore::Page* core(CEWebKitFrameWindowImpl* frameWindow)
{
	WebCore::Page* core = NULL;
	if (frameWindow && frameWindow->getPrivate())
	{
		core = frameWindow->getPrivate()->getPage();
	}
	return core;
}

#include "Node.h"
#include "RenderObject.h"
#include "RenderText.h"
#include "RenderView.h"
#include "HTMLNames.h"
#include "InlineTextBox.h"

static WebCore::IntRect _getAbsoluteClippedOverflowRect(WebCore::RenderObject* renderer)
{
	WebCore::IntRect outerRect = renderer->absoluteClippedOverflowRect();
	WebCore::RenderView* view = renderer->view();
	if (view && view->maximalOutlineSize() > 0)
	{
		outerRect.inflate(-(view->maximalOutlineSize()));
	}
	return outerRect;
}

CEHResult CEWebKitFrameWindowImpl::hitTest(INT32 x, INT32 y, ICEURL** linkUrlOut, ICEURL** imageUrlOut, ICEUString** text, CERect* zoomRect)
{
	CEHResult err = CE_SILK_ERR_OPERATION_FAILED;
	WebCore::HitTestResult result = WebCore::HitTestResult(WebCore::IntPoint());
	err = _hitTestResult(x, y, result);

	if (!err)
	{
		if (!result.innerNode())
		{
			return err;
		}
		err = CE_S_OK;
		WebCore::IntRect boxRect(0,0,0,0);
		WebCore::RenderObject* pRenderObject = result.innerNode()->renderer();
		if (pRenderObject)
		{
			if (pRenderObject->isText())
			{
				WebCore::FloatPoint absPos = pRenderObject->localToAbsolute();
				boxRect.move(absPos.x(), absPos.y());
				WebCore::RenderText* rt = static_cast<WebCore::RenderText*>(pRenderObject);
				const WebCore::InlineTextBox* itb = rt->firstTextBox();
				while (itb)
				{
					WebCore::IntRect itbRect(absPos.x() + itb->x(), absPos.y() + itb->y(), itb->width(), itb->height());
					boxRect.unite(itbRect);

					if (itb == rt->lastTextBox()){ break; }
					itb = itb->nextTextBox();
				}

				WebCore::IntRect outerRect = _getAbsoluteClippedOverflowRect(pRenderObject);
				if (!outerRect.contains(boxRect))
				{
					boxRect.intersect(outerRect);
				}
			}
			else
			{
				boxRect = pRenderObject->containingBlock()->absoluteContentBox();
			}
		}
		else
		{
			boxRect =  result.innerNode()->getRect();
		}
		
		if (_isDocumentCoordinateMode())
		{
			*zoomRect = boxRect;
		}
		else
		{
			*zoomRect = result.innerNode()->document()->frame()->view()->contentsToWindow(boxRect);
		}

		if (!result.absoluteLinkURL().isEmpty()) 
		{
			err = result.absoluteLinkURL().createICEURL(linkUrlOut);
			if (CEFailed(err))
				return err;
		}
		if (!result.absoluteImageURL().isEmpty())
		{
			err = result.absoluteImageURL().createICEURL(imageUrlOut);
			if (CEFailed(err))
				return err;
		}

		WebCore::Node* node = result.innerNode();
		while (node)
		{
			// for imagemaps, URLElement is the associated area element not the image itself
			if (node->hasTagName(WebCore::HTMLNames::aTag))
			{
				WebCore::String w_text(static_cast<WebCore::HTMLAnchorElement*>(node)->text());
				err = w_text.createICEUString(text);
				break;
			}
			node = node->eventParentNode();
		}
	}
	return err;
}

void CEWebKitFrameWindowImpl::rebuildFocusNavigation()
{
	CEWebKitFocusImpl* focus = CEWebKitFocus::toSubstance(_iFocusNavigatorPeerRef);
	if (focus)
		focus->rebuildFocusNavigation();
}

static CEWebKitHtmlInputType getCEWebKitHtmlInputTypeFromWebCoreInputType(WebCore::HTMLInputElement::InputType coreType)
{
#define getCEWebKitHtmlInputTypeFromWebCoreInputType_convert(c) ((CEWebKitHtmlInputType)((int)(c)+1))
#define getCEWebKitHtmlInputTypeFromWebCoreInputType_check(t) ASSERT(getCEWebKitHtmlInputTypeFromWebCoreInputType_convert(WebCore::HTMLInputElement::t) == CEWebKitHtmlInputType_ ## t)

	ASSERT(WebCore::HTMLInputElement::TEXT <= coreType && coreType <= WebCore::HTMLInputElement::URL);

	getCEWebKitHtmlInputTypeFromWebCoreInputType_check(TEXT);
	getCEWebKitHtmlInputTypeFromWebCoreInputType_check(PASSWORD);
	getCEWebKitHtmlInputTypeFromWebCoreInputType_check(ISINDEX);
	getCEWebKitHtmlInputTypeFromWebCoreInputType_check(CHECKBOX);
	getCEWebKitHtmlInputTypeFromWebCoreInputType_check(RADIO);
	getCEWebKitHtmlInputTypeFromWebCoreInputType_check(SUBMIT);
	getCEWebKitHtmlInputTypeFromWebCoreInputType_check(RESET);
	getCEWebKitHtmlInputTypeFromWebCoreInputType_check(FILE);
	getCEWebKitHtmlInputTypeFromWebCoreInputType_check(HIDDEN);
	getCEWebKitHtmlInputTypeFromWebCoreInputType_check(IMAGE);
	getCEWebKitHtmlInputTypeFromWebCoreInputType_check(BUTTON);
	getCEWebKitHtmlInputTypeFromWebCoreInputType_check(SEARCH);
	getCEWebKitHtmlInputTypeFromWebCoreInputType_check(RANGE);
	getCEWebKitHtmlInputTypeFromWebCoreInputType_check(EMAIL);
	getCEWebKitHtmlInputTypeFromWebCoreInputType_check(NUMBER);
	getCEWebKitHtmlInputTypeFromWebCoreInputType_check(TELEPHONE);
	getCEWebKitHtmlInputTypeFromWebCoreInputType_check(URL);

	return getCEWebKitHtmlInputTypeFromWebCoreInputType_convert(coreType);
}

void CEWebKitFrameWindowImpl::updateInputMethodState()
{
	WebCore::Page* page = core(this);

	bool comittedLoad = false;

	CEWebKitHtmlScrollMode scrollMode;
	CEHResult err = _iChromeClientRef.getScrollMode(&scrollMode);
	if (!err)
	{
		if (scrollMode==CEWebKitHtmlScrollMode_DocumentZoomFactor2)
		{	
			if (page->mainFrame() && page->mainFrame()->loader())
			{
				FrameLoaderClientSilk* client = static_cast<FrameLoaderClientSilk*>(page->mainFrame()->loader()->client());
				if (client)
				{
					comittedLoad = client->getComittedLoadNow();
				}
			}
		}
	}
		
	if (!_isFocusNodeChanged_Mouse && page && page->focusController() && !comittedLoad)
	{
		WebCore::FocusController* focusController = page->focusController();
		if (focusController->focusedOrMainFrame() && focusController->focusedOrMainFrame()->document())
		{
			bool nextIsTextInput = false;
			WebCore::Node* focusedNode = focusController->focusedOrMainFrame()->document()->focusedNode();
			if (focusedNode)
			{
				bool isTextField = false;
				bool isPassword = false;
				bool isSingleLine = true;
				bool readOnly = false;
				WebCore::String value;
				INT32 maxLength = -1;
				CEWebKitHtmlInputType inputType = CEWebKitHtmlInputType_Undefined;
				int caretPos = -1;

				if (focusedNode->hasTagName(WebCore::HTMLNames::inputTag))
				{
					WebCore::HTMLInputElement* inputElement = static_cast<WebCore::HTMLInputElement*>(focusedNode);
					isTextField = inputElement->isTextField();
					isPassword = inputElement->isPasswordField();
					maxLength = inputElement->maxLength();
					readOnly = inputElement->readOnly();
					inputType = getCEWebKitHtmlInputTypeFromWebCoreInputType(inputElement->inputType());
					if (isTextField && !readOnly && inputElement->renderer())
					{
						WebCore::RenderTextControlSingleLine* renderTextControl = static_cast<WebCore::RenderTextControlSingleLine*>(inputElement->renderer());
						value = renderTextControl->text();
						caretPos = inputElement->selectionStart();

						if (caretPos == 0)
						{
							inputElement->updateFocusAppearance(true);
							caretPos = inputElement->selectionStart();
						}
					}
				}
				else if (focusedNode->hasTagName(WebCore::HTMLNames::textareaTag))
				{
					WebCore::HTMLTextAreaElement* textAreaElement = static_cast<WebCore::HTMLTextAreaElement*>(focusedNode);
					readOnly = textAreaElement->readOnly();
					if (!readOnly && textAreaElement->renderer())
					{
						WebCore::RenderTextControlMultiLine* renderTextControl = static_cast<WebCore::RenderTextControlMultiLine*>(textAreaElement->renderer());
						value = renderTextControl->text();
						caretPos = textAreaElement->selectionStart();

						if (caretPos == 0)
						{
							textAreaElement->updateFocusAppearance(true);
							caretPos = textAreaElement->selectionStart();
						}
					}

					isTextField = true;
					isSingleLine = false;
				}

				nextIsTextInput = isTextField && !readOnly;
				if (nextIsTextInput)
				{
					if (_focusedTextInputNode != focusedNode)
					{
						inactivateInputMethodState(true);

						WebCore::IntRect rect = focusedNode->getRect();
						if (!_isDocumentCoordinateMode())
						{
							rect = focusController->focusedOrMainFrame()->view()->contentsToWindow(rect);
						}

						const UTF16CHAR* strData = reinterpret_cast<const UTF16CHAR*>(value.characters());
						UINT32 strLen = static_cast<UINT32>(value.length());
						_hasInitializedTextInput = true;

						getWidget().initializeTextInput(isPassword, isSingleLine, inputType, strData, strLen, maxLength, 0, rect.x(), rect.y(), rect.width(), rect.height(), false);
						_focusedTextInputNode = focusedNode;
					}
					getWidget().setCaretPosition(caretPos);
				}
			}

			if (!nextIsTextInput)
			{
				inactivateInputMethodState(false);
			}
		}
	}
}

CEHResult CEWebKitFrameWindowImpl::getPage(void** newPageOut)
{
	CEHResult hr = CE_SILK_ERR_BADARGS;
	if (newPageOut && _private)
	{
		*newPageOut = reinterpret_cast<void*>(_private->getPage());
		hr = CE_S_OK;
	}
	return hr;
}

void CEWebKitFrameWindowImpl::inactivateInputMethodState(bool nextIsUseInputMethod)
{
	if (_hasInitializedTextInput)
	{
		// If editor has compostion string, confirm it.
		WebCore::Page* page = core(this);
		if (page)
		{
			page->focusController()->focusedOrMainFrame()->editor()->clear();
		}

		getWidget().shutdownTextInput(nextIsUseInputMethod);
		_hasInitializedTextInput = false;
		_focusedTextInputNode = NULL;
	}
}

CEHResult CEWebKitFrameWindowImpl::getDeviceScreenRect(CERect* const deviceScreenRectOut)
{
	return _iChromeClientRef.getDeviceScreenRect(deviceScreenRectOut);
}

CEHResult CEWebKitFrameWindowImpl::setScrollPosition(INT32 x, INT32 y)
{
	CEHResult err = CE_S_OK;
	// for Bug 18432 when Platform changed scroll position do not notify scroll position.
	if (_skipNotifyScrollPosition)
	{
		_skipNotifyScrollPosition = false;
	}
	else
	{
		err = _iChromeClientRef.setScrollPosition(x, y);
	}
	return err;
}

CEHResult CEWebKitFrameWindowImpl::notifyChangeContentSize(INT32 width, INT32 height, INT32 fixedLayoutWidth, INT32 fixedLayoutHeight, float zoomFactor)
{
	CEHResult err = CE_SILK_ERR_BADSTATE;
	if (_iChromeClientRef)
	{
		err = _iChromeClientRef.notifyChangeContentSize(width, height, fixedLayoutWidth, fixedLayoutHeight, zoomFactor);
	}
	return err;
}

////////////////////////////////////////
// ICEWebKitCopyPasteEventListener
////////////////////////////////////////
CEHResult CEWebKitFrameWindowImpl::handleCopyEvent()
{
	CEHResult err = CE_SILK_ERR_OPERATION_FAILED;

	WebCore::Page* page = core(this);
	WebCore::Frame* frame = page->focusController()->focusedOrMainFrame();
	
	frame->editor()->copy();
	if (frame->editor()->canCopy())
	{
		WebCore::String str = WebCore::Pasteboard::generalPasteboard()->plainText();
		if (0 < str.length())
		{
			CEComICEUStringRef iStringRef;
			err = ICEUStringCreate(CEComStdClassID_CEUString, iStringRef);
			if (!err)
			{
				err = iStringRef.initWithUTF16Array(str.charactersSilk(), str.length() * sizeof(UTF16CHAR));
				if (!err)
				{
					CEComICEHtmlClipboardRef iHtmlClipboardRef;
					err = iHtmlClipboardRef.initByQueryInterface(_iChromeClientRef);
					if (!err)
					{
						err = iHtmlClipboardRef.setString(iStringRef);
					}
				}
			}
		}
	}

	return err;
}

CEHResult CEWebKitFrameWindowImpl::handlePasteEvent()
{
	return CE_SILK_ERR_NOTIMPLEMENTED;
}

CEHResult CEWebKitFrameWindowImpl::confirmPlugin(bool result, bool keepResult, void *const cData)
{
	CEHResult hr = CE_S_OK;

	if (_mainWindow && _mainWindow->object())
	{
		hr = _mainWindow->object()->confirmPlugin(result, keepResult, cData);
	}

	return hr;
}

CEHResult CEWebKitFrameWindowImpl::replyNetworkLibraryError(bool result, ICEUString* errorPageStr)
{
	CEWebKitWindowImpl* webkitWindowImpl = CEWebKitWindow::toSubstance(CEWebKitWindow::toICEWebKitWindow(_mainWindow));

	if (webkitWindowImpl)
		webkitWindowImpl->networkErrorEventPost(result, errorPageStr);

	return CE_S_OK;
}

CEHResult CEWebKitFrameWindowImpl::setHitTestPadding(INT32 docPadding)
{
	_hitTestPadding = docPadding;
	return CE_S_OK;
}

CEHResult CEWebKitFrameWindowImpl::_hitTestResult(INT32 x, INT32 y, WebCore::HitTestResult& resultOut)
{
	WebCore::Page* page = core(this);
	CEHResult err = CE_SILK_ERR_OPERATION_FAILED;
    const WebCore::Frame* frame = page->mainFrame();

    if (frame->view())
	{
		bool isDocumentCoordinateMode = _isDocumentCoordinateMode();
		WebCore::IntPoint hitPoint;
		bool allowShadowContent = false;
		bool ignoreClipping = false;

		if (isDocumentCoordinateMode)
		{
			hitPoint = WebCore::IntPoint(x,y);
			allowShadowContent = true;
			ignoreClipping = true;
		}
		else
			hitPoint = frame->view()->windowToContents(WebCore::IntPoint(x, y));

		resultOut = frame->eventHandler()->hitTestResultAtPoint(hitPoint, allowShadowContent, ignoreClipping);
		err = CE_S_OK;
	}

	return err;
}

CEHResult CEWebKitFrameWindowImpl::_getElementRect(WebCore::Node* node, CERect* rectOut)
{
	CEHResult err = CE_SILK_ERR_BADARGS;
	if (node)
	{
		WebCore::IntRect boxRect(0,0,0,0);
		WebCore::RenderObject* pRenderObject = node->renderer();
		if (pRenderObject)
		{
			if (pRenderObject->isText())
			{
				WebCore::FloatPoint absPos = pRenderObject->localToAbsolute();
				boxRect.move(absPos.x(), absPos.y());
				WebCore::RenderText* rt = static_cast<WebCore::RenderText*>(pRenderObject);
				const WebCore::InlineTextBox* itb = rt->firstTextBox();
				while (itb)
				{
					WebCore::IntRect itbRect(absPos.x() + itb->x(), absPos.y() + itb->y(), itb->width(), itb->height());
					boxRect.unite(itbRect);

					if (itb == rt->lastTextBox()){ break; }
					itb = itb->nextTextBox();
				}

				WebCore::IntRect outerRect = _getAbsoluteClippedOverflowRect(pRenderObject);
				if (!outerRect.contains(boxRect))
				{
					boxRect.intersect(outerRect);
				}
			}
			else
			{
				boxRect = _getAbsoluteClippedOverflowRect(pRenderObject);
				// take measure against IR(image replacement) 
				WebCore::FloatPoint absPos = pRenderObject->localToAbsolute();
				boxRect.setWidth(boxRect.x() + boxRect.width() - absPos.x());
				boxRect.setHeight(boxRect.y() + boxRect.height() - absPos.y());
				boxRect.setX(absPos.x());
				boxRect.setY(absPos.y());
			}
		}
		else
		{
			boxRect =  node->getRect();
		}
		
		if (_isDocumentCoordinateMode())
		{
			*rectOut = boxRect;
		}
		else
		{
			*rectOut = node->document()->frame()->view()->contentsToWindow(boxRect);
		}
		err = CE_S_OK;
	}
	return err;
}

CEHResult CEWebKitFrameWindowImpl::getWebKitWidget(CEWebKitWidget* *const ceWebKitWidgetOut)
{
	CEHResult err = CE_SILK_ERR_OPERATION_FAILED;
	if (_mainWindow)
	{
		CEWebKitWindowImpl* pImpl = _mainWindow->object();
		if (pImpl)
		{
			*ceWebKitWidgetOut = pImpl->getView();
			err = CE_S_OK;
		}
	}
	return err;

}

CEHResult CEWebKitFrameWindowImpl::initializeIME(bool password, bool singleLine, CEWebKitWidgetInputType inputType, const unsigned char* defaultStringStrUTF8, unsigned int defaultStringLen, INT32 maxLength, INT32 imeRect_x, INT32 imeRect_y, INT32 imeRect_width, INT32 imeRect_height, bool isPlugin, WebCore::Widget* pTarget)
{

	CEHResult err =  CE_SILK_ERR_OPERATION_FAILED;

	inactivateInputMethodState(false);
	_imeTargetPlugin = pTarget;
	CEComICEUStringRef iDefaultStringRef = NULL;
	err = ICEUStringCreate(CEComStdClassID_CEUString, iDefaultStringRef);
	if (!err)
	{
		err = iDefaultStringRef.initWithByteArray(reinterpret_cast<const UCHAR8*>(defaultStringStrUTF8), eICEI18nEncoding_utf_8, defaultStringLen);
		if (!err)
		{
			const UTF16CHAR* strData = NULL;
			UINT32 strLen = 0;
			err = iDefaultStringRef.getCharArray16(&strData, &strLen);
			if (!err)
			{
				CEWebKitHtmlInputType htmlInputType = CEWebKitHtmlInputType_Undefined;
				switch(inputType)
				{
				case CEWebKitWidgetInputType_TEXT:
					htmlInputType = CEWebKitHtmlInputType_TEXT;
					break;
				case CEWebKitWidgetInputType_PASSWORD:
					htmlInputType = CEWebKitHtmlInputType_PASSWORD;
					break;
				case CEWebKitWidgetInputType_ISINDEX:
					htmlInputType = CEWebKitHtmlInputType_ISINDEX;
					break;
				case CEWebKitWidgetInputType_CHECKBOX:
					htmlInputType = CEWebKitHtmlInputType_CHECKBOX;
					break;
				case CEWebKitWidgetInputType_RADIO:
					htmlInputType = CEWebKitHtmlInputType_RADIO;
					break;
				case CEWebKitWidgetInputType_SUBMIT:
					htmlInputType = CEWebKitHtmlInputType_SUBMIT;
					break;
				case CEWebKitWidgetInputType_RESET:
					htmlInputType = CEWebKitHtmlInputType_RESET;
					break;
				case CEWebKitWidgetInputType_FILE:
					htmlInputType = CEWebKitHtmlInputType_FILE;
					break;
				case CEWebKitWidgetInputType_HIDDEN:
					htmlInputType = CEWebKitHtmlInputType_HIDDEN;
					break;
				case CEWebKitWidgetInputType_IMAGE:
					htmlInputType = CEWebKitHtmlInputType_IMAGE;
					break;
				case CEWebKitWidgetInputType_BUTTON:
					htmlInputType = CEWebKitHtmlInputType_BUTTON;
					break;
				case CEWebKitWidgetInputType_SEARCH:
					htmlInputType = CEWebKitHtmlInputType_SEARCH;
					break;
				case CEWebKitWidgetInputType_RANGE:
					htmlInputType = CEWebKitHtmlInputType_RANGE;
					break;
				case CEWebKitWidgetInputType_EMAIL:
					htmlInputType = CEWebKitHtmlInputType_EMAIL;
					break;
				case CEWebKitWidgetInputType_NUMBER:
					htmlInputType = CEWebKitHtmlInputType_NUMBER;
					break;
				case CEWebKitWidgetInputType_TELEPHONE:
					htmlInputType = CEWebKitHtmlInputType_TELEPHONE;
					break;
				case CEWebKitWidgetInputType_URL:
					htmlInputType = CEWebKitHtmlInputType_URL;
					break;
				default:
					err = CE_SILK_ERR_OPERATION_FAILED;
					break;
				}
				if (!err)
				{
					err = getWidget().initializeTextInput(password, singleLine, htmlInputType, strData, strLen, maxLength, 0, imeRect_x, imeRect_y, imeRect_width, imeRect_height, isPlugin);
				}
			}
		}
	}
	return err;
}

CEHResult CEWebKitFrameWindowImpl::shutdownIME(bool nextIsTextInput)
{
	CEHResult err =  CE_SILK_ERR_OPERATION_FAILED;
	if (_imeTargetPlugin)
	{
		getWidget().shutdownTextInput(nextIsTextInput);
		_imeTargetPlugin = NULL;
	}
	return err;
}

CEHResult CEWebKitFrameWindowImpl::getUpdateDocRect(WebCore::FloatRect* const updateRect)
{	
	CEHResult err =  CE_SILK_ERR_OPERATION_FAILED;
	CERectF sizeRectF;
	err = getWidget().getUpdateDocRect(&sizeRectF);
	if (!err)
	{
		updateRect->setX(sizeRectF._x);
		updateRect->setY(sizeRectF._y);
		updateRect->setWidth(sizeRectF._width);
		updateRect->setHeight(sizeRectF._height);
	}
	return err;
}

CEHResult CEWebKitFrameWindowImpl::getElementUrl(void* hView, ICEURL** anchorUrlOut) const
{
	CEHResult hr = CE_S_OK;

	if (hView)
	{
		WebCore::Node* node = reinterpret_cast<WebCore::Node*>(hView);
		if (node && node->hasTagName(WebCore::HTMLNames::aTag))
		{
			WebCore::HTMLAnchorElement* anchor = static_cast<WebCore::HTMLAnchorElement*>(node);
			WebCore::KURL href = anchor->href();
			hr = href.createICEURL(anchorUrlOut);			
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
	}
	else
	{
		hr = CE_SILK_ERR_BADARGS;
	}
	
	return hr;
}

CEHResult CEWebKitFrameWindowImpl::getElementRect(void* hView, CERect* rectOut)
{
	CEHResult hr = CE_S_OK;
	if (hView)
	{
		if (WebCore::Node* node = reinterpret_cast<WebCore::Node*>(hView))
		{
			hr = _getElementRect(node, rectOut);
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
	}
	else
	{
		hr = CE_SILK_ERR_BADARGS;
	}
	return hr;
}

CEHResult CEWebKitFrameWindowImpl::isDocumentCoordinateMode(bool* resultOut)
{
	*resultOut = _isDocumentCoordinateMode();
	return CE_S_OK;
}

CEHResult CEWebKitFrameWindowImpl::windowRectToContentsRect(const CERect* windowRect, CERect* contentRectOut)
{
	WebCore::Page* page = core(this);
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
    const WebCore::Frame* frame = page->mainFrame();

	if (frame->view())
	{
		*contentRectOut = frame->view()->windowToContents(WebCore::IntRect(windowRect->_x, windowRect->_y, windowRect->_width, windowRect->_height));
		hr = CE_S_OK;
	}

	return hr;
}

CEHResult CEWebKitFrameWindowImpl::contentsRectToWindowRect(const CERect* contentRect, CERect* windowRectOut)
{
	WebCore::Page* page = core(this);
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
    const WebCore::Frame* frame = page->mainFrame();

	if (frame->view())
	{
		*windowRectOut = frame->view()->windowToContents(WebCore::IntRect(contentRect->_x, contentRect->_y, contentRect->_width, contentRect->_height));
		hr = CE_S_OK;
	}

	return hr;
}

CEHResult CEWebKitFrameWindowImpl::windowPointToContentsPoint(const CEPointBase* windowPos, CEPointBase* contentPosOut)
{
	WebCore::Page* page = core(this);
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
    const WebCore::Frame* frame = page->mainFrame();

	if (frame->view())
	{
		*contentPosOut = frame->view()->windowToContents(WebCore::IntPoint(windowPos->_x, windowPos->_y));
		hr = CE_S_OK;
	}

	return hr;
}

CEHResult CEWebKitFrameWindowImpl::contentsPointToWindowPoint(const CEPointBase* contentPos, CEPointBase* windowPosOut)
{
	WebCore::Page* page = core(this);
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
    const WebCore::Frame* frame = page->mainFrame();

	if (frame->view())
	{
		*windowPosOut = frame->view()->contentsToWindow(WebCore::IntPoint(contentPos->_x, contentPos->_y));
		hr = CE_S_OK;
	}

	return hr;
}

CEHResult CEWebKitFrameWindowImpl::changeZoomFactor(float scale)
{
	CEHResult err =  CE_SILK_ERR_OPERATION_FAILED;
	WebCore::Page* page = getPrivate()->getPage();
	CEU_VALIDATE_PTR(page);
	WebCore::Frame* frame = page->mainFrame();
	CEU_VALIDATE_PTR(frame);
	
	CERectF ceWindowRect;
	CEComICEHtmlWebKitFrameWindowChromeClientRef iWebKitFrameWindowChromeClientRef = getChromeClient();
	if(iWebKitFrameWindowChromeClientRef)
	{
		err = iWebKitFrameWindowChromeClientRef.getWindowRect(&ceWindowRect);
		if (!err)
		{
			CEU_VALIDATE_PTR(frame->view());
			WebCore::IntRect boundsRect = frame->view()->frameRect();

			WebCore::IntSize boundsSize;
			boundsSize.setWidth(static_cast<int>(ceilf(ceWindowRect._width*scale)));
			boundsSize.setHeight(boundsRect.height());
			frame->view()->resize(boundsSize);
			frame->setZoomFactor(scale, false);
			err = CE_S_OK;
		}
	}
	return err;
}

float CEWebKitFrameWindowImpl::getZoomFactor()
{
	WebCore::Page* page = getPrivate()->getPage();
	CEU_VALIDATE_PTR(page);
	WebCore::Frame* frame = page->mainFrame();
	CEU_VALIDATE_PTR(frame);
	return frame->zoomFactor();
}

CEHResult CEWebKitFrameWindowImpl::repaint(const WebCore::IntRect& rect)
{
	CEHResult err = CE_SILK_ERR_OPERATION_FAILED;
	WebCore::IntRect windowRepaintRect = rect;
	CEWebKitHtmlScrollMode scrollMode;
	err = _iChromeClientRef.getScrollMode(&scrollMode);
	if (!err)
	{
		CERect cerect = {windowRepaintRect.x(), windowRepaintRect.y(), windowRepaintRect.width(), windowRepaintRect.height()};
		//CEComDebugPrintf("CEWebKitFrameWindowImpl::repain x:%d y:%d width:%d height:%d before rect x:%d y:%d w:%d h:%d\n", cerect._x, cerect._y, cerect._width, cerect._height, rect.x(), rect.y(), rect.width(), rect.height());
		err = getChromeClient().repaint(&cerect);
	}
	return err;
}

CEHResult CEWebKitFrameWindowImpl::getScrollMode(CEWebKitScrollMode *const scrollMode)
{
	CEHResult err = CE_SILK_ERR_OPERATION_FAILED;
	CEWebKitHtmlScrollMode htmlScrollMode;
	err = _iChromeClientRef.getScrollMode(&htmlScrollMode);

	switch(htmlScrollMode)
	{
	case CEWebKitHtmlScrollMode_Window:
		*scrollMode = CEWebKitScrollMode_Window;
		err = CE_S_OK;
		break;
	case CEWebKitHtmlScrollMode_Document:
		*scrollMode = CEWebKitScrollMode_Document;
		err = CE_S_OK;
		break;
	case CEWebKitHtmlScrollMode_DocumentZoomFactor2:
		*scrollMode = CEWebKitScrollMode_DocumentZoomFactor2;
		err = CE_S_OK;
		break;
	default:
		*scrollMode = CEWebKitScrollMode_None;
		err = CE_SILK_ERR_OPERATION_FAILED;
		break;
	}

	return err;
}
