///////////////////////////////////////////////////////////////////////////////
// Copyright 2008,2009 Sony Corporation
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
// 
///////////////////////////////////////////////////////////////////////////////
#include "config.h"
#include "ScrollbarThemeSilk.h"
#include "Scrollbar.h"
#include "PlatformMouseEvent.h"
#include "GraphicsContextPlatformPrivateSilk.h"


namespace WebCore {

static const int scrollbarWidth = 16;
static ScrollbarTheme* pTheme;
ScrollbarTheme* ScrollbarTheme::nativeTheme()
{
	if (!pTheme)
	{
		pTheme = new ScrollbarThemeSilk();
	}
    return pTheme;
}


ScrollbarThemeSilk::ScrollbarThemeSilk():m_zoomFactorValue(1.0f)
{
	CEComGetThreadContext(CEComIID_ICEHtmlWebKitTheme, reinterpret_cast<void**>(&_iWebKitThemeRef));
	_iWebKitThemeRef.release();
}


bool ScrollbarThemeSilk::paint(Scrollbar* scrollbar , GraphicsContext* gc, const IntRect& damageRect)
{
	bool result = false;

	if (gc && scrollbar)
	{
		CERect dirty = {damageRect.x(), damageRect.y(), damageRect.width(), damageRect.height()};
		CERect rect = {scrollbar->x(), scrollbar->y(), scrollbar->width(), scrollbar->height()};
		_iWebKitThemeRef.drawScrollbar(
			gc->platformContext(),
			&rect, &dirty,
			scrollbar->orientation() == HorizontalScrollbar,
			scrollbar->currentPos(),
			scrollbar->visibleSize(),
			scrollbar->totalSize(),
			m_zoomFactorValue
			);
	}
	return result;
}

ScrollbarPart ScrollbarThemeSilk::hitTest(Scrollbar* scrollbar, const PlatformMouseEvent& mouseEvent)
{
	ScrollbarPart part = NoPart;
	if (scrollbar)
	{
		IntPoint mousePosition = scrollbar->convertFromContainingWindow(mouseEvent.pos());
		mousePosition.move(scrollbar->x(), scrollbar->y());
		
		CEHtmlDrawScrollbarPart cepart;
		CERect rect = {scrollbar->x(), scrollbar->y(), scrollbar->width(), scrollbar->height()};
		CEHResult hr = _iWebKitThemeRef.hitTestScrollbar(
			&rect,
			scrollbar->orientation() == HorizontalScrollbar,
			(HorizontalScrollbar == scrollbar->orientation()) ? mousePosition.x() : mousePosition.y(),
			scrollbar->currentPos(),
			scrollbar->visibleSize(),
			scrollbar->totalSize(),
			m_zoomFactorValue,
			&cepart
			);

		if (CESucceeded(hr))
		{
			part =
				cepart == CEHtmlDrawScrollbarPart_BackButton ? BackButtonStartPart :
				cepart == CEHtmlDrawScrollbarPart_BackTrack ? BackTrackPart :
				cepart == CEHtmlDrawScrollbarPart_ThumbPart ? ThumbPart :
				cepart == CEHtmlDrawScrollbarPart_ForwardTrack ? ForwardTrackPart :
				cepart == CEHtmlDrawScrollbarPart_ForwardButton ? ForwardButtonStartPart :
				NoPart;
		}
	}
	return part;
}
int ScrollbarThemeSilk::scrollbarThickness(ScrollbarControlSize size)
{
	UINT32 width;
	CEHResult hr = _iWebKitThemeRef.scrollberTickness(
		size == RegularScrollbar ? CEHtmlScrollbarControlSize_Regular :
		CEHtmlScrollbarControlSize_Small,
		&width);

	if (CEFailed(hr))
	{
		width = 16;
	}

	return static_cast<int>(width*m_zoomFactorValue);
}
void ScrollbarThemeSilk::invalidatePart(Scrollbar* scrollbar, ScrollbarPart part)
{
	scrollbar->invalidate();
}

int ScrollbarThemeSilk::thumbPosition(Scrollbar* scrollbar)
{
    if (scrollbar->enabled())
        return (int)((float)scrollbar->currentPos() * (trackLength(scrollbar) - thumbLength(scrollbar)) / scrollbar->maximum());
    return 0;
}

static int max(int x, int y) { return x > y ? x : y; }

int ScrollbarThemeSilk::thumbLength(Scrollbar* scrollbar)
{
    if (!scrollbar->enabled())
        return 0;

    float proportion = (float)scrollbar->visibleSize() / scrollbar->totalSize();
    int trackLen = trackLength(scrollbar);
    int length = proportion * trackLen;
    length = max(length, scrollbarThickness(scrollbar->controlSize()));
    if (length > trackLen)
        length = 0; // Once the thumb is below the track length, it just goes away (to make more room for the track).
    return length;
}

int ScrollbarThemeSilk::trackPosition(Scrollbar* scrollbar)
{
    return scrollbarWidth;
}

int ScrollbarThemeSilk::trackLength(Scrollbar* scrollbar)
{
	int ret = scrollbar->visibleSize() - (scrollbarWidth * 2);
	if (ret < 0) {
		ret = 0;
	}
    return ret;
}

void ScrollbarThemeSilk::registerScrollbar(Scrollbar* scrollbar)
{
}

void ScrollbarThemeSilk::unregisterScrollbar(Scrollbar* scrollbar)
{
}

} // namespace WebCore
