/*
 * Copyright (C) 2004, 2005, 2006 Apple Computer, Inc.  All rights reserved.
 * Copyright     2010 Sony Corporation
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
#include "config.h"
#include "ScrollView.h"
#include "HostWindow.h"
#include "FrameView.h"
#include "EventHandler.h"
#include "Frame.h"

using std::max;

namespace WebCore {


void ScrollView::platformInit()
{
}

void ScrollView::platformDestroy()
{
}

void ScrollView::platformAddChild(Widget*)
{
}

void ScrollView::platformRemoveChild(Widget*)
{
}

void ScrollView::platformSetScrollbarsSuppressed(bool repaintOnUnsuppress)
{
	if (repaintOnUnsuppress && !m_scrollbarsSuppressed) {
        if (m_horizontalScrollbar)
            m_horizontalScrollbar->invalidate();
        if (m_verticalScrollbar)
            m_verticalScrollbar->invalidate();

        // Invalidate the scroll corner too on unsuppress.
        invalidateRect(scrollCornerRect());
    }
}

void ScrollView::platformSetScrollbarModes()
{
}

void ScrollView::platformScrollbarModes(ScrollbarMode& horizontal, ScrollbarMode& vertical) const
{
    horizontal = m_horizontalScrollbarMode;
    vertical = m_verticalScrollbarMode;
}
void ScrollView::platformSetCanBlitOnScroll(bool b)
{
    m_canBlitOnScroll = b;
}

bool ScrollView::platformCanBlitOnScroll() const
{
    return m_canBlitOnScroll;
}

IntRect ScrollView::platformVisibleContentRect(bool includeScrollbars) const
{
	return IntRect(IntPoint(m_scrollOffset.width(), m_scrollOffset.height()), m_boundsSize);
}

IntSize ScrollView::platformContentsSize() const
{
    return m_contentsSize;
}

void ScrollView::platformSetContentsSize()
{
	if (parent())
		return;
	if (platformWidget())
	{
		float fzoomFactor = 1.0f;
		if (this->isFrameView())
		{
			FrameView* fv = static_cast<FrameView*>(this);
			if (fv)
			{
				Frame* frame = fv->frame();
				if (frame)
				{
					fzoomFactor = frame->zoomFactor();
				}
			}
		}
		platformWidget()->notifyChangeContentSize(m_contentsSize, m_fixedLayoutSize, fzoomFactor);
	}
}

IntRect ScrollView::platformContentsToScreen(const IntRect& rect) const
{
    if (!hostWindow())
        return IntRect();
    return hostWindow()->windowToScreen(contentsToWindow(rect));
}

IntPoint ScrollView::platformScreenToContents(const IntPoint& point) const
{
    if (!hostWindow())
        return IntPoint();
    return windowToContents(hostWindow()->screenToWindow(point));
}

void ScrollView::platformSetScrollPosition(const IntPoint& position)
{
	if (parent())
		return;
	if (platformWidget())
	{
	    IntPoint newScrollPosition = position.shrunkTo(maximumScrollPosition());
	    newScrollPosition.clampNegativeToZero();

		platformWidget()->setScrollPosition(newScrollPosition);
		if (newScrollPosition == scrollPosition())
			return;
		m_scrollOffset = IntSize(newScrollPosition.x(), newScrollPosition.y());

	//	CEComDebugPrintf("Scroll offset x:%d, y:%d \n", m_scrollOffset.width(), m_scrollOffset.height());

		ASSERT(this->isFrameView());
		if (this->isFrameView())
		{
			FrameView* fv = static_cast<FrameView*>(this);
			if (fv)
			{
				Frame* frame = fv->frame();
				if (frame)
				{
					EventHandler* eventHandler = frame->eventHandler();
					if (eventHandler)
					{
						eventHandler->sendScrollEvent();
					}
				}
			}
		}
		// for Fixed scroll position
		if (!canBlitOnScroll() || rootPreventsBlitting()) 
		{
			IntRect visibleRect = actualVisibleContentRect();
			hostWindow()->repaint(visibleRect, true, false, true); // Invalidate the backing store and repaint it synchronously
		}
	}
}

bool ScrollView::platformScroll(ScrollDirection direction, ScrollGranularity granularity)
{
    if (direction == ScrollUp || direction == ScrollDown) {
        if (m_verticalScrollbar)
            return m_verticalScrollbar->scroll(direction, granularity);
    } else {
        if (m_horizontalScrollbar)
            return m_horizontalScrollbar->scroll(direction, granularity);
    }
    return false;
}

void ScrollView::platformRepaintContentRectangle(const IntRect& rect, bool now)
{
	if (hostWindow())
		hostWindow()->repaint(rect, true, now);
}

bool ScrollView::platformIsOffscreen() const
{
    if (!isVisible())
        return true;
	return false;
}

} // namespace WebCore
