/*
 * Copyright (C) 2006 Apple Computer, Inc.  All rights reserved.
 * Copyright (C) 2006 Michael Emmel mike.emmel@gmail.com
 * All rights reserved.
 * Copyright     2009 Sony Corporation
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#ifndef ResourceHandleManagerCore_h
#define ResourceHandleManagerCore_h

#include "Frame.h"
#include "Page.h"
#include "PageGroup.h"
#include "Timer.h"
#include "CString.h"
#include "ResourceHandleClient.h"
#include "ResourceLoader.h"
#include "DocumentLoader.h"
#include "FrameView.h"
#include <wtf/Vector.h>
#include "ICEHtmlWebKit.h"

#include "ICENetwork.h"

#define PAGE_IDENTIFIER_SHIFT (sizeof(unsigned long)*8 - 8)

namespace WebCore {

class WebCoreSynchronousLoader : public ResourceHandleClient {
public:
	WebCoreSynchronousLoader(ICENetworkEventListener* networkListener);

	virtual void didReceiveResponse(ResourceHandle* webCoreJob, const char* responseHeaderBuffer, UINT32 responseHeaderLen, const char* url, UINT32 httpCode, UINT32 expectedContentLength, bool noContext);
	virtual void didReceiveData(ResourceHandle* webCoreJob, void* data, UINT32 size);
	virtual void didFinishLoading(ResourceHandle* webCoreJob);
	virtual void didFail(ResourceHandle* webCoreJob);

	ICENetworkEventListener* getNetworkListener() { return m_networklistener; };
	ResourceResponse resourceResponse() const { return m_response; }
	ResourceError resourceError() const { return m_error; }
	Vector<char> data() const { return m_data; }

private:
	ResourceResponse m_response;
	ResourceError m_error;
	Vector<char> m_data;
	CEComICENetworkEventListenerRef m_networklistener;
};

class ResourceHandleManager
{
	friend class WebCoreSynchronousLoader;

public:
	static ResourceHandleManager* sharedInstance();

private:
	ResourceHandleManager();
	virtual ~ResourceHandleManager();
	ResourceHandleManager(const ResourceHandleManager& other);
	ResourceHandleManager& operator=(const ResourceHandleManager& other);

public:
	void add(ResourceHandle*, WebCore::Frame* frame);
	void cancel(ResourceHandle*);
	void cancel(unsigned long);
	void dispatchSynchronousJob(ResourceHandle*, WebCore::Frame* frame);
	void setCookieJarFileName(const char* cookieJarFileName);
	void setCookies(const KURL& url, const String& value);
	String getCookies(const KURL& url);
	bool cookiesEnabled();
	void didFinishLoading(unsigned long identifier, const String* metaTagInfo);
	void deleteFrameLoderClientCert(void* cert);

	static CEHResult init(ICENetwork* iCENetwork);
	static void shutdown();
	static CEHResult setCookieEnable(bool isCookieEnable);
	static CEHResult deleteAllCookies();
	static CEHResult headerCallback(ResourceHandle* webCoreJob, const char* responseHeaderBuffer, UINT32 responseHeaderLen, const char* url, UINT32 httpCode, UINT32 expectedContentLength, bool noContext);
	static CEHResult DataAvailableCallback(ResourceHandle* webCoreJob, UINT32 httpCode, const char* url, void* data, UINT32 size);
	static CEHResult CompleteCallback(ResourceHandle* webCoreJob);
	static CEHResult redirectCallback(ResourceHandle* webCoreJob, const char* url);
	static CEHResult errorCallback(ResourceHandle* webCoreJob, INT32 lowLevelError, CEHResult resultCode, bool noContext);
	static CEHResult allInOneCallback(ResourceHandle* webCoreJob, const char* responseHeaderBuffer, UINT32 responseHeaderLen, const char* url, UINT32 httpCode, UINT32 expectedContentLength, bool noContext, void* data, UINT32 size);
	static CEHResult readCallback(ResourceHandle* webCoreJob, void* postData, UINT32 blockSize, UINT32 blockNum);
	static CEHResult replyBadCertCallback(unsigned long identifier, bool sslContinue);
	static CEHResult replyAuthCallback(unsigned long identifier, ICEUString* site, ICEUString* realm, ICEUString* username, ICEUString* password, bool authContinue, bool updateSavedFlag);

	static CEHResult _ICEUStringToCString(ICEUString* in, char** out);

private:
	void startJob(ResourceHandle*);
	CEHResult startJob(ResourceHandle* job, bool syncRequest, WebCore::Frame* frame);
	void downloadTimerCallback(Timer<ResourceHandleManager>*);
	bool removeScheduledJob(ResourceHandle*);

	static CEHResult _handleHeaderCallback(ResourceHandle* webCoreJob, const char* responseHeaderBuffer, UINT32 responseHeaderLen, const char* url, UINT32 httpCode, UINT32 expectedContentLength, bool noContext);
	static CEHResult _handleDataAvailableCallback(ResourceHandle* webCoreJob, UINT32 httpCode, const char* url, void* data, UINT32 size);
	static CEHResult _handleCompleteCallback(ResourceHandle* webCoreJob);
	static CEHResult _handleHeaderCallbackInternal(ResourceHandle* webCoreJob, const char* responseHeaderBuffer, UINT32 responseHeaderLen, const char* url, UINT32 httpCode, UINT32 expectedContentLength, ResourceResponse& response);

	static ICENetworkEventListener* getNetworkListenerFromJob(WebCore::ResourceHandle* job, bool syncRequest, WebCore::Frame* frame);

	char* m_cookieJarFileName;
	Vector<ResourceHandle*> m_resourceHandleList;
	int m_runningJobs;
};

}

#endif // ResourceHandleManagerCore_h
