/*
 * Copyright (C) 2004, 2005, 2006, 2008 Apple Inc. All rights reserved.
 * Copyright     2009 Sony Corporation
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */

#include "config.h"
#include "ImageSource.h"

#if PLATFORM(SILK)

#include "IntSize.h"
#include "SharedBuffer.h"
#include "URL.h"
#include "CEImageBufferGraphicsContextImpl.h"

#include "GraphicsContextPlatformPrivateSilk.h"

#define INFINITY_LOOP	0

namespace WebCore {

ImageSource::ImageSource()
: m_vgImageSource(0), m_isDraw(false), m_offset(0), m_isStereo(false)
{
}

ImageSource::~ImageSource()
{
	m_vgImageSource=0;
}

void ImageSource::clear(bool destroyAll, size_t clearBeforeFrame, SharedBuffer* data, bool allDataReceived)
{
	m_offset = 0;

	//USIZE_T allocedSize = CEComGetAllocedSize(0);
	//printf("first allocsize:%d, destroyAll:%d m_vgSurface:%08x \n", allocedSize, destroyAll, m_vgSurface.object());

	if(destroyAll)
	{
		m_isDraw = false;
		if(initialized())
		{
			CEComICEVGSurfaceRef vgSurface;
			vgSurface.initByQueryInterface(m_vgImageSource);
			if (vgSurface.object())
			{
				vgSurface.clean();
				CEVGSurfaceInitParam* initParam = CENEW(CEVGSurfaceInitParam);
				if (initParam)
				{
					initParam->pixelFormat = CEPixelFormat_PA8R8G8B8;
					CEHResult ret = vgSurface.initialize(initParam);
					CEASSERT(CESucceeded(ret));
					CEDELETE(initParam);
				}
			}
		}
	}
	
	//allocedSize = CEComGetAllocedSize(0);
	//printf("second allocsize:%d, destroyAll:%d m_vgSurface:%08x\n", allocedSize, destroyAll, m_vgSurface.object());
	if(destroyAll && data)
	{
		setData(data, allDataReceived);
	}
	//allocedSize = CEComGetAllocedSize(0);
	//printf("third allocsize:%d, destroyAll:%d m_vgSurface:%08x\n\n", allocedSize, destroyAll, m_vgSurface.object());
}

bool ImageSource::initialized() const
{
	return m_vgImageSource.object() ? true : false;
}

bool ImageSource::isDraw()
{
	return m_isDraw;
}

void ImageSource::setData(SharedBuffer* data, bool allDataReceived)
{
	CEHResult hr = CE_S_OK;

	////////////////////////////////////////////////////////////////////////////////////////////////
	// Initialize ICECompressedImage object.
	if(!initialized())
	{
		m_isDraw = false;
		CEComICEVGFactoryRef vgFactoryRef;

		vgFactoryRef = GraphicsContextPlatformPrivate::Env::getVGFactory();

		if (vgFactoryRef)
		{
			CEComICEVGSurfaceRef vgSurfaceRef;
			hr = vgFactoryRef.createImageSurface(m_isStereo, &vgSurfaceRef);
			if (CESucceeded(hr))
			{
				hr = m_vgImageSource.initByQueryInterface(vgSurfaceRef);
			}
		}
	}

	if (CESucceeded(hr))
	{
		////////////////////////////////////////////////////////////////////////////////////////////////
		// Data arrived
		if(initialized() && data)
		{
						
			//Set data on Compressed Image.
					   
			size_t amtOfNewDataSize = data->buffer().size();
			amtOfNewDataSize -= m_offset;

			const UINT8* buffer = reinterpret_cast<const UINT8*>(data->buffer().data());
			buffer += m_offset;

			// append data.
			hr = m_vgImageSource.appendData(buffer, amtOfNewDataSize, allDataReceived);
			m_offset += amtOfNewDataSize;
		}
	}

	////////////////////////////////////////////////////////////////////////////////////////////////
	// Data complete
	if(initialized() && allDataReceived)
	{
		// Finish creating image.
		m_isDraw = true;
	}
}

String ImageSource::filenameExtension() const
{
    return String();
}

bool ImageSource::isSizeAvailable()
{
	bool result = false;

	//fist check
	if(m_isDraw)
	{
		if (initialized())
		{
			CEDim dim;
			CEHResult hr;
			CEComICEVGSurfaceRef vgSurface;
			hr = vgSurface.initByQueryInterface(m_vgImageSource);
			if (CESucceeded(hr))
			{
				hr = vgSurface.getDimension(&dim);
				if(CESucceeded(hr))
				{
					result = ((dim._height > 0 && dim._width > 0) ? true : false);
				}
			}
		}
	}
	return result;
}

IntSize ImageSource::frameSizeAtIndex(size_t index) const
{
	//CEASSERT(0 && "frameSizeAtIndex Not Implemented");
	//Silk DX cann't return the each frame, therfore, it has always returned same frame that is frame one.
	return size();
}

IntSize ImageSource::size() const 
{
	CEDim dim;
	dim._height = dim._width = 0;

	//Silk DX cann't return the each frame, therfore, it has always returned same frame that is frame one.
	if (initialized())
	{
		// why build error.
//		m_vgSurface.getDimension(&dim);
		CEComICEVGSurfaceRef vgSurface;
		vgSurface.initByQueryInterface(m_vgImageSource);
		vgSurface.object()->_vtbl->_getDimension(vgSurface, &dim);
	}
	return IntSize(dim);
}

// Get LoopCount.
// If return 0 is an infinite loop.
// Else if return 1 is a no loop.
int ImageSource::repetitionCount()
{
	int loopCount = INFINITY_LOOP;

	if (initialized())
	{
		CEHResult hr = m_vgImageSource.getLoopCount(&loopCount);
		if(!hr)
		{
			if(0 < loopCount)
			{
				// If GIFfile's loop count 1, WebCore loop count must be 2.
				// Namely have to increment.
				loopCount++;
			}
		}
	}
	return loopCount;
}

// Get FrameCount.
size_t ImageSource::frameCount() const
{
	int frameCount = 0;

	if (initialized())
	{
		CEHResult hr = m_vgImageSource.object()->_vtbl->_getFrameCount(m_vgImageSource, &frameCount);
	}

	return frameCount;
}

// Get VGSurface.
NativeImagePtr ImageSource::createFrameAtIndex(size_t index)
{
	if (initialized())
	{
		CEComICEVGSurfaceRef vgSurface;
		vgSurface.initByQueryInterface(m_vgImageSource);
		return vgSurface;
	}
	return NULL;
}

// Get isCompleted at index.
bool ImageSource::frameIsCompleteAtIndex(size_t index)
{
	bool ret = false;

	if (initialized())
	{
		eCEVGSurfaceLoadStatus status;	
		m_vgImageSource.getStatus(&status);
		if(status == eCEVGSurfaceLoadStatus_COMPLETED)
		{
			ret = true;
		}
	}
	return ret;
}

// Get DelayTime at index.
// Return value convert msec to sec.
float ImageSource::frameDurationAtIndex(size_t index)
{
	int delayTime = 0;

	if(initialized())
	{
		m_vgImageSource.getDelayTime(index, &delayTime);
	}

	return (float)delayTime / 1000;
}

// Get hasAlpha at index.
bool ImageSource::frameHasAlphaAtIndex(size_t index)
{
    return true;
}
}

#endif // PLATFORM(SILK)
