/*
 * Copyright (C) 2006 Nikolas Zimmermann <zimmermann@kde.org>
 * Copyright (C) 2008 Apple Inc. All rights reserved.
 * Copyright     2009 Sony Corporation
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "ImageBuffer.h"

#include "Base64.h"
#include "BitmapImage.h"
#include "CString.h"
#include "GraphicsContext.h"
#include "GraphicsContext.h"
#include "GraphicsContextPlatformPrivateSilk.h"
#include "ImageData.h"
#include "PlatformString.h"
#include "MIMETypeRegistry.h"

#include "CEImageBufferGraphicsContext.h"

using namespace std;

namespace WebCore {

ImageBufferData::ImageBufferData(const IntSize&)
: m_surface(0)
{	
	
}

#if defined(VGC_LATEST_COLORSPACE_17634)
ImageBuffer::ImageBuffer(const IntSize& size, ImageColorSpace imageColorSpace, bool& success)
#else #if defined(VGC_LATEST_COLORSPACE_17634)
ImageBuffer::ImageBuffer(const IntSize& size, bool grayScale, bool& success)
#endif //#if defined(VGC_LATEST_COLORSPACE_17634)
    : m_data(size)
    , m_size(size)
{
	success = false;
	CEComICEVGFactoryRef vgFactory = GraphicsContextPlatformPrivate::Env::getVGFactory();
	ASSERT(!grayScale && "todo: support gray scale surface");
	CEHResult result = vgFactory.createOffScreenSurface(NULL,grayScale? CEPixelFormat_A8:CEPixelFormat_PA8R8G8B8,
		static_cast<UINT32>(size.width()), static_cast<UINT32>(size.height()), &(m_data.m_surface));
	if (CESucceeded(result))
	{
		CEComICEVGContextRef vgc;
		CEComICEUnknownRef unk;
		unk.initByQueryInterface(m_data.m_surface);
		result = vgFactory.createContext(unk, &vgc);
		if (CESucceeded(result))
		{
			m_context.set(new GraphicsContext(vgc));
			success = true;
		}
	}
}

ImageBuffer::~ImageBuffer()
{
	m_data.m_surface = 0;
}

GraphicsContext* ImageBuffer::context() const
{
	return m_context.get();
}

Image* ImageBuffer::image() const
{
	if (!m_image) {
		// It's assumed that if image() is called, the actual rendering to the
		// GraphicsContext must be done.
		ASSERT(context());

		// BitmapImage will release the passed in surface on destruction
		CEComICEVGSurfaceRef thisSurface = m_data.m_surface;
		CEComICEVGSurfaceRef newSurface;
		if (CESucceeded(thisSurface.clone(&newSurface))) {
			m_image = BitmapImage::create(newSurface);
		}
	}
	return m_image ? m_image.get() : Image::nullImage() ;
}

#if defined(VGC_LATEST_COLORSPACE_17634)
void ImageBuffer::platformTransformColorSpace(const Vector<int>& lookUpTable)
{
	ASSERT(cairo_surface_get_type(m_data.m_surface) == CAIRO_SURFACE_TYPE_IMAGE);

	unsigned char* dataSrc = cairo_image_surface_get_data(m_data.m_surface);
	int stride = cairo_image_surface_get_stride(m_data.m_surface);
	for (int y = 0; y < m_size.height(); ++y) {
		unsigned* row = reinterpret_cast<unsigned*>(dataSrc + stride * y);
		for (int x = 0; x < m_size.width(); x++) {
			unsigned* pixel = row + x;
			Color pixelColor = colorFromPremultipliedARGB(*pixel);
			pixelColor = Color(lookUpTable[pixelColor.red()],
				lookUpTable[pixelColor.green()],
				lookUpTable[pixelColor.blue()],
				pixelColor.alpha());
			*pixel = premultipliedARGBFromColor(pixelColor);
		}
	}
	cairo_surface_mark_dirty_rectangle (m_data.m_surface, 0, 0, m_size.width(), m_size.height());
}
#endif //#if defined(VGC_LATEST_COLORSPACE_17634)
#if defined(VGC_FILTERS_17935)
#else //#if defined(VGC_FILTERS_17935)
enum Multiply {
	Premultiplied,
	Unmultiplied
};
#endif //#if defined(VGC_FILTERS_17935)
template <Multiply multiplied>
PassRefPtr<ImageData> _getImageData(const IntRect& rect, const ImageBufferData& data, const IntSize& size)
{
	CEComICEVGSurfaceRef csurface(const_cast<ICEVGSurface*>(data.m_surface.object()));
	PassRefPtr<ImageData> result = ImageData::create(rect.width(), rect.height());
	unsigned char* dataSrc = 0;
	int stride = 0;
	csurface.getImageDataRef(&stride,(UCHAR8* *const)&dataSrc);
	unsigned char* dataDst = result->data()->data()->data();
	if (dataSrc && dataDst)	{
		if (rect.x() < 0 || rect.y() < 0 || (rect.x() + rect.width()) > size.width() || (rect.y() + rect.height()) > size.height())
			memset(dataDst, 0, result->data()->length());

		int originx = rect.x();
		int destx = 0;
		if (originx < 0) {
			destx = -originx;
			originx = 0;
		}
		int endx = rect.x() + rect.width();
		if (endx > size.width())
			endx = size.width();
		int numColumns = endx - originx;

		int originy = rect.y();
		int desty = 0;
		if (originy < 0) {
			desty = -originy;
			originy = 0;
		}
		int endy = rect.y() + rect.height();
		if (endy > size.height())
			endy = size.height();
		int numRows = endy - originy;

		unsigned destBytesPerRow = 4 * rect.width();

		unsigned char* destRows = dataDst + desty * destBytesPerRow + destx * 4;
		for (int y = 0; y < numRows; ++y) {
			unsigned* row = reinterpret_cast<unsigned*>(dataSrc + stride * (y + originy));
			for (int x = 0; x < numColumns; x++) {
				int basex = x * 4;
				unsigned* pixel = row + x + originx;
				Color pixelColor;

				if (multiplied == Unmultiplied)
					pixelColor = colorFromPremultipliedARGB(*pixel);
				else
					pixelColor = Color(*pixel);

				destRows[basex]     = pixelColor.red();
				destRows[basex + 1] = pixelColor.green();
				destRows[basex + 2] = pixelColor.blue();
				destRows[basex + 3] = pixelColor.alpha();
			}
			destRows += destBytesPerRow;
		}
	}
	return result;
}

PassRefPtr<ImageData> ImageBuffer::getImageData(const IntRect& rect) const
{
	return _getImageData<Unmultiplied>(rect, m_data, m_size);
}

#if defined(VGC_FILTERS_17935)
PassRefPtr<ImageData> ImageBuffer::getUnmultipliedImageData(const IntRect& rect) const
{
	return _getImageData<Unmultiplied>(rect, m_data, m_size);
}

PassRefPtr<ImageData> ImageBuffer::getPremultipliedImageData(const IntRect& rect) const
{
	return _getImageData<Premultiplied>(rect, m_data, m_size);
}
#endif //#if defined(VGC_FILTERS_17935)

template <Multiply multiplied>
void _putImageData(ImageData*& source, const IntRect& sourceRect, const IntPoint& destPoint, ImageBufferData& data, const IntSize& size)
{
	CEComICEVGSurfaceRef csurface(data.m_surface);
	unsigned char* dataDst = 0;
	int stride = 0;
	csurface.getImageDataRef(&stride,(UCHAR8* *const)&dataDst);
	
	if (dataDst) {

		ASSERT(sourceRect.width() > 0);
		ASSERT(sourceRect.height() > 0);

		int originx = sourceRect.x();
		int destx = destPoint.x() + sourceRect.x();
		ASSERT(destx >= 0);
		ASSERT(destx < size.width());
		ASSERT(originx >= 0);
		ASSERT(originx <= sourceRect.right());

		int endx = destPoint.x() + sourceRect.right();
		ASSERT(endx <= size.width());

		int numColumns = endx - destx;

		int originy = sourceRect.y();
		int desty = destPoint.y() + sourceRect.y();
		ASSERT(desty >= 0);
		ASSERT(desty < size.height());
		ASSERT(originy >= 0);
		ASSERT(originy <= sourceRect.bottom());

		int endy = destPoint.y() + sourceRect.bottom();
		ASSERT(endy <= size.height());
		int numRows = endy - desty;

		unsigned srcBytesPerRow = 4 * source->width();
	
		unsigned char* srcRows = source->data()->data()->data() + originy * srcBytesPerRow + originx * 4;
		for (int y = 0; y < numRows; ++y) {
			unsigned* row = reinterpret_cast<unsigned*>(dataDst + stride * (y + desty));
			for (int x = 0; x < numColumns; x++) {
				int basex = x * 4;
				unsigned* pixel = row + x + destx;
				Color pixelColor(srcRows[basex],
					srcRows[basex + 1],
					srcRows[basex + 2],
					srcRows[basex + 3]);
				if (multiplied == Unmultiplied)
					*pixel = premultipliedARGBFromColor(pixelColor);
				else
					*pixel = pixelColor.rgb();
			}
			srcRows += srcBytesPerRow;
		}
		data.m_surface.makeDirtyRectangleI(destx, desty,numColumns, numRows);
	}
}

void ImageBuffer::putImageData(ImageData* source, const IntRect& sourceRect, const IntPoint& destPoint)
{
	_putImageData<Unmultiplied>(source, sourceRect, destPoint, m_data, m_size);
}

#if defined(VGC_FILTERS_17935)
void ImageBuffer::putUnmultipliedImageData(ImageData* source, const IntRect& sourceRect, const IntPoint& destPoint)
{
	_putImageData<Unmultiplied>(source, sourceRect, destPoint, m_data, m_size);
}

void ImageBuffer::putPremultipliedImageData(ImageData* source, const IntRect& sourceRect, const IntPoint& destPoint)
{
	_putImageData<Premultiplied>(source, sourceRect, destPoint, m_data, m_size);
}
#endif //#if defined(VGC_FILTERS_17935)

String ImageBuffer::toDataURL(const String& mimeType) const
{
	CEComICEVGContextRef vgc = context()->platformContext();
	if(vgc)
	{
		String actualMimeType("image/png");
		if (MIMETypeRegistry::isSupportedImageMIMETypeForEncoding(mimeType))
			actualMimeType = mimeType;

		Vector<char> in;
		// Only PNG output is supported for now.

		CEWebCoreVGSurfaceWriteFunc* pFunc = new CEWebCoreVGSurfaceWriteFunc(in);
		if(pFunc)
		{
			CEComICEVGSurfaceWriteFunctionRef iFunc(pFunc->toICEVGSurfaceWriteFunction(pFunc));
			CEHResult result = vgc.writeTargetSurfaceToPNGStream(iFunc,0);
			if (CESucceeded(result)) {
				Vector<char> out;
				base64Encode(in, out);
				return "data:" + actualMimeType + ";base64," + String(out.data(), out.size());
			}
		}
	}
	return "data:,";
}

} // namespace WebCore
