/*
 * Copyright (C) 2006, 2007, 2008 Apple Inc.  All rights reserved.
 * Copyright     2008, 2009 Sony Corporation
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */

#include "config.h"

#if USE(SILK_CEUIFONT)
#include "Font.h"

#include "FontFallbackList.h"
#include "GlyphBuffer.h"
#include "GraphicsContext.h"
#include "IntRect.h"
#include "NotImplemented.h"
#include "SimpleFontData.h"
#include <wtf/MathExtras.h>
#include "ICEVG.h"
#include "ICEDXMath.h"
#include "ceustrmacros.h"
#include "FontCache.h"
#include "GraphicsContextPlatformPrivateSilk.h"

namespace WebCore {

#define WITH_FONT_OPTION(_optGc, _font, _run, _posX, _foption, _pc16s) \
	const UTF16CHAR* _pc16s = 0;\
	CEUIDirectionOption dirOption;\
	CEUIFontOption _foption;\
	CEUITabOption tabOption;\
	CEUIVariantOption variantOption;\
	CEUIWhiteSpaceOption wsOption;\
	CEUIGlyphStrokeOption strokeOption;\
	_prepare(_optGc, _font, _run, (INT32)_posX, _pc16s, dirOption, _foption, tabOption, variantOption, wsOption, strokeOption);

static void _prepare
(
 GraphicsContext* optGc,
 const Font& font,
 const TextRun& run, 
 INT32 posX,
 const UTF16CHAR*& pC16sOut, 
 CEUIDirectionOption& dirOptOut, 
 CEUIFontOption& fontOptOut, 
 CEUITabOption& tabOptOut,
 CEUIVariantOption& variantOption,
 CEUIWhiteSpaceOption& wsOption,
 CEUIGlyphStrokeOption& strokeOption
 )
{
	CESysFillMemory(&dirOptOut, 0, sizeof(CEUIDirectionOption));
	dirOptOut.direction = run.ltr() ? eCEI18nBiDiDirection_LTR : eCEI18nBiDiDirection_RTL;

	CESysFillMemory(&fontOptOut, 0, sizeof(CEUIFontOption));
	fontOptOut.directionOption = &dirOptOut;

	CESysFillMemory(&tabOptOut, 0, sizeof(CEUITabOption));
	if (run.allowTabs()) {
		fontOptOut.tabOption = &tabOptOut;
		tabOptOut.tabStopWidth = font.tabWidth();
		tabOptOut.leftX = posX;
	}

	CESysFillMemory(&variantOption, 0, sizeof(CEUIVariantOption));
	if (font.fontDescription().smallCaps())
	{
		variantOption.variant = eCEUIFontVariant_smallCaps;
		fontOptOut.variantOption = &variantOption;
	}

	CESysFillMemory(&wsOption, 0, sizeof(CEUIWhiteSpaceOption));
	{
		wsOption.whiteSpaceWidth = font.primaryFont()->adjustedSpaceWidth();
		wsOption.letterSpacing = font.letterSpacing();
		wsOption.wordSpacing = font.wordSpacing();
		fontOptOut.whiteSpaceOption = &wsOption;
	}

	if (optGc) {
		CESysFillMemory(&strokeOption, 0, sizeof(CEUIGlyphStrokeOption));
		int tdmode = optGc->textDrawingMode();
		strokeOption.fill = (tdmode & cTextFill) ? 1 : 0;

		if (tdmode & cTextStroke) {
			Color sc = optGc->strokeColor();
			GraphicsContextPlatformPrivate::Color2CERGBAColor(sc, strokeOption.strokeColor);
			strokeOption.stroke = 1;
		}
		fontOptOut.strokeOption = &strokeOption;
	}

	pC16sOut = (const UTF16CHAR*) run.characters();
}

void Font::copyMembers(const Font& other)
{
	m_cecfFont = other.m_cecfFont;
	m_srcFontDescription = other.m_srcFontDescription;
}

bool Font::canReturnFallbackFontsForComplexText()
{
    return true;
}

ICEUICompositeFont*	Font::cecfFont() const
{
	if (!m_cecfFont || m_srcFontDescription != m_fontDescription) {
		m_cecfFont = 0;
		CEComICEUIFontFamilyRef family;
		if (CEUIFontSupport::toCEUIFontFamily(*this, family))
		{
			family = 0;
			AtomicString astr("default");
			CEUIFontSupport::toCEUIFontFamily(astr, family);
		}
			
		CEASSERT(family);
		{
			CEComICEUICompositeFontFactoryRef cfFactory = CEUIFontSupport::getCFFactory();
			CEComICEI18nLocaleRef lang;
			CEUIFontSupport::getCFLang(lang);
			CEPFFontStyle style = CEUIFontSupport::toCEUIFontStyle(m_fontDescription);
			cfFactory.createCompositeFontBySizeInPixels(lang, family, &style, m_fontDescription.computedPixelSize(), true, &m_cecfFont);
			m_srcFontDescription = m_fontDescription;
		}
	}
	return m_cecfFont;
}

void Font::drawComplexText(GraphicsContext* context, const TextRun& run, const FloatPoint& point,
						   int from, int to) const
{
	if (!context) return;

	if (cecfFont() && from < to) {
		WITH_FONT_OPTION(0, *this, run, point.x(), foption, pC16s);

		CEComICEUITextMeasurerRef tm;
		CEUIFontStrRangeOption rangeOption = {from, (to - from)};
		m_cecfFont.getTextMeasurer(pC16s, run.length(), &foption, &tm);
		CEComICEUIFontGlyphBufferRef gb;
		if (CESucceeded(tm.getGlyphBuffer(&rangeOption, &gb))) {
			CEUIFontSupport::drawGlyphs(context, gb, point);
		}
	}
}

float Font::floatWidthForComplexText(const TextRun& run, HashSet<const SimpleFontData*>* fallbackFonts) const
{
	float ret=0;
	if (cecfFont()) {
		WITH_FONT_OPTION(0, *this, run, 0, foption, pC16s);
		const SimpleFontData* primarySfd = primaryFont();
		CEUIFontPFArrayOption pfArrayOption;
		CESysFillMemory(&pfArrayOption, 0, sizeof(CEUIFontPFArrayOption));

		ICEUIPlatformFont* pfs[256];
		if (fallbackFonts) {
			pfArrayOption.nPFs = sizeof(pfs)/sizeof(ICEUICompositeFont);
			pfArrayOption.pPFArray = pfs;
			foption.normalPfArrayOption = &pfArrayOption;
		}

		CEComICEUITextMeasurerRef tm;
		m_cecfFont.getTextMeasurer(pC16s, run.length(), &foption, &tm);
		CEUTextMeasurement w = 0;
		tm.measureWidth(&w);
		ret = (float)w;
		{
			UINT32 i=0;
			for (i=0; i<pfArrayOption.nPFs; i++) {
				CEComICEUIPlatformFontRef ipf;
				ipf.attach(pfArrayOption.pPFArray[i]);
				INT32 szPix = 0;
				ipf.getValue(CEUIFontValueType_PixelSize, &szPix);
				FontPlatformData fpd;
				CEUIFontSupport::initFontPlatformData(szPix, ipf, fpd);	
				SimpleFontData* pSfd = WebCore::fontCache()->getCachedFontData(&fpd);
				if (pSfd && (primarySfd != pSfd))
					fallbackFonts->add(pSfd);
			}
		}
	}
	return ret;
}

FloatRect Font::selectionRectForComplexText(const TextRun& run, const IntPoint& point, int h,
											int from, int to) const
{
	FloatRect ret;

	//CEASSERT(from < to);
	if (cecfFont() && from < to) {
		WITH_FONT_OPTION(0, *this, run, point.x(), foption, pC16s);

		CEComICEUITextMeasurerRef tm;
		m_cecfFont.getTextMeasurer(pC16s, run.length(), &foption, &tm);

		CEComICEDXRegionRef hilightRegion;
		CEUIFontStrRangeOption range = {from, (to - from)};
		
		CEHResult err = tm.getBoundaryRegion(&range, 1, (UINT32)h, &hilightRegion);
		if (CESucceeded(err)) {
			CERect rect;
			err = hilightRegion.getExtents(&rect);
			if (CESucceeded(err)) {
				ret.setX(point.x() + rect._x);
				ret.setWidth(rect._width);
				ret.setY(point.y());
				ret.setHeight(h);
			}
		}
	}
	return ret;
}

int Font::offsetForPositionForComplexText(const TextRun& run, int x, bool includePartialGlyphs) const
{
	int ret = 0;	// boundary16.
	if (cecfFont()) {
		WITH_FONT_OPTION(0, *this, run, 0, foption, pC16s);
		CEComICEUITextMeasurerRef tm;
		m_cecfFont.getTextMeasurer(pC16s, run.length(), &foption, &tm);

		UINT32 lBoundary16=0;
		UINT32 rBoundary16=0;
		CEUTextMeasurement lDist = 0;
		CEUTextMeasurement rDist = 0;
		if (CESucceeded(tm.positionToBoundaryOffset(x, &lBoundary16, &rBoundary16, &lDist, &rDist, 0, 0))) {
#if 0
			CEComDebugPrintf("pick: %d, %d, %d, %d\n", lBoundary16, rBoundary16, lDist, rDist);
#endif // 0
			if (lDist < rDist)
				ret = lBoundary16;
			else
				ret = rBoundary16;
		} else {
			ret = 0;
			int end16 = run.length();  // boundary16. not offset16.
			if (x <= 0) {
				if (run.rtl())
					ret = end16;
			} else {
				if (run.ltr())
					ret = end16;
			}
		}
	}
	return ret;
}

} // namespace.
#else //#if USE(SILK_CEUIFONT)
bad config.
#endif //#if USE(SILK_CEUIFONT)
