/*
 * Copyright (C) 2006, 2007, 2008 Apple Inc.  All rights reserved.
 * Copyright 2008, 2009 Sony Corporation
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer. 
 * 2.  Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution. 
 * 3.  Neither the name of Apple Computer, Inc. ("Apple") nor the names of
 *     its contributors may be used to endorse or promote products derived
 *     from this software without specific prior written permission. 
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE AND ITS CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE OR ITS CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"

#if USE(SILK_CEUIFONT)
 
#include "FontCache.h"
#include "Font.h"
#include "FontSelector.h"
#include "FontFamily.h"
#include "SimpleFontData.h"
#include "SegmentedFontData.h"
#include "StringHash.h"
#include "UnicodeRange.h"

#include "ICEVG.h"
#include "GraphicsContext.h"
#include "GraphicsContextPlatformPrivateSilk.h"
#include "CEPFRasterizer.h"
#include "ICEI18nSupport.h"
namespace WebCore
{

namespace CEUIFontSupport
{
	class Env {
	public:
		CEComICEUICompositeFontFactoryRef&	getCFFactory()
		{
			_init();
			return _cfFactory;
		}

		CEComICEUIFontFamilyNameUtilityRef&	getFFUtil()
		{
			_init();
			return _ffUtil;
		}

		void getCFLang(CEComICEI18nLocaleRef& langOut) 
		{
			if (!_defaultLang) {
				CEComICEI18nClassFactoryRef i18nCf;
				CEHResult hr = ICEI18nClassFactoryCreate(CEComStdClassID_CEI18nClassFactory, i18nCf);
				if ( CESucceeded(hr) ) 
				{
					hr = i18nCf.getLocaleFromLanguageTagCStringLiteral("en", &_defaultLang);
				}
			}
			langOut = _defaultLang;
		}

		~Env()
		{
			CEASSERT(!_cfFactory && !_ffUtil);
		}

		void shutdown()
		{
			_cfFactory = 0;
			_ffUtil = 0;
			_defaultLang = 0;
		}

	private:
		void _init() {
			if (!_cfFactory) {
				CEComGetThreadContext(CEComIID_ICEUICompositeFontFactory, (void* *const)&_cfFactory);
			}
			if (!_ffUtil) {
				CEComGetThreadContext(CEComIID_ICEUIFontFamilyNameUtility, (void* *const)&_ffUtil);
			}

			CEASSERT(_cfFactory && _ffUtil);
		}

		CEComICEUICompositeFontFactoryRef	_cfFactory;
		CEComICEUIFontFamilyNameUtilityRef	_ffUtil;
		CEComICEI18nLocaleRef				_defaultLang;
	};

	Env _env;

	// TODO: some arguments are required to determine CECF's language.
	void getCFLang(CEComICEI18nLocaleRef& langOut) 
	{
		CEASSERT(!langOut.object());
		_env.getCFLang(langOut);
	}

	void initFontPlatformData(int szPix, CEComICEUIPlatformFontRef& pf, FontPlatformData& fpdOut)
	{
		CEPFFontStyle ceFontStyles;
		pf.getStyle(&ceFontStyles);
		
		bool synthesizeBold = false;
		bool synthesizeItalic = false;
		getSynthesizeFlags(ceFontStyles, pf, synthesizeBold, synthesizeItalic);
		FontPlatformData fpd(pf, szPix, synthesizeBold, synthesizeItalic);
		fpdOut = fpd;
	}

	CEComICEUICompositeFontFactoryRef&	getCFFactory() { return  _env.getCFFactory(); }

	CEHResult toCEUIFontFamily(const WebCore::AtomicString& astr, CEComICEUIFontFamilyRef& ffOut)
	{
		CEComICEUIFontFamilyNameUtilityRef ffUtil = _env.getFFUtil();

		CEASSERT(sizeof(UChar) == sizeof(UTF16CHAR));
		const UTF16CHAR* pC16s = (const UTF16CHAR*)astr.characters();
		return ffUtil.createSingleFontFamilyFromUTF16Array(pC16s, astr.length(), &ffOut);
	}

	CEHResult fdFamilyToCEUIFontFamily(const WebCore::AtomicString& fdFamilyStr, CEComICEUIFontFamilyRef& ifamily)
	{
		CEHResult hr = CE_S_OK;
		WebCore::AtomicString genericFamilyPrefix("-webkit-");
		UINT32 prefixLen = genericFamilyPrefix.length();

		CEComICEUIFontFamilyRef tfamily;
		if (!fdFamilyStr.isEmpty()) {
			if (fdFamilyStr.startsWith(genericFamilyPrefix))
			{
				if (fdFamilyStr.length() > prefixLen)
					hr = _env.getFFUtil().createSingleFontFamilyFromUTF16Array((const UTF16CHAR*)&(fdFamilyStr.characters()[prefixLen]), 
						fdFamilyStr.length()-prefixLen, &tfamily);
			}
			else
				hr = CEUIFontSupport::toCEUIFontFamily(fdFamilyStr, tfamily);
		}
		return hr;
	}

	CEHResult toCEUIFontFamily(CEComICEUStringRef& str, CEComICEUIFontFamilyRef& ffOut)
	{
		CEComICEUIFontFamilyNameUtilityRef ffUtil = _env.getFFUtil();
		UINT32 nC16s = 0;
		const UTF16CHAR* pC16s = 0;
		str.getCharArray16(&pC16s, &nC16s);
		return ffUtil.createSingleFontFamilyFromUTF16Array(pC16s, nC16s, &ffOut);
	}

	CEHResult toCEUIFontFamily(const SimpleFontData& sfd, const AtomicString& fdFamily, CEComICEUIFontFamilyRef& ffOut)
	{
		CEHResult hr = CE_S_OK;
		if (sfd.isCustomFont())
		{
			CEComICEUIPlatformFontRef ifont = sfd.platformData().pfFont();
			ifont.getFamilyName(&ffOut);
		}
		else
		{
			fdFamilyToCEUIFontFamily(fdFamily, ffOut);
		}
		return hr;
	}

	CEHResult toCEUIFontFamily(const Font& font, CEComICEUIFontFamilyRef& ffOut)
	{
		CEComICEUIFontFamilyNameUtilityRef ffUtil = _env.getFFUtil();
		ICEUIFontFamily* ifamilies[256];
		UINT32 max = sizeof(ifamilies)/sizeof(ICEUIFontFamily*);
		CESysFillMemory(ifamilies, 0, sizeof(ifamilies));

		const FontDescription& fd = font.fontDescription();
		FontSelector* fsel = const_cast<Font&>(font).fontSelector();
		unsigned i = 0;
		UINT32 j = 0;
		const FontFamily& firstFamily = fd.family();
		const FontFamily* family = firstFamily.familyIsEmpty() ? 0 : &firstFamily;
		
		if (fsel) {
			while (family) {
				const FontData* fdata = fsel->getFontData(fd, family->family());
				if (fdata) {
					if(!fdata->isSegmented()) {
						SimpleFontData* sfd = (SimpleFontData*)fdata;
						if (sfd) {
							CEComICEUIFontFamilyRef ifamily;
							toCEUIFontFamily(*sfd, family->family(), ifamily);
#if defined(_DEBUG) || !defined(NDEBUG)
							//CEComDebugPrintf("%p:[%d]:ifamily: %s\n", &font, i, ifamily.dbgToString());
#endif //#if defined(_DEBUG) || !defined(NDEBUG)
							if (ifamily) {
								ifamilies[j] = ifamily.detach();
								j++;
							}
						}
					} else {
						SegmentedFontData* sgfd = (SegmentedFontData*)fdata;
						for (unsigned ri = 0; ri < sgfd->numRanges(); ri++) {
							const FontDataRange& fdrange = sgfd->rangeAt(ri);
							const SimpleFontData* sfd = fdrange.fontData();
							if (sfd) {
								CEComICEUIFontFamilyRef ifamily;
								toCEUIFontFamily(*sfd, family->family(), ifamily);
#if defined(_DEBUG) || !defined(NDEBUG)
								//CEComDebugPrintf("%p:[%d][%d]:ifamily: %s\n", &font, i, ri, ifamily.dbgToString());
#endif //#if defined(_DEBUG) || !defined(NDEBUG)
								if (ifamily) {
									ifamilies[j] = ifamily.detach();
									j++;
								}
							}
						}
					}
				}
				else
				{
					CEComICEUIFontFamilyRef ifamily;
					toCEUIFontFamily(family->family(), ifamily);
#if defined(_DEBUG) || !defined(NDEBUG)
					//CEComDebugPrintf("%p: [%d]:ifamily: %s\n", &font, i, ifamily.dbgToString());
#endif //#if defined(_DEBUG) || !defined(NDEBUG)
					if (ifamily) {
						ifamilies[j] = ifamily.detach();
						j++;
					}
				}
				i++;
				family = family->next();
			}
		}

		CEHResult hr = CE_S_OK;
		if (j) {
			hr = ffUtil.createFontFamilyFromFamilyArray(ifamilies, j, &ffOut);
			UINT32 k = 0;
			for (k=0; k<j; k++) {
				CEComICEUIFontFamilyRef t = ifamilies[k];  // release each interface pointer.
				t.release();
			}
		} else {
			hr = ffUtil.createSingleFontFamilyFromCStringLiteral((const UCHAR8*)"default", 7, &ffOut);
		}
		return hr;
	}

	CEHResult toCEUIFontFamily(const WebCore::FontDescription& fontDescription, CEComICEUIFontFamilyRef& ifamily)
	{
		ifamily = 0;
		CEHResult hr = CE_S_OK;
		const WebCore::FontFamily f = fontDescription.family();
		{
			ICEUIFontFamily* ifamilies[256];
			UINT32 i=0;
			UINT32 max = sizeof(ifamilies)/sizeof(ICEUIFontFamily*);
			CESysFillMemory(ifamilies, 0, sizeof(ifamilies));

			const WebCore::FontFamily* pF = &f;
			while (pF && i<max) {
				CEComICEUIFontFamilyRef tfamily;
				if (!pF->familyIsEmpty()) {
					hr = fdFamilyToCEUIFontFamily(pF->family(), tfamily);
					if (tfamily) {
						ifamilies[i] = tfamily.detach();
						i++;
					}
				}
				pF = pF->next();
			}

			if (i) {
				hr = _env.getFFUtil().createFontFamilyFromFamilyArray(ifamilies, i, &ifamily);
				UINT32 j=0;
				for (j=0; j<i; j++) {
					CEComICEUIFontFamilyRef t = ifamilies[j];  // release each interface pointer.
					t.release();
				}
			}
		}
		return hr;
	}

	CEComICEUIFontFamilyNameUtilityRef getFFUtil()
	{
		return _env.getFFUtil();
	}

	CEPFFontStyle toCEUIFontStyle(const FontDescription& fontDescription)
	{
		CEPFFontStyle style = CEPFFONTSTYLE_REGULAR;
		eCEPFFontStyleWeightClass wc = style.weightClass;

		switch (fontDescription.weight())
		{
		case FontWeight100: wc = eCEPFFontStyleWeightClass_100; break;
		case FontWeight200: wc = eCEPFFontStyleWeightClass_200; break;
		case FontWeight300: wc = eCEPFFontStyleWeightClass_300; break;
		case FontWeight400: wc = eCEPFFontStyleWeightClass_400; break;
		case FontWeight500: wc = eCEPFFontStyleWeightClass_500; break;
		case FontWeight600: wc = eCEPFFontStyleWeightClass_600; break;
		case FontWeight700: wc = eCEPFFontStyleWeightClass_700; break;
		case FontWeight800: wc = eCEPFFontStyleWeightClass_800; break;
		case FontWeight900: wc = eCEPFFontStyleWeightClass_900; break;
		default:
			break;
		}
		style.weightClass = wc;

		if (fontDescription.italic())
			style.eCEPFFontStyleBits |= eCEPFFontStyle_Italic;
		
		// TODO: oblique and stretch style support.

		return style;
	}

	void getSynthesizeFlags(const CEPFFontStyle& requiredStyle, CEComICEUIPlatformFontRef& pf, bool& synthesizeBoldOut, bool& synthesizeItalicOut)
	{
		CEPFFaceInfo faceInfo;
		CESysFillMemory(&faceInfo, 0, sizeof(CEPFFaceInfo));
		pf.getFaceInfo(&faceInfo);

		synthesizeBoldOut = (requiredStyle.weightClass >= eCEPFFontStyleWeightClass_SemiBold) &&
			(faceInfo.style.weightClass < eCEPFFontStyleWeightClass_SemiBold);
		synthesizeItalicOut = (requiredStyle.eCEPFFontStyleBits & eCEPFFontStyle_Italic) && !(faceInfo.style.eCEPFFontStyleBits & eCEPFFontStyle_Italic);
	}

	void setupPlatformGC(GraphicsContext& graphicsContext, CEComICEVGContextRef& vgc, bool bSetStrokeColor)
	{
		int tmode = graphicsContext.textDrawingMode();
		if (tmode & cTextFill) {
			if (!bSetStrokeColor) {
				Color fillColor = graphicsContext.fillColor();
				CERGBAColor rgbaColor;
				GraphicsContextPlatformPrivate::Color2CERGBAColor(fillColor, rgbaColor);
				vgc.setSourceRGBA(&rgbaColor);
			}
		}

		if (tmode & cTextStroke) {
			Color strokeColor = graphicsContext.strokeColor();
			if (bSetStrokeColor) {
				CERGBAColor rgbaColor;
				GraphicsContextPlatformPrivate::Color2CERGBAColor(strokeColor, rgbaColor);
				vgc.setSourceRGBA(&rgbaColor); 
			}
			vgc.setLineWidthD(graphicsContext.strokeThickness());
		}
	}

	void shutdown()
	{
		_env.shutdown();
		{
			WebCore::FontCache::shutdown();	

			if (WebCore::GlyphPageTreeNode::roots)
				delete WebCore::GlyphPageTreeNode::roots;

			if (WebCore::GlyphPageTreeNode::pageZeroRoot)
				delete WebCore::GlyphPageTreeNode::pageZeroRoot;
		}
	}
};


//////////////////////////////////////////////////////////
// RefCountedICEUIFont
//////////////////////////////////////////////////////////
FontPlatformData::RefCountedICEUIPlatformFont::~RefCountedICEUIPlatformFont() 
{
	if (m_pfFont != reinterpret_cast<ICEUIPlatformFontRef>(-1))
	{
		if (m_pfFont)
		{
			m_pfFont->_vtbl->_release(m_pfFont);
			m_pfFont = 0;
		}
	}
}

//////////////////////////////////////////////////////////
// FontCache
//////////////////////////////////////////////////////////
void FontCache::platformInit()
{
	// TODO: fill this part.
	//FontPlatformData::init();
}

//! this method should be changed to getFontDataForCharacter(const Font& font, UTF32CHAR c32), isn't it?
const SimpleFontData* FontCache::getFontDataForCharacters(const Font& font, const UChar* characters, int length)
{
	const SimpleFontData* pRet = 0;
	if (length <=0) 
		return pRet;

	UChar32 c32 = characters[0];

	if (length == 2) {
		if (U_IS_SURROGATE(characters[0]) && U_IS_SURROGATE(characters[1])) {
			if (U16_IS_LEAD(characters[0]) && U16_IS_TRAIL(characters[1])) {
				c32 = U16_GET_SUPPLEMENTARY(characters[0], characters[1]);
			}
		}
		else
			ASSERT(0 && "fix");
	}
	else if (length == 1) {
		// nothing todo.
	}
	else {
		// error.
		ASSERT(0 && "fix this error");
	}
	
	
	const SimpleFontData* primarySfd = font.primaryFont();
	pRet = primarySfd->fontDataForCharacter(c32);

	ASSERT(pRet);

	CEComICEUICompositeFontRef cf = font.cecfFont();
	if (cf)
	{
		CEComICEUIFontFamilyRef ifamily;
		cf.getFamilyName(&ifamily);

		CEComICEUIPlatformFontRef pf;
		if (CESucceeded(cf.getPlatformFont(c32, false, &pf, 0))) {
			CEComICEI18nCharMapperRef cm;
			if (pf != pRet->platformData().pfFont()) {
				FontPlatformData alternateFont;
				CEUIFontSupport::initFontPlatformData(font.fontDescription().computedPixelSize(), pf, alternateFont);
				SimpleFontData* sfd = getCachedFontData(&alternateFont);
				pRet = sfd;
			}
		}
	}
	return pRet;
}


FontPlatformData* FontCache::getSimilarFontPlatformData(const Font& font)
{
	return 0;
}

FontPlatformData* FontCache::getLastResortFallbackFont(const FontDescription& fontDescription)
{
	// see fontinfo.xml
	AtomicString defaultStr("default");
	return getCachedFontPlatformData(fontDescription, defaultStr);
}

FontPlatformData* FontCache::createFontPlatformData(const FontDescription& fontDescription, const AtomicString& family)
{
	FontPlatformData* ret = 0;

	CEHResult hr = CE_S_OK;
	CEComICEUIFontFamilyRef ifamily; 

	hr = CEUIFontSupport::toCEUIFontFamily(family, ifamily);
	
	if (CESucceeded(hr)) {
		CEComICEUICompositeFontFactoryRef cfFactory = CEUIFontSupport::getCFFactory();
		CEComICEUICompositeFontRef cfFont;

		CEPFFontStyle style = CEUIFontSupport::toCEUIFontStyle(fontDescription);

		int px = fontDescription.computedPixelSize();
		CEComICEI18nLocaleRef lang;
		CEUIFontSupport::getCFLang(lang);

		cfFactory.createCompositeFontBySizeInPixels(lang, ifamily, &style, px, false, &cfFont);

		if (cfFont)
		{
			CEComICEUIPlatformFontRef pf;
			cfFont.getFirstPlatformFont(&pf);
			if (pf) {
				bool synthesizeBold = false;
				bool synthesizeItalic = false;
				CEUIFontSupport::getSynthesizeFlags(style, pf, synthesizeBold, synthesizeItalic);
				ret = new FontPlatformData(pf, px, synthesizeBold, synthesizeItalic);
			}
		}
	}
    return ret;
}

void FontCache::getTraitsInFamily(const AtomicString& familyName, Vector<unsigned>& traitsMasks)
{
	CEComICEUICompositeFontFactoryRef cfFactory;

	CEComICEUIFontFamilyRef ifamily;
	if (!CEUIFontSupport::toCEUIFontFamily(familyName, ifamily)) {
		CEPFFontStyle traits[50];
		UINT32 nTraitsRef = sizeof(traits)/sizeof(CEPFFontStyle);
		CEHResult hr = cfFactory.getTraitsInFamily(ifamily, traits, &nTraitsRef);
		if (CESucceeded(hr)) {
			UINT32 i = 0;
			for (i=0; i<nTraitsRef; i++) {
				UINT32 v = (traits[i].eCEPFFontStyleBits) ? 0 : FontVariantNormalMask;
				if (traits[i].eCEPFFontStyleBits & eCEPFFontStyle_Italic) {
					v |= FontStyleItalicMask;
				}

				v |= FontVariantNormalMask;

				switch (traits[i].weightClass)
				{
				case eCEPFFontStyleWeightClass_100:
					v |= FontWeight100Bit; break;
				case eCEPFFontStyleWeightClass_200:
					v |= FontWeight200Bit; break;
				case eCEPFFontStyleWeightClass_300:
					v |= FontWeight300Bit; break;
				case eCEPFFontStyleWeightClass_400:
					v |= FontWeight400Bit; break;
				case eCEPFFontStyleWeightClass_500:
					v |= FontWeight500Bit; break;
				case eCEPFFontStyleWeightClass_600:
					v |= FontWeight600Bit; break;
				case eCEPFFontStyleWeightClass_700:
					v |= FontWeight700Bit; break;
				case eCEPFFontStyleWeightClass_800:
					v |= FontWeight800Bit; break;
				case eCEPFFontStyleWeightClass_900:
					v |= FontWeight900Bit; break;
				default:
					break;
				}
				traitsMasks.append((unsigned)v);
			}
		}
	}
}

} //namespace WebCore.

void CEUIFontSupport_shutdown()
{
	WebCore::CEUIFontSupport::shutdown();
}

#else //#if USE(SILK_CEUIFONT)
bad config.
#endif //#if USE(SILK_CEUIFONT)
