/* Cairo - a vector graphics library with display and print output
 *
 * Copyright © 2009 Chris Wilson
 *
 * "The contents of this file are subject to the Mozilla Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and
 * limitations under the License.
 *
 * Alternatively, the contents of this file may be used under the terms
 * of the _____ license (the  "[___] License"), in which case the
 * provisions of [______] License are applicable instead of those
 * above. If you wish to allow use of your version of this file only
 * under the terms of the [____] License and not to allow others to use
 * your version of this file under the MPL, indicate your decision by
 * deleting the provisions above and replace them with the notice and
 * other provisions required by the [___] License. If you do not delete
 * the provisions above, a recipient may use your version of this file
 * under either the MPL or the [___] License."
 *
 * NOTE: The text of this Exhibit A may differ slightly from the text
 * of the notices in the Source Code files of the Original Code. You
 * should use the text of this Exhibit A rather than the text found in
 * the Original Code Source Code for Your Modifications. 
 */

#include "cairoint.h"

#include "cairo-drm-private.h"
#include "cairo-drm-ioctl-private.h"

#include "cairo-error-private.h"

#include <sys/ioctl.h>
#include <errno.h>

#define ERR_DEBUG(x) x

struct drm_gem_close {
	/** Handle of the object to be closed. */
	uint32_t handle;
	uint32_t pad;
};

struct drm_gem_flink {
	/** Handle for the object being named */
	uint32_t handle;

	/** Returned global name */
	uint32_t name;
};

struct drm_gem_open {
	/** Name of object being opened */
	uint32_t name;

	/** Returned handle for the object */
	uint32_t handle;

	/** Returned size of the object */
	uint64_t size;
};

#define DRM_IOCTL_GEM_CLOSE		DRM_IOW (0x09, struct drm_gem_close)
#define DRM_IOCTL_GEM_FLINK		DRM_IOWR(0x0a, struct drm_gem_flink)
#define DRM_IOCTL_GEM_OPEN		DRM_IOWR(0x0b, struct drm_gem_open)

cairo_status_t
_cairo_drm_bo_open_for_name (const cairo_drm_device_t *dev,
			     cairo_drm_bo_t *bo,
			     uint32_t name)
{
    struct drm_gem_open open;
    int ret;

    open.name = name;
    open.handle = 0;
    open.size = 0;
    do {
	ret = ioctl (dev->fd, DRM_IOCTL_GEM_OPEN, &open);
    } while (ret == -1 && errno == EINTR);
    if (ret == -1) {
	ERR_DEBUG((fprintf (stderr, "Failed to open bo for name %d: %s\n",
			    name, strerror (errno))));
	return _cairo_error (CAIRO_STATUS_NO_MEMORY);
    }

    bo->name = name;
    bo->size = open.size;
    bo->handle = open.handle;

    return CAIRO_STATUS_SUCCESS;
}

cairo_status_t
_cairo_drm_bo_flink (const cairo_drm_device_t *dev,
		     cairo_drm_bo_t *bo)
{
    struct drm_gem_flink flink;
    int ret;

    memset (&flink, 0, sizeof (flink));
    flink.handle = bo->handle;
    ret = ioctl (dev->fd, DRM_IOCTL_GEM_FLINK, &flink);
    if (ret == -1) {
	ERR_DEBUG((fprintf (stderr, "Failed to flink bo: %s\n",
			    strerror (errno))));
	return _cairo_error (CAIRO_STATUS_NO_MEMORY);
    }

    bo->name = flink.name;

    return CAIRO_STATUS_SUCCESS;
}

void
_cairo_drm_bo_close (const cairo_drm_device_t *dev,
		     cairo_drm_bo_t *bo)
{
    struct drm_gem_close close;
    int ret;

    close.handle = bo->handle;
    do {
	ret = ioctl (dev->fd, DRM_IOCTL_GEM_CLOSE, &close);
    } while (ret == -1 && errno == EINTR);
}
