/* cairo - a vector graphics library with display and print output
 *
 * Copyright © 2005 Red Hat, Inc
 *
 * "The contents of this file are subject to the Mozilla Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and
 * limitations under the License.
 *
 * The Original Code is the cairo graphics library.
 *
 * The Initial Developer of the Original Code is Red Hat, Inc.
 *
 * Contributor(s):
 *	Kristian Høgsberg <krh@redhat.com>
 *	Adrian Johnson <ajohnson@redneon.com>
 *
 * Alternatively, the contents of this file may be used under the terms
 * of the _____ license (the  "[___] License"), in which case the
 * provisions of [______] License are applicable instead of those
 * above. If you wish to allow use of your version of this file only
 * under the terms of the [____] License and not to allow others to use
 * your version of this file under the MPL, indicate your decision by
 * deleting the provisions above and replace them with the notice and
 * other provisions required by the [___] License. If you do not delete
 * the provisions above, a recipient may use your version of this file
 * under either the MPL or the [___] License."
 *
 * NOTE: The text of this Exhibit A may differ slightly from the text
 * of the notices in the Source Code files of the Original Code. You
 * should use the text of this Exhibit A rather than the text found in
 * the Original Code Source Code for Your Modifications. 
 */

#ifndef CAIRO_RECORDING_SURFACE_H
#define CAIRO_RECORDING_SURFACE_H

#include "cairoint.h"
#include "cairo-path-fixed-private.h"
#include "cairo-clip-private.h"

typedef enum {
    /* The 5 basic drawing operations. */
    CAIRO_COMMAND_PAINT,
    CAIRO_COMMAND_MASK,
    CAIRO_COMMAND_STROKE,
    CAIRO_COMMAND_FILL,
    CAIRO_COMMAND_SHOW_TEXT_GLYPHS,
} cairo_command_type_t;

typedef enum {
    CAIRO_RECORDING_REGION_ALL,
    CAIRO_RECORDING_REGION_NATIVE,
    CAIRO_RECORDING_REGION_IMAGE_FALLBACK
} cairo_recording_region_type_t;

typedef struct _cairo_command_header {
    cairo_command_type_t	 type;
    cairo_recording_region_type_t     region;
    cairo_operator_t		 op;
    cairo_clip_t		 clip;
} cairo_command_header_t;

typedef struct _cairo_command_paint {
    cairo_command_header_t       header;
    cairo_pattern_union_t	 source;
} cairo_command_paint_t;

typedef struct _cairo_command_mask {
    cairo_command_header_t       header;
    cairo_pattern_union_t	 source;
    cairo_pattern_union_t	 mask;
} cairo_command_mask_t;

typedef struct _cairo_command_stroke {
    cairo_command_header_t       header;
    cairo_pattern_union_t	 source;
    cairo_path_fixed_t		 path;
    cairo_stroke_style_t	 style;
    cairo_matrix_t		 ctm;
    cairo_matrix_t		 ctm_inverse;
    double			 tolerance;
    cairo_antialias_t		 antialias;
} cairo_command_stroke_t;

typedef struct _cairo_command_fill {
    cairo_command_header_t       header;
    cairo_pattern_union_t	 source;
    cairo_path_fixed_t		 path;
    cairo_fill_rule_t		 fill_rule;
    double			 tolerance;
    cairo_antialias_t		 antialias;
} cairo_command_fill_t;

typedef struct _cairo_command_show_text_glyphs {
    cairo_command_header_t       header;
    cairo_pattern_union_t	 source;
    char			*utf8;
    int				 utf8_len;
    cairo_glyph_t		*glyphs;
    unsigned int		 num_glyphs;
    cairo_text_cluster_t	*clusters;
    int				 num_clusters;
    cairo_text_cluster_flags_t   cluster_flags;
    cairo_scaled_font_t		*scaled_font;
} cairo_command_show_text_glyphs_t;

typedef union _cairo_command {
    cairo_command_header_t      header;

    cairo_command_paint_t			paint;
    cairo_command_mask_t			mask;
    cairo_command_stroke_t			stroke;
    cairo_command_fill_t			fill;
    cairo_command_show_text_glyphs_t		show_text_glyphs;
} cairo_command_t;

typedef struct _cairo_recording_surface {
    cairo_surface_t base;

    cairo_content_t content;

    /* A recording-surface is logically unbounded, but when used as a
     * source we need to render it to an image, so we need a size at
     * which to create that image. */
    cairo_rectangle_t extents_pixels;
    cairo_rectangle_int_t extents;
    cairo_bool_t unbounded;

    cairo_clip_t clip;

    cairo_array_t commands;

    int replay_start_idx;
} cairo_recording_surface_t;

slim_hidden_proto (cairo_recording_surface_create);

cairo_private cairo_int_status_t
_cairo_recording_surface_get_path (cairo_surface_t	 *surface,
				   cairo_path_fixed_t *path);

cairo_private cairo_status_t
_cairo_recording_surface_replay (cairo_surface_t *surface,
				 cairo_surface_t *target);


cairo_private cairo_status_t
_cairo_recording_surface_replay_analyze_recording_pattern (cairo_surface_t *surface,
							   cairo_surface_t *target);

cairo_private cairo_status_t
_cairo_recording_surface_replay_and_create_regions (cairo_surface_t *surface,
						    cairo_surface_t *target);
cairo_private cairo_status_t
_cairo_recording_surface_replay_region (cairo_surface_t			*surface,
					const cairo_rectangle_int_t *surface_extents,
					cairo_surface_t			*target,
					cairo_recording_region_type_t	region);

cairo_private cairo_status_t
_cairo_recording_surface_get_bbox (cairo_recording_surface_t *recording,
				   cairo_box_t *bbox,
				   const cairo_matrix_t *transform);

cairo_private cairo_bool_t
_cairo_surface_is_recording (const cairo_surface_t *surface);

#endif /* CAIRO_RECORDING_SURFACE_H */
